﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <climits>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>

#include <nn/util/util_Compression.h>
#include <nn/util/util_StreamingCompression.h>

#include "zutil.h"
#include "zlib.h"
#include "inftrees.h"
#include "inflate.h"

#include "detail/util_CompressionUtillity.h"

namespace {

// zlib で定義してある圧縮レベルのデフォルト値
const int DefaultCompressionLevel = Z_DEFAULT_COMPRESSION;

// zlib で定義してあるメモリ使用量のデフォルト値
const int DefaultMemLevel =  8;

// Streamig 圧縮のエラーを表示する関数
void DisplayErr(int errCode)  NN_NOEXCEPT
{
    switch(errCode)
    {
    case Z_STREAM_ERROR:
        // Z_STREAM_ERROR はユーザーがバッファに NULL を指定した時に返る(事前条件違反)ので、
        // ASSERT で止める。
        NN_SDK_ASSERT(false, "ERROR (%d) : Source Buffer or Destination Buffer is inconsistent.\n", errCode);
        break;

    case Z_VERSION_ERROR:
        // サポートされていないバージョンのデータが入ったときに返るエラー
        NN_SDK_LOG("ERROR (%d) : This library can not support this zlib version file.\n", errCode);
        break;

    case Z_MEM_ERROR:
        // ワークバッファ不足で返る。
        // NN_SDK_REQUIRES で事前にチェックしているはずなので、DisplayErr が返るタイミングで
        // これが返る = ライブラリのバグなので想定外のメッセージ。
        NN_SDK_ASSERT(false, "ERROR (%d) : Unexpected Error\n", errCode);
        break;

    case Z_BUF_ERROR:
        // 最終出力時に出力バッファが足りていないときに返るエラー
        // これは入力データ(各バッファ)に問題があるので、ログを出す。
        NN_SDK_LOG("ERROR (%d) : Src Buffer Size or Dst Buffer Size is 0.\n", errCode);
        break;

    case Z_DATA_ERROR:
        // stream が意図せず free されている場合のエラー
        NN_SDK_LOG("ERROR (%d) : the stream was freed prematurely.\n", errCode);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

// memLevel の範囲チェック用 inline 関数
#ifndef NN_SDK_BUILD_RELEASE
inline bool CheckMemLevelValueRange(int memLevel) NN_NOEXCEPT
{
    return ((1 <= memLevel) && (memLevel <= 9)) ;
}

inline bool CheckCompressionLevelValueRange(int compressionLevel) NN_NOEXCEPT
{
    return ((0 <= compressionLevel) && (compressionLevel <= 9));
}
#endif

// 内部で使用する Context の実態
class StreamingCompressContextImpl
{
public:

    // Context の初期化処理
    void Initialize(void* pWorkBuffer, size_t workBufferSize)
    {
        m_IsStreamEnd      = false;

        // zlib に渡す stream の設定を初期化
        m_Stream.next_in   = NULL;
        m_Stream.avail_in  = 0;
        m_Stream.next_out  = NULL;
        m_Stream.avail_out = 0;

        m_Stream.zalloc    = nn::util::detail::ZlibAllocator::Allocate;
        m_Stream.zfree     = nn::util::detail::ZlibAllocator::Free;

        // placement new を使って ZlibAllocator として初期化した結果をセットする。
        m_Stream.opaque    = new (m_AllocatorBuffer) nn::util::detail::ZlibAllocator(pWorkBuffer, workBufferSize);
    }

    // ストリームに Src と Dst を設定する関数
    // zlib 用の stream 構造体に渡すため、64bit 環境であっても一度に 4GB 以上のサイズは扱えない
    void SetStream(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize)
    {
        NN_SDK_ASSERT(srcSize <= UINT_MAX);
        NN_SDK_ASSERT(dstSize <= UINT_MAX);
        m_Stream.next_in   = reinterpret_cast<nnutilZlib_Bytef*>(const_cast<void*>(pSrc));
        m_Stream.avail_in  = static_cast<nnutilZlib_uInt>(srcSize);
        m_Stream.next_out  = reinterpret_cast<nnutilZlib_Bytef*>(pDst);
        m_Stream.avail_out = static_cast<nnutilZlib_uInt>(dstSize);
    }


    z_stream* GetStream()
    {
        return &m_Stream;
    }

    void SetStreamEnd(bool isStreamEnd)
    {
        m_IsStreamEnd = isStreamEnd;
    }

    bool IsStreamEnd()
    {
        return m_IsStreamEnd;
    }

private:

    nn::Bit8                         m_AllocatorBuffer[sizeof(nn::util::detail::ZlibAllocator)];
    z_stream                         m_Stream;
    bool                             m_IsStreamEnd;
};

// static_assert で context のサイズチェック
NN_STATIC_ASSERT( sizeof(nn::util::StreamingCompressZlibContext::_context) >= sizeof(StreamingCompressContextImpl) );


// Context 初期化の共通処理
void InitializeStreamingCompressContextCommon(nn::util::StreamingCompressZlibContext* pContext,
                                                void* pWorkBuffer, size_t workBufferSize,
                                                int memLevel, int compressionLevel,
                                                int wbits) NN_NOEXCEPT
{
    // placement new を使って、コンテキストを内部で使用する形に割り当てる
    StreamingCompressContextImpl* pContextImpl = new (pContext->_context) StreamingCompressContextImpl;

    // Context の初期化処理
    pContextImpl->Initialize(pWorkBuffer, workBufferSize);

    // zlib の初期化処理。ここでオプションを渡す。
    int err = deflateInit2(pContextImpl->GetStream(),
            compressionLevel,
            Z_DEFLATED,
            wbits,
            memLevel,
            Z_DEFAULT_STRATEGY );
    if(err != Z_OK)
    {
        // ここで deflateIniti2 から返りえる値は、
        // Z_OK, Z_MEM_ERROR, Z_STREAM_ERROR, Z_VERSION_ERROR の四種類
        // 実装ミス以外ではどれも返らないはず。
        NN_ABORT("ERROR (%d) : Unexpected Error\n", err);
    }
}


// StreamingCompress の共通処理
bool StreamingCompressCommon(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                               void* pDst,          size_t dstSize,
                               const void* pSrc,    size_t srcSize,
                               nn::util::StreamingCompressZlibContext* pContext) NN_NOEXCEPT
{
    // 内部で使用するコンテキストへ変換
    StreamingCompressContextImpl* pContextImpl = reinterpret_cast<StreamingCompressContextImpl*>(pContext->_context);

    // ストリーミング処理が終了していた場合は入出力を両方 0 にして true を返すことで、
    // 処理の完了を伝える。
    if(pContextImpl->IsStreamEnd())
    {
        *pOutDstSize = 0;
        *pOutConsumedSrcSize = 0;
        return true;
    }

    // ストリームに Src と Dst をセットする
    pContextImpl->SetStream(pDst, dstSize, pSrc, srcSize);

    // 圧縮前の入出力バッファの残りのサイズを保持しておく
    const uint32_t availInBefore  = static_cast<uint32_t>(srcSize);
    const uint32_t availOutBefore = static_cast<uint32_t>(dstSize);

    if(srcSize > 0)
    {
        // deflate(圧縮) を実行
        int err = nnutilZlib_deflate(pContextImpl->GetStream(), Z_NO_FLUSH);
        if (err != Z_OK)
        {
            // Z_STREAM_ERROR, Z_BUF_ERROR
            DisplayErr(err);
            return false;
        }
    }
    else
    {
        // deflate(圧縮) を実行
        int err = nnutilZlib_deflate(pContextImpl->GetStream(), Z_FINISH);
        if(err != Z_OK)
        {
            if(err == Z_STREAM_END)
            {
                pContextImpl->SetStreamEnd(true);

                // deflate の終了処理
                err = nnutilZlib_deflateEnd(pContextImpl->GetStream());
                if (err != Z_OK)
                {
                    // Z_STREAM_ERROR, Z_DATA_ERROR
                    DisplayErr(err);
                    return false;
                }
            }
            else
            {
                // Z_BUF_ERROR
                DisplayErr(err);
                return false;
            }
        }
    }

    // 出力サイズ = 処理前の出力先バッファのサイズ - 今の出力先バッファのサイズ
    *pOutDstSize = availOutBefore - pContextImpl->GetStream()->avail_out;

    // 消費した入力 = 処理前の入力バッファのサイズ - 今の入力バッファのサイズ
    *pOutConsumedSrcSize = availInBefore - pContextImpl->GetStream()->avail_in;

    return true;
}


}



namespace nn {
namespace util {


void InitializeStreamingCompressZlibContext(StreamingCompressZlibContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingCompressContextImpl) == 0);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    NN_SDK_ASSERT(workBufferSize >= static_cast<size_t>(NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(DefaultMemLevel)));

    InitializeStreamingCompressContextCommon(pContext, pWorkBuffer, workBufferSize, DefaultMemLevel, DefaultCompressionLevel, detail::ZlibFormatWbits);
}

void InitializeStreamingCompressZlibContext(StreamingCompressZlibContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize,
                                            int memLevel,   int compressionLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingCompressContextImpl) == 0);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    NN_SDK_ASSERT(workBufferSize >= static_cast<size_t>(NN_UTIL_CALCULATE_COMPRESS_ZLIB_WORKBUFFER_SIZE(memLevel)));

    // memLevel と compressionLevel の範囲を確認する。
    NN_SDK_REQUIRES(CheckMemLevelValueRange(memLevel), "out of memLevel Range");
    NN_SDK_REQUIRES(CheckCompressionLevelValueRange(compressionLevel), "out of compressionLevel Range");

    InitializeStreamingCompressContextCommon(pContext, pWorkBuffer, workBufferSize, memLevel, compressionLevel, detail::ZlibFormatWbits);
}

void InitializeStreamingCompressGzipContext(StreamingCompressGzipContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingCompressContextImpl) == 0);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    NN_SDK_ASSERT(workBufferSize >= static_cast<size_t>(NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(DefaultMemLevel)));

    InitializeStreamingCompressContextCommon(pContext, pWorkBuffer, workBufferSize, DefaultMemLevel, DefaultCompressionLevel, detail::GzipFormatWbits);
}

void InitializeStreamingCompressGzipContext(StreamingCompressGzipContext* pContext,
                                            void* pWorkBuffer, size_t workBufferSize,
                                            int memLevel,   int compressionLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingCompressContextImpl) == 0);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    NN_SDK_ASSERT(workBufferSize >= static_cast<size_t>(NN_UTIL_CALCULATE_COMPRESS_GZIP_WORKBUFFER_SIZE(memLevel)));

    // memLevel と compressionLevel の範囲を確認する。
    NN_SDK_REQUIRES(CheckMemLevelValueRange(memLevel), "out of memLevel Range");
    NN_SDK_REQUIRES(CheckCompressionLevelValueRange(compressionLevel), "out of compressionLevel Range");

    InitializeStreamingCompressContextCommon(pContext, pWorkBuffer, workBufferSize, memLevel, compressionLevel, detail::GzipFormatWbits);
}

void InitializeStreamingCompressDeflateContext(StreamingCompressDeflateContext* pContext,
                                                void* pWorkBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingCompressContextImpl) == 0);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    NN_SDK_ASSERT(workBufferSize >= static_cast<size_t>(NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(DefaultMemLevel)));

    InitializeStreamingCompressContextCommon(pContext, pWorkBuffer, workBufferSize, DefaultMemLevel, DefaultCompressionLevel, detail::RawDeflateFormatWbits);
}

void InitializeStreamingCompressDeflateContext(StreamingCompressDeflateContext* pContext,
                                                void* pWorkBuffer, size_t workBufferSize,
                                                int memLevel,   int compressionLevel) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pContext);
    NN_SDK_REQUIRES(reinterpret_cast<uintptr_t>(pContext) % NN_ALIGNOF(StreamingCompressContextImpl) == 0);
    NN_SDK_ASSERT_NOT_NULL(pWorkBuffer);
    NN_SDK_ASSERT(workBufferSize >= static_cast<size_t>(NN_UTIL_CALCULATE_COMPRESS_DEFLATE_WORKBUFFER_SIZE(memLevel)));

    // memLevel と compressionLevel の範囲を確認する。
    NN_SDK_REQUIRES(CheckMemLevelValueRange(memLevel), "out of memLevel Range");
    NN_SDK_REQUIRES(CheckCompressionLevelValueRange(compressionLevel), "out of compressionLevel Range");

    InitializeStreamingCompressContextCommon(pContext, pWorkBuffer, workBufferSize, memLevel, compressionLevel, detail::RawDeflateFormatWbits);
}

bool StreamingCompressZlib(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingCompressZlibContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return StreamingCompressCommon(pOutDstSize,  pOutConsumedSrcSize, pDst, dstSize, pSrc, srcSize, pContext);
}

bool StreamingCompressGzip(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingCompressGzipContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return StreamingCompressCommon(pOutDstSize,  pOutConsumedSrcSize, pDst, dstSize, pSrc, srcSize, pContext);
}

bool StreamingCompressDeflate(size_t* pOutDstSize, size_t* pOutConsumedSrcSize,
                             void* pDst,          size_t dstSize,
                             const void* pSrc,    size_t srcSize,
                             StreamingCompressDeflateContext* pContext) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);

    return StreamingCompressCommon(pOutDstSize,  pOutConsumedSrcSize, pDst, dstSize, pSrc, srcSize, pContext);
}

} //util
} //nn
