﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>

#include <nn/nn_SdkLog.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Abort.h>

#include <nn/util/util_Decompression.h>

#include "zutil.h"
#include "zlib.h"
#include "inftrees.h"
#include "inflate.h"

#include "detail/util_CompressionUtillity.h"

namespace
{

// static_assert でビルド環境ごとの work のサイズチェック
NN_STATIC_ASSERT(nn::util::DecompressZlibWorkBufferSize >= sizeof(inflate_state) );

void DisplayErr(int errCode)  NN_NOEXCEPT
{
    switch(errCode)
    {
    case Z_STREAM_ERROR:
        // Z_STREAM_ERROR はユーザーがバッファに NULL を指定した時に返る(事前条件違反)ので、
        // ASSERT で止める。
        NN_SDK_ASSERT(false, "ERROR (%d) : Source Buffer or Destination Buffer is inconsistent.\n", errCode);
        break;

    case Z_VERSION_ERROR:
        // サポートされていないバージョンのデータが入ったときに返るエラーなので、ASSERT で止める。
        NN_SDK_ASSERT(false, "ERROR (%d) : This library can not support this zlib version file.\n", errCode);
        break;

    case Z_MEM_ERROR:
        // ワークバッファ不足で返る。
        // NN_SDK_REQUIRES で事前にチェックしているはずなので、DisplayErr が返るタイミングで
        // これが返る = ライブラリのバグなので想定外のメッセージ。
        NN_SDK_ASSERT(false, "ERROR (%d) : Unexpected Error\n", errCode);
        break;

    case Z_BUF_ERROR:
        // 入力が足りていないもしくは出力バッファが足りていないときに返るエラー
        // 事前条件として、サイズを指定しているが、たまたまデータが化けてそう判断される
        // = データが壊れている可能性もある。
        NN_SDK_LOG("ERROR (%d) : Source Buffer or Destination Buffer is short.\n", errCode);
        break;

    case Z_DATA_ERROR:
        // 入力データが壊れている場合のエラー
        NN_SDK_LOG("ERROR (%d) : the input data was corrupted.\n", errCode);
        break;

    case Z_NEED_DICT:
        // 今回は DICT の指定をしていないため、このエラーは返らないはず
        NN_SDK_ASSERT(false, "ERROR (%d) : DICT is inconsistent.\n", errCode);
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

bool DecompressZlibCommon(void* pDst,       size_t dstSize,
                          const void* pSrc, size_t srcSize,
                          void* pWork,      size_t workSize,
                          int wbits)  NN_NOEXCEPT
{
    // Zlib 用のアロケーター。渡されたバッファから単純にとってくるだけ
    nn::util::detail::ZlibAllocator za(pWork, workSize);

    // zlib で使用する構造体
    z_stream stream;
    int err;

    // zlib に渡されたバッファを設定する
    // zlib 用の stream 構造体に一括で渡すため、64bit 環境であっても 4GB 以上のサイズは扱えない
    // TODO : もし要望があるならば分割して渡すことで、4GB 以上のサイズも扱えるようにする。
    NN_SDK_ASSERT(srcSize <= UINT_MAX);
    NN_SDK_ASSERT(dstSize <= UINT_MAX);
    stream.next_in   = reinterpret_cast<nnutilZlib_Bytef*>(const_cast<void*>(pSrc));
    stream.avail_in  = static_cast<nnutilZlib_uInt>(srcSize);
    stream.next_out  = reinterpret_cast<nnutilZlib_Bytef*>(pDst);
    stream.avail_out = static_cast<nnutilZlib_uInt>(dstSize);

    // アロケーターもセット
    stream.zalloc    = nn::util::detail::ZlibAllocator::Allocate;
    stream.zfree     = nn::util::detail::ZlibAllocator::Free;
    stream.opaque    = &za;

    // zlib の初期化処理。
    err = inflateInit2(&stream, wbits);
    if(err != Z_OK)
    {
        // ここで inflateIniti2 から返りえる値は、
        // Z_OK, Z_STREAM_ERROR, Z_VERSION_ERROR, Z_MEM_ERROR の四種類
        DisplayErr(err);
        return false;
    }

    // zlib インフレート(伸長処理)
    err = nnutilZlib_inflate(&stream, Z_FINISH);
    if(err != Z_STREAM_END)
    {
        // ここで inflate から返りえる値は、
        // Z_STREAM_END, Z_NEED_DICT, Z_DATA_ERROR, Z_STREAM_ERROR, Z_MEM_ERROR, Z_BUF_ERRORの六種類
        DisplayErr(err);
        return false;
    }

    // zlib の終了処理
    err = nnutilZlib_inflateEnd(&stream);
    if (err != Z_OK)
    {
        // ここで inflateEnd から返りえる値は、
        // Z_OK, Z_STREAM_ERROR の二種類
        DisplayErr(err);
        return false;
    }

    return true;
}

}

namespace nn {
namespace util {

// Gzip 形式は圧縮データから圧縮前のサイズが取得できるため、取得するための関数
size_t GetGzipDecompressedSize(const void* pSrcBuf, size_t srcSize)  NN_NOEXCEPT
{
    return static_cast<size_t>(*reinterpret_cast<uint32_t*>(reinterpret_cast<uintptr_t>(pSrcBuf) + srcSize - 4));
}


bool DecompressZlib(void* pDst,       size_t dstSize,
                    const void* pSrc, size_t srcSize,
                    void* pWork,      size_t workSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    NN_SDK_REQUIRES(workSize >= DecompressZlibWorkBufferSize);

    return DecompressZlibCommon(pDst, dstSize, pSrc, srcSize,  pWork, workSize, detail::ZlibFormatWbits);
}

bool DecompressGzip(void* pDst,       size_t dstSize,
                    const void* pSrc, size_t srcSize,
                    void* pWork,      size_t workSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    NN_SDK_REQUIRES(workSize >= DecompressGzipWorkBufferSize);

    return DecompressZlibCommon(pDst, dstSize, pSrc, srcSize,  pWork, workSize, detail::GzipFormatWbits);
}


bool DecompressDeflate(void* pDst,       size_t dstSize,
                       const void* pSrc, size_t srcSize,
                       void* pWork,      size_t workSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pDst);
    NN_SDK_ASSERT_NOT_NULL(pSrc);
    NN_SDK_ASSERT_NOT_NULL(pWork);

    NN_SDK_REQUIRES(workSize >= DecompressDeflateWorkBufferSize);

    return DecompressZlibCommon(pDst, dstSize, pSrc, srcSize,  pWork, workSize, detail::RawDeflateFormatWbits);
}

} //util
} //nn
