﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/util/util_CharacterEncoding.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace util {

namespace
{

const int ToBufferLength = 32;  // GetLength で利用する変換用一時バッファの長さ

// UTF-8 の 1 バイト目から 1 コードポイントを何バイトで表現するかを
// 引くためのテーブル
const int8_t Utf8NbytesInnerTable[257] =
{
    -1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  // 0x00
    1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x10
    1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x20
    1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x30
    1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x40
    1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x50
    1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x60
    1,  1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,     // 0x70
    0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0x80
    0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0x90
    0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0xA0
    0,  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,     // 0xB0
    2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,     // 0xC0
    2,  2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2,     // 0xD0
    3,  3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3,     // 0xE0
    4,  4, 4, 4, 4, 4, 4, 4, 5, 5, 5, 5, 6, 6, 7, 8,     // 0xF0
};
const char* const Utf8NbytesTable = reinterpret_cast<const char*>(&Utf8NbytesInnerTable[1]);

/**
 * @brief       UTF-8 から UTF-16 への変換を行います。
 *
 * @param[out]  pDstCount   変換された文字数
 * @param[out]  pSrcCount   変換に成功した文字数
 * @param[out]  pDst        変換された UTF-16 文字列が格納される領域へのポインタ(ヌル終端しない)
 * @param[in]   dstLength   pDst のバッファサイズ(文字数)
 * @param[in]   pSrc        UTF-8 文字列が格納されている領域へのポインタ(ヌル終端しない)
 * @param[in]   srcLength   pSrc のバッファサイズ(文字数)
 *
 * @retval      CharacterEncodingResult_Success             正常に変換されました。
 * @retval      CharacterEncodingResult_InsufficientLength  変換先の配列の長さが不足しました。
 * @retval      CharacterEncodingResult_InvalidFormat       変換できない文字が検出されました。
 */
CharacterEncodingResult ConvertStringUtf8ToUtf16Impl(
    size_t* pDstCount, size_t* pSrcCount, uint16_t* pDst, int dstLength, const char* pSrc, int srcLength)
{
    NN_SDK_ASSERT_NOT_NULL(pDstCount);
    NN_SDK_ASSERT_NOT_NULL(pSrcCount);

    if (srcLength == 0)
    {
        *pDstCount = 0;
        *pSrcCount = 0;
        return CharacterEncodingResult_Success;
    }

    const unsigned char* ptr = reinterpret_cast<const unsigned char*>(pSrc);
    uint32_t character;
    size_t srcRemains = srcLength;
    size_t dstRemains = dstLength;

    for (;;)
    {
        character = *ptr;
        // 先頭バイトから変換テーブルを使い 1 コードポイントの長さ(バイト数)を取得
        size_t length = Utf8NbytesTable[character];
        if (srcRemains < length)
        {
            // 変換するべき 1 コードポイントの長さより残りの pSrc の長さが少ない
            goto MEMUTF8_UTF16_EILSEQ;
        }

        if (dstRemains == 0)
        {
            goto MEMUTF8_UTF16_FULL;
        }

        // switch内のコードに共通項があるがcaseごとに書いた方が高速だった
        switch (length)
        {
        case 1:
            *pDst = static_cast<uint16_t>(character);
            ++pDst;
            --dstRemains;
            ptr += 1;
            srcRemains -= 1;
            if (srcRemains == 0)
            {
                goto MEMUTF8_UTF16_END;
            }
            break;
        case 2:
            if ((*ptr & 0x1E) != 0)
            {
                if ((*(ptr + 1) & 0xC0) == 0x80)
                {
                    character = ((*ptr & 0x1F) << 6) | (*(ptr + 1) & 0x3F);
                    *pDst = static_cast<uint16_t>(character);
                    ++pDst;
                    --dstRemains;
                    ptr += 2;
                    srcRemains -= 2;
                    if (srcRemains == 0)
                    {
                        goto MEMUTF8_UTF16_END;
                    }
                    break;
                }
                // エラー: 上位2bitは10である必要がある。
            }
            // エラー: 冗長表現はエラー
            goto MEMUTF8_UTF16_EILSEQ;
        case 3:
            if ((*(ptr + 1) & 0xC0) == 0x80 &&
                (*(ptr + 2) & 0xC0) == 0x80)
            {
                character = ((*ptr & 0x0F) << 12) |
                    ((*(ptr + 1) & 0x3F) << 6) |
                    (*(ptr + 2) & 0x3F);
                // U+800未満とU+D800-U+DFFF(サロゲートコード)を弾いている
                if (character & 0xF800 &&
                    (character & 0xF800) != 0xD800)
                {
                    *pDst = static_cast<uint16_t>(character);
                    ++pDst;
                    --dstRemains;
                    ptr += 3;
                    srcRemains -= 3;
                    if (srcRemains == 0)
                    {
                        goto MEMUTF8_UTF16_END;
                    }
                    break;
                }
                // エラー: 冗長表現やサロゲートコードはエラー
            }
            // エラー: 上位2bitは10である必要がある。
            goto MEMUTF8_UTF16_EILSEQ;
        case 4:
            if ((*(ptr + 1) & 0xC0) == 0x80 &&
                (*(ptr + 2) & 0xC0) == 0x80 &&
                (*(ptr + 3) & 0xC0) == 0x80)
            {
                character = ((*ptr & 0x07) << 18) |
                    ((*(ptr + 1) & 0x3F) << 12) |
                    ((*(ptr + 2) & 0x3F) << 6) |
                    (*(ptr + 3) & 0x3F);
                if (character >= 0x10000 && character < 0x110000)
                {
                    if (dstRemains == 1)
                    {
                        goto MEMUTF8_UTF16_FULL;
                    }
                    character -= 0x10000;  // 0 - 0xFFFFF
                    pDst[0] = 0xD800 + static_cast<uint16_t>(character >> 10);
                    pDst[1] = 0xDC00 + static_cast<uint16_t>(character & 0x3FF);
                    pDst += 2;
                    dstRemains -= 2;
                    ptr += 4;
                    srcRemains -= 4;
                    if (srcRemains == 0)
                    {
                        goto MEMUTF8_UTF16_END;
                    }
                    break;
                }
                // エラー: 0x110000以降はエラー
                // エラー: 冗長表現はエラー
            }
            // エラー: 上位2bitは10である必要がある。
            goto MEMUTF8_UTF16_EILSEQ;
        default:
            goto MEMUTF8_UTF16_EILSEQ;
        }
    }

MEMUTF8_UTF16_END:
    *pDstCount = dstLength - dstRemains;
    *pSrcCount = srcLength - srcRemains;
    return CharacterEncodingResult_Success;

MEMUTF8_UTF16_EILSEQ:
    *pDstCount = dstLength - dstRemains;
    *pSrcCount = srcLength - srcRemains;
    return CharacterEncodingResult_InvalidFormat;

MEMUTF8_UTF16_FULL:
    *pDstCount = dstLength - dstRemains;
    *pSrcCount = srcLength - srcRemains;
    return CharacterEncodingResult_InsufficientLength;
} // NOLINT(impl/function_size)

/**
 * @brief       UTF-16 から UTF-32 に 1 文字だけ変換します。
 *
 * @param[out]  pOutUtf32   変換された文字
 * @param[in]   upper       UTF-16 文字(サロゲートペアの場合は上位サロゲート)
 * @param[in]   lower       UTF-16 文字下位サロゲート。サロゲートペアでない場合は利用されません
 *
 * @retval      -2          下位サロゲートが不正で変換できなかった場合
 * @retval      -1          上位サロゲートが不正で変換できなかった場合
 * @retval      1           upper のみが UTF-32 に変換された場合
 * @retval      2           upper と lower が組み合わされて UTF-32 に変換された場合
 */
int ConvertCharacterUtf16ToUtf32(uint32_t* pOutUtf32, uint16_t upper, uint16_t lower)
{
    if ((upper & 0xF800) == 0xD800)
    {
        if (upper & 0x0400)
        {
            return -1;
        }
        *pOutUtf32 = 0x10000 + ((upper - 0xD800) << 10) + (lower - 0xDC00);
        if ((lower & 0xFC00) != 0xDC00)
        {
            return -2;
        }
        return 2;
    }
    else
    {
        *pOutUtf32 = upper;
        return 1;
    }
}

/**
 * @brief       UTF-16 から UTF-8 への変換を行います。
 *
 * @param[out]  pDstCount   変換された文字数
 * @param[out]  pSrcCount   変換に成功した文字数
 * @param[out]  pDst        変換された UTF-8 文字列が格納される領域へのポインタ(ヌル終端しない)
 * @param[in]   dstLength   pDst のバッファサイズ(文字数)
 * @param[in]   pSrc        UTF-16 文字列が格納されている領域へのポインタ(ヌル終端しない)
 * @param[in]   srcLength   pSrc のバッファサイズ(文字数)
 *
 * @retval      CharacterEncodingResult_Success             正常に変換されました。
 * @retval      CharacterEncodingResult_InsufficientLength  変換先の配列の長さが不足しました。
 * @retval      CharacterEncodingResult_InvalidFormat       変換できない文字が検出されました。
 */
CharacterEncodingResult ConvertStringUtf16ToUtf8Impl(
    size_t* pDstCount, size_t* pSrcCount, char* pDst, int dstLength, const uint16_t* pSrc, int srcLength)
{
    NN_SDK_ASSERT_NOT_NULL(pDstCount);
    NN_SDK_ASSERT_NOT_NULL(pSrcCount);

    if (srcLength == 0)
    {
        *pDstCount = 0;
        *pSrcCount = 0;
        return CharacterEncodingResult_Success;
    }

    uint16_t character;
    size_t srcRemains = srcLength;
    size_t dstRemains = dstLength;
    for (;;)
    {
        while ((character = *pSrc) < 0x80)
        {
            if (dstRemains == 0)
            {
                goto MEMUTF16_UTF8_FULL;
            }
            *pDst = static_cast<char>(character);
            ++pDst;
            ++pSrc;
            --dstRemains;
            if (--srcRemains == 0)
            {
                goto MEMUTF16_UTF8_END;
            }
        }
        while ((character = *pSrc) >= 0x80)
        {
            if (!(character & ~0x7FF))
            {
                if (dstRemains < 2)
                {
                    goto MEMUTF16_UTF8_FULL;
                }
                pDst[0] = static_cast<char>(((character >> 6) & 0x1F) | 0xC0);
                pDst[1] = static_cast<char>((character & 0x3F) | 0x80);
                dstRemains -= 2;
                pDst += 2;
            }
            else if (character < 0xD800 || character >= 0xE000)
            {
                if (dstRemains < 3)
                {
                    goto MEMUTF16_UTF8_FULL;
                }
                pDst[0] = static_cast<char>(((character >> 12) & 0x0F) | 0xE0);
                pDst[1] = static_cast<char>(((character >> 6) & 0x3F) | 0x80);
                pDst[2] = static_cast<char>((character & 0x3F) | 0x80);
                dstRemains -= 3;
                pDst += 3;
            }
            else
            {
                uint32_t utf32 = 0;
                if (srcRemains == 1)
                {
                    // 上位サロゲートしかない場合は不正
                    // ただしサロゲート上位のコード自体が正しいならば、部分的に変換
                    if ((character & 0xF800) == 0xD800 && !(character & 0x0400))
                    {
                        utf32 = 0x10000 + ((character - 0xD800) << 10);
                        pDst[0] = static_cast<char>((utf32 >> 18) | 0xF0);
                        --dstRemains;
                        ++pDst;
                    }
                    goto MEMUTF16_UTF8_EILSEQ;
                }
                ++pSrc;
                int returnVal = ConvertCharacterUtf16ToUtf32(&utf32, character, *pSrc);
                if (returnVal < 0)
                {
                    if (returnVal == -2)
                    {
                        // 不正なサロゲート下位が検出されたが、
                        // サロゲート上位は不正でないので部分的に変換。
                        pDst[0] = static_cast<char>((utf32 >> 18) | 0xF0);
                        --dstRemains;
                        ++pDst;
                    }
                    goto MEMUTF16_UTF8_EILSEQ;
                }
                if (dstRemains < 4)
                {
                    goto MEMUTF16_UTF8_FULL;
                }
                --srcRemains;
                pDst[0] = static_cast<char>((utf32 >> 18) | 0xF0);
                pDst[1] = static_cast<char>(((utf32 >> 12) & 0x3F) | 0x80);
                pDst[2] = static_cast<char>(((utf32 >> 6) & 0x3F) | 0x80);
                pDst[3] = static_cast<char>((utf32 & 0x3F) | 0x80);
                dstRemains -= 4;
                pDst += 4;
            }
            ++pSrc;
            if (--srcRemains == 0)
            {
                goto MEMUTF16_UTF8_END;
            }
        }
    }

MEMUTF16_UTF8_END:
    *pDstCount = dstLength - dstRemains;
    *pSrcCount = srcLength - srcRemains;
    return CharacterEncodingResult_Success;

MEMUTF16_UTF8_EILSEQ:
    *pDstCount = dstLength - dstRemains;
    *pSrcCount = srcLength - srcRemains;
    return CharacterEncodingResult_InvalidFormat;

MEMUTF16_UTF8_FULL:
    *pDstCount = dstLength - dstRemains;
    *pSrcCount = srcLength - srcRemains;
    return CharacterEncodingResult_InsufficientLength;
} // NOLINT(impl/function_size)

/**
 * @brief       UTF-16 文字列の長さを返します。(\0は含みません)
 */
size_t GetLengthOfUtf16(const uint16_t* str)
{
    if (!str)
    {
        return 0;
    }
    size_t len = 0;
    // 単純な while より 1, 2 割処理時間が削減される
    while (str[len] != '\0')
    {
        if (str[len + 1] == '\0')
        {
            return len + 1;
        }
        if (str[len + 2] == '\0')
        {
            return len + 2;
        }
        if (str[len + 3] == '\0')
        {
            return len + 3;
        }
        len += 4;
    }
    return len;
}

}   // unnamed namespace

CharacterEncodingResult ConvertStringUtf8ToUtf16Native(
    uint16_t* pDst, int dstLength, const char* pSrc, int srcLength
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES(0 <= srcLength, "srcLength must not be negative.");
    NN_SDK_REQUIRES(0 <= dstLength, "dstLength must not be negative.");

    size_t dstCount;
    size_t srcCount;
    NN_UNUSED(dstCount);
    NN_UNUSED(srcCount);
    return ConvertStringUtf8ToUtf16Impl(&dstCount, &srcCount, pDst, dstLength, pSrc, srcLength);
}
CharacterEncodingResult ConvertStringUtf8ToUtf16Native(
    char16_t* pDst, int dstLength, const char* pSrc, int srcLength
) NN_NOEXCEPT
{
    return ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(pDst), dstLength, pSrc, srcLength);
}

CharacterEncodingResult ConvertStringUtf8ToUtf16Native(
    uint16_t* pDst, int dstLength, const char* pSrc
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES(0 <= dstLength, "dstLength must not be negative.");

    int length = static_cast<int>(std::strlen(pSrc));
    NN_SDK_ASSERT(0 <= length);

    size_t dstCount;
    size_t srcCount;
    NN_UNUSED(srcCount);
    CharacterEncodingResult result = ConvertStringUtf8ToUtf16Impl(&dstCount, &srcCount, pDst, dstLength - 1, pSrc, length);
    if (result == CharacterEncodingResult_Success)
    {
        pDst[dstCount] = '\0';
    }
    return result;
}
CharacterEncodingResult ConvertStringUtf8ToUtf16Native(
    char16_t* pDst, int dstLength, const char* pSrc
) NN_NOEXCEPT
{
    return ConvertStringUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(pDst), dstLength, pSrc);
}

CharacterEncodingResult ConvertStringUtf16NativeToUtf8(
    char* pDst, int dstLength, const uint16_t* pSrc, int srcLength
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES(0 <= srcLength, "srcLength must not be negative.");
    NN_SDK_REQUIRES(0 <= dstLength, "dstLength must not be negative.");

    size_t dstCount;
    size_t srcCount;
    NN_UNUSED(dstCount);
    NN_UNUSED(srcCount);
    return ConvertStringUtf16ToUtf8Impl(&dstCount, &srcCount, pDst, dstLength, pSrc, srcLength);
}
CharacterEncodingResult ConvertStringUtf16NativeToUtf8(
    char* pDst, int dstLength, const char16_t* pSrc, int srcLength
) NN_NOEXCEPT
{
    return ConvertStringUtf16NativeToUtf8(pDst, dstLength, reinterpret_cast<const uint16_t*>(pSrc), srcLength);
}

CharacterEncodingResult ConvertStringUtf16NativeToUtf8(
    char* pDst, int dstLength, const uint16_t* pSrc
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES(0 <= dstLength, "dstLength must not be negative.");

    int length = static_cast<int>(GetLengthOfUtf16(pSrc));
    NN_SDK_ASSERT(0 <= length);

    size_t dstCount;
    size_t srcCount;
    NN_UNUSED(srcCount);
    CharacterEncodingResult result = ConvertStringUtf16ToUtf8Impl(&dstCount, &srcCount, pDst, dstLength - 1, pSrc, length);
    if (result == CharacterEncodingResult_Success)
    {
        pDst[dstCount] = '\0';
    }
    return result;
}
CharacterEncodingResult ConvertStringUtf16NativeToUtf8(
    char* pDst, int dstLength, const char16_t* pSrc
) NN_NOEXCEPT
{
    return ConvertStringUtf16NativeToUtf8(pDst, dstLength, reinterpret_cast<const uint16_t*>(pSrc));
}

CharacterEncodingResult GetLengthOfConvertedStringUtf8ToUtf16Native(
    int* pOutLength, const char* pSrc, int srcLength
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pOutLength);
    NN_SDK_REQUIRES(0 <= srcLength, "srcLength must not be negative.");

    uint16_t pBuffer[ToBufferLength];
    int dstLength = 0;
    int srcIndex = 0;

    while (0 < srcLength)
    {
        size_t dstCount;
        size_t srcCount;
        CharacterEncodingResult result = ConvertStringUtf8ToUtf16Impl(&dstCount, &srcCount, pBuffer, ToBufferLength, &pSrc[srcIndex], srcLength);
        if (CharacterEncodingResult_InvalidFormat == result)
        {
            return result;
        }
        srcIndex += static_cast<int>(srcCount);
        srcLength -= static_cast<int>(srcCount);
        dstLength += static_cast<int>(dstCount);
    }
    NN_SDK_ASSERT(0 <= dstLength);
    *pOutLength = dstLength;
    return CharacterEncodingResult_Success;
}

CharacterEncodingResult GetLengthOfConvertedStringUtf16NativeToUtf8(
    int* pOutLength, const uint16_t* pSrc, int srcLength
) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pOutLength);
    NN_SDK_REQUIRES(0 <= srcLength, "srcLength must not be negative.");

    char pBuffer[ToBufferLength];
    int dstLength = 0;
    int srcIndex = 0;

    while (0 < srcLength)
    {
        size_t dstCount;
        size_t srcCount;
        CharacterEncodingResult result = ConvertStringUtf16ToUtf8Impl(&dstCount, &srcCount, pBuffer, ToBufferLength, &pSrc[srcIndex], srcLength);
        if (CharacterEncodingResult_InvalidFormat == result)
        {
            return result;
        }
        srcIndex += static_cast<int>(srcCount);
        srcLength -= static_cast<int>(srcCount);
        dstLength += static_cast<int>(dstCount);
    }
    NN_SDK_ASSERT(0 <= dstLength);
    *pOutLength = dstLength;
    return CharacterEncodingResult_Success;
}
CharacterEncodingResult GetLengthOfConvertedStringUtf16NativeToUtf8(
    int* pOutLength, const char16_t* pSrc, int srcLength
) NN_NOEXCEPT
{
    return GetLengthOfConvertedStringUtf16NativeToUtf8(pOutLength, reinterpret_cast<const uint16_t*>(pSrc), srcLength);
}

CharacterEncodingResult ConvertStringUtf8ToUtf32(
    uint32_t* pDst, int dstLength, const char* pSrc
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES(0 <= dstLength, "dstLength must not be negative.");

    int n = 0;
    const unsigned char* p = reinterpret_cast<const unsigned char*>(pSrc);
    uint32_t c;
    while ((c = *p) != 0)
    {
        switch (Utf8NbytesTable[c])
        {
        case 1:
            p++;
            break;
        case 2:
            if ((*p & 0x1E) != 0)
            {
                if (Utf8NbytesTable[*(p + 1)] == 0)
                {
                    c = ((*p & 0x1F) << 6) | (*(p + 1) & 0x3F);
                    p += 2;
                    break;
                }
                // エラー: 上位2bitは10である必要がある。
            }
            // エラー: 冗長表現はエラー
            return CharacterEncodingResult_InvalidFormat;
        case 3:
            if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0)
            {
                c = ((*p & 0x0F) << 12) | ((*(p + 1) & 0x3F) << 6) | (*(p + 2) & 0x3F);
                if ((c & 0xF800) && (c & 0xF800) != 0xD800)
                {
                    p += 3;
                    break;
                }
                // エラー: 冗長表現やサロゲートコードはエラー
            }
            // エラー: 上位2bitは10である必要がある。
            return CharacterEncodingResult_InvalidFormat;
        case 4:
            if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0 && Utf8NbytesTable[*(p + 3)] == 0)
            {
                c = ((*p & 0x07) << 18) | ((*(p + 1) & 0x3F) << 12) | ((*(p + 2) & 0x3F) << 6) | (*(p + 3) & 0x3F);
                if (c >= 0x10000 && c < 0x110000)
                {
                    p += 4;
                    break;
                }
                // エラー: 0x110000以降はエラー
                // エラー: 冗長表現はエラー
            }
            // エラー: 上位2bitは10である必要がある。
            return CharacterEncodingResult_InvalidFormat;
        default:
            return CharacterEncodingResult_InvalidFormat;
        }
        if (n >= dstLength)
        {
            return CharacterEncodingResult_InsufficientLength;
        }
        pDst[n] = c;
        n++;
    }
    if (n >= dstLength)
    {
        return CharacterEncodingResult_InsufficientLength;
    }
    pDst[n] = 0;
    return CharacterEncodingResult_Success;
}
CharacterEncodingResult ConvertStringUtf8ToUtf32(
    char32_t* pDst, int dstLength, const char* pSrc
) NN_NOEXCEPT
{
    return ConvertStringUtf8ToUtf32(reinterpret_cast<uint32_t*>(pDst), dstLength, pSrc);
}

CharacterEncodingResult ConvertStringUtf32ToUtf8(
    char* pDst, int dstLength, const uint32_t* pSrc
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);
    NN_SDK_REQUIRES(0 <= dstLength, "dstLength must not be negative.");

    int n = 0;
    uint32_t c;
    while ((c = *pSrc) != 0)
    {
        int len4;
        if (c < 0x80)
        {
            if (n >= dstLength)
            {
                return CharacterEncodingResult_InsufficientLength;
            }
            pDst[n] = static_cast<char>(c);
            len4 = 1;
        }
        else if (!(c & ~0x7FF))
        {
            if (n + 1 >= dstLength)
            {
                return CharacterEncodingResult_InsufficientLength;
            }
            pDst[n] = static_cast<char>(((c >> 6) & 0x1F) | 0xC0);
            pDst[n + 1] = static_cast<char>((c & 0x3F) | 0x80);
            len4 = 2;
        }
        else if (!(c & ~0xFFFF))
        {
            if (c >= 0xD800 && c <= 0xDFFF)
            {
                // ペアになってないので不正
                return CharacterEncodingResult_InvalidFormat;
            }
            if (n + 2 >= dstLength)
            {
                return CharacterEncodingResult_InsufficientLength;
            }
            pDst[n] = static_cast<char>(((c >> 12) & 0x0F) | 0xE0);
            pDst[n + 1] = static_cast<char>(((c >> 6) & 0x3F) | 0x80);
            pDst[n + 2] = static_cast<char>((c & 0x3F) | 0x80);
            len4 = 3;
        }
        else if (c < 0x110000)
        {
            if (n + 3 >= dstLength)
            {
                return CharacterEncodingResult_InsufficientLength;
            }
            pDst[n] = static_cast<char>((c >> 18) | 0xF0);
            pDst[n + 1] = static_cast<char>(((c >> 12) & 0x3F) | 0x80);
            pDst[n + 2] = static_cast<char>(((c >> 6) & 0x3F) | 0x80);
            pDst[n + 3] = static_cast<char>((c & 0x3F) | 0x80);
            len4 = 4;
        }
        else
        {
            return CharacterEncodingResult_InvalidFormat;
        }
        pSrc++;
        n += len4;
    }
    if (n >= dstLength)
    {
        return CharacterEncodingResult_InsufficientLength;
    }
    pDst[n] = 0;
    return CharacterEncodingResult_Success;
}
CharacterEncodingResult ConvertStringUtf32ToUtf8(
    char* pDst, int dstLength, const char32_t* pSrc
) NN_NOEXCEPT
{
    return ConvertStringUtf32ToUtf8(pDst, dstLength, reinterpret_cast<const uint32_t*>(pSrc));
}

CharacterEncodingResult GetLengthOfConvertedStringUtf8ToUtf32(
    int* pOutLength, const char* pSrc
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pOutLength);

    *pOutLength = 0;
    const unsigned char* p = reinterpret_cast<const unsigned char*>(pSrc);
    uint32_t c;
    while ((c = *p) != 0)
    {
        switch (Utf8NbytesTable[c])
        {
        case 1:
            p++;
            break;
        case 2:
            if ((*p & 0x1E) != 0)
            {
                if (Utf8NbytesTable[*(p + 1)] == 0)
                {
                    c = ((*p & 0x1F) << 6) | (*(p + 1) & 0x3F);
                    p += 2;
                    break;
                }
                // エラー: 上位2bitは10である必要がある。
            }
            // エラー: 冗長表現はエラー
            return CharacterEncodingResult_InvalidFormat;
        case 3:
            if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0)
            {
                c = ((*p & 0x0F) << 12) | ((*(p + 1) & 0x3F) << 6) | (*(p + 2) & 0x3F);
                if ((c & 0xF800) && (c & 0xF800) != 0xD800)
                {
                    p += 3;
                    break;
                }
                // エラー: 冗長表現やサロゲートコードはエラー
            }
            // エラー: 上位2bitは10である必要がある。
            return CharacterEncodingResult_InvalidFormat;
        case 4:
            if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0 && Utf8NbytesTable[*(p + 3)] == 0)
            {
                c = ((*p & 0x07) << 18) | ((*(p + 1) & 0x3F) << 12) | ((*(p + 2) & 0x3F) << 6) | (*(p + 3) & 0x3F);
                if (c >= 0x10000 && c < 0x110000)
                {
                    p += 4;
                    break;
                }
                // エラー: 0x110000以降はエラー
                // エラー: 冗長表現はエラー
            }
            // エラー: 上位2bitは10である必要がある。
            return CharacterEncodingResult_InvalidFormat;
        default:
            return CharacterEncodingResult_InvalidFormat;
        }
        (*pOutLength)++;
    }
    return CharacterEncodingResult_Success;
}

CharacterEncodingResult GetLengthOfConvertedStringUtf32ToUtf8(
    int* pOutLength, const uint32_t* pSrc
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pOutLength);

    *pOutLength = 0;
    uint32_t c;
    while ((c = *pSrc) != 0)
    {
        int len4;
        if (c < 0x80)
        {
            len4 = 1;
        }
        else if (!(c & ~0x7FF))
        {
            len4 = 2;
        }
        else if (!(c & ~0xFFFF))
        {
            if (c >= 0xD800 && c <= 0xDFFF)
            {
                // ペアになってないので不正
                return CharacterEncodingResult_InvalidFormat;
            }
            len4 = 3;
        }
        else if (c < 0x110000)
        {
            len4 = 4;
        }
        else
        {
            return CharacterEncodingResult_InvalidFormat;
        }
        pSrc++;
        *pOutLength += len4;
    }
    return CharacterEncodingResult_Success;
}
CharacterEncodingResult GetLengthOfConvertedStringUtf32ToUtf8(
    int* pOutLength, const char32_t* pSrc
) NN_NOEXCEPT
{
    return GetLengthOfConvertedStringUtf32ToUtf8(pOutLength, reinterpret_cast<const uint32_t*>(pSrc));
}

CharacterEncodingResult ConvertCharacterUtf8ToUtf16Native(
    uint16_t* pDst, const char* pSrc
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);

    const unsigned char *ptr = reinterpret_cast<const unsigned char *>(pSrc);
    char bufferSrc[5] = { pSrc[0], 0, 0, 0, 0 };

    if (ptr[0] >= 0xC2u && ptr[0] < 0xE0u)
    {
        bufferSrc[1] = pSrc[1];
    }
    else if (ptr[0] >= 0xE0u && ptr[0] < 0xF0u)
    {
        bufferSrc[1] = pSrc[1];
        bufferSrc[2] = pSrc[2];
    }
    else if (ptr[0] >= 0xF0u && ptr[0] < 0xF8u)
    {
        bufferSrc[1] = pSrc[1];
        bufferSrc[2] = pSrc[2];
        bufferSrc[3] = pSrc[3];
    }
    uint16_t bufferDst[3] = { 0 };
    CharacterEncodingResult result = ConvertStringUtf8ToUtf16Native(bufferDst, 3, bufferSrc);
    pDst[0] = bufferDst[0];
    pDst[1] = bufferDst[1];

    return result;
}
CharacterEncodingResult ConvertCharacterUtf8ToUtf16Native(
    char16_t* pDst, const char* pSrc
) NN_NOEXCEPT
{
    return ConvertCharacterUtf8ToUtf16Native(reinterpret_cast<uint16_t*>(pDst), pSrc);
}

CharacterEncodingResult ConvertCharacterUtf16NativeToUtf8(
    char* pDst, const uint16_t* pSrc
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);
    NN_SDK_REQUIRES_NOT_NULL(pDst);

    uint16_t bufferSrc[3] = { pSrc[0], 0, 0 };
    if (pSrc[0] >= 0xD800 && pSrc[0] < 0xE000)
    {
        // サロゲートペアなら 2 バイト必要
        bufferSrc[1] = pSrc[1];
    }

    char bufferDst[5] = { 0 };
    CharacterEncodingResult result = ConvertStringUtf16NativeToUtf8(bufferDst, 5, bufferSrc);
    pDst[0] = bufferDst[0];
    pDst[1] = bufferDst[1];
    pDst[2] = bufferDst[2];
    pDst[3] = bufferDst[3];
    return result;
}
CharacterEncodingResult ConvertCharacterUtf16NativeToUtf8(
    char* pDst, const char16_t* pSrc
) NN_NOEXCEPT
{
    return ConvertCharacterUtf16NativeToUtf8(pDst, reinterpret_cast<const uint16_t*>(pSrc));
}

CharacterEncodingResult ConvertCharacterUtf8ToUtf32(
    uint32_t* pDst, const char* pSrc
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSrc);

    const unsigned char* p = reinterpret_cast<const unsigned char*>(pSrc);
    switch (Utf8NbytesTable[*p])
    {
    case 1:
        *pDst = static_cast<uint32_t>(*p);
        return CharacterEncodingResult_Success;
    case 2:
        if ((*p & 0x1E) != 0)
        {
            if (Utf8NbytesTable[*(p + 1)] == 0)
            {
                *pDst = static_cast<uint32_t>(((*p & 0x1F) << 6) | (*(p + 1) & 0x3F));
                return CharacterEncodingResult_Success;
            }
        }
        break;
    case 3:
        if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0)
        {
            uint32_t c = ((*p & 0x0F) << 12) | ((*(p + 1) & 0x3F) << 6) | (*(p + 2) & 0x3F);
            if ((c & 0xF800) && (c & 0xF800) != 0xD800)
            {
                *pDst = static_cast<uint32_t>(c);
                return CharacterEncodingResult_Success;
            }
            // エラー: 冗長表現やサロゲートコードはエラー
        }
        break;
    case 4:
        if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0 && Utf8NbytesTable[*(p + 3)] == 0)
        {
            uint32_t c = ((*p & 0x07) << 18) | ((*(p + 1) & 0x3F) << 12) | ((*(p + 2) & 0x3F) << 6) | (*(p + 3) & 0x3F);
            if (c >= 0x10000 && c < 0x110000)
            {
                *pDst = static_cast<uint32_t>(c);
                return CharacterEncodingResult_Success;
            }
            // エラー: 0x110000以降はエラー
            // エラー: 冗長表現はエラー
        }
        break;
    default:
        break;
    }
    return CharacterEncodingResult_InvalidFormat;
}
CharacterEncodingResult ConvertCharacterUtf8ToUtf32(
    char32_t* pDst, const char* pSrc
) NN_NOEXCEPT
{
    return ConvertCharacterUtf8ToUtf32(reinterpret_cast<uint32_t*>(pDst), pSrc);
}

CharacterEncodingResult ConvertCharacterUtf32ToUtf8(
    char* pDst, uint32_t src
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDst);

    pDst[0] = 0;
    pDst[1] = 0;
    pDst[2] = 0;
    pDst[3] = 0;
    if (!(src & ~0x7F))
    {
        pDst[0] = static_cast<char>(src);
    }
    else if (!(src & ~0x7FF))
    {
        pDst[0] = static_cast<char>(((src >> 6) & 0x1F) | 0xC0);
        pDst[1] = static_cast<char>((src & 0x3F) | 0x80);
    }
    else if (!(src & ~0xFFFF))
    {
        if (src >= 0xD800 && src <= 0xDFFF)
        {
            // ペアになってないので不正
            return CharacterEncodingResult_InvalidFormat;
        }
        pDst[0] = static_cast<char>(((src >> 12) & 0x0F) | 0xE0);
        pDst[1] = static_cast<char>(((src >> 6) & 0x3F) | 0x80);
        pDst[2] = static_cast<char>((src & 0x3F) | 0x80);
    }
    else if (src < 0x110000)
    {
        pDst[0] = static_cast<char>((src >> 18) | 0xF0);
        pDst[1] = static_cast<char>(((src >> 12) & 0x3F) | 0x80);
        pDst[2] = static_cast<char>(((src >> 6) & 0x3F) | 0x80);
        pDst[3] = static_cast<char>((src & 0x3F) | 0x80);
    }
    else
    {
        return CharacterEncodingResult_InvalidFormat;
    }
    return CharacterEncodingResult_Success;
}
CharacterEncodingResult ConvertCharacterUtf32ToUtf8(
    char* pDst, char32_t src
) NN_NOEXCEPT
{
    return ConvertCharacterUtf32ToUtf8(pDst, static_cast<uint32_t>(src));
}

CharacterEncodingResult PickOutCharacterFromUtf8String(
    char* pChar, const char** pStr
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pChar);
    NN_SDK_REQUIRES_NOT_NULL(pStr);
    NN_SDK_REQUIRES_NOT_NULL(*pStr);

    pChar[0] = 0;
    pChar[1] = 0;
    pChar[2] = 0;
    pChar[3] = 0;

    uint32_t c;
    const unsigned char* p = reinterpret_cast<const unsigned char*>(*pStr);
    c = *p;
    switch (Utf8NbytesTable[c])
    {
    case 1:
        pChar[0] = (*pStr)[0];
        (*pStr)++;
        break;
    case 2:
        if ((*p & 0x1E) != 0)
        {
            if (Utf8NbytesTable[*(p + 1)] == 0)
            {
                c = ((*p & 0x1F) << 6) | (*(p + 1) & 0x3F);
                pChar[0] = (*pStr)[0];
                pChar[1] = (*pStr)[1];
                (*pStr) += 2;
                break;
            }
            // エラー: 上位2bitは10である必要がある。
        }
        // エラー: 冗長表現はエラー
        return CharacterEncodingResult_InvalidFormat;
    case 3:
        if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0)
        {
            c = ((*p & 0x0F) << 12) | ((*(p + 1) & 0x3F) << 6) | (*(p + 2) & 0x3F);
            if ((c & 0xF800) && (c & 0xF800) != 0xD800)
            {
                pChar[0] = (*pStr)[0];
                pChar[1] = (*pStr)[1];
                pChar[2] = (*pStr)[2];
                (*pStr) += 3;
                break;
            }
            // エラー: 冗長表現やサロゲートコードはエラー
        }
        // エラー: 上位2bitは10である必要がある。
        return CharacterEncodingResult_InvalidFormat;
    case 4:
        if (Utf8NbytesTable[*(p + 1)] == 0 && Utf8NbytesTable[*(p + 2)] == 0 && Utf8NbytesTable[*(p + 3)] == 0)
        {
            c = ((*p & 0x07) << 18) | ((*(p + 1) & 0x3F) << 12) | ((*(p + 2) & 0x3F) << 6) | (*(p + 3) & 0x3F);
            if (c >= 0x10000 && c < 0x110000)
            {
                pChar[0] = (*pStr)[0];
                pChar[1] = (*pStr)[1];
                pChar[2] = (*pStr)[2];
                pChar[3] = (*pStr)[3];
                (*pStr) += 4;
                break;
            }
            // エラー: 0x110000以降はエラー
            // エラー: 冗長表現はエラー
        }
        // エラー: 上位2bitは10である必要がある。
        return CharacterEncodingResult_InvalidFormat;
    default:
        return CharacterEncodingResult_InvalidFormat;
    }
    return CharacterEncodingResult_Success;
}

}}  // nn::util
