﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cctype>
#include <climits>
#include <cstring>
#include <cstddef>

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_StaticAssert.h>

#include <nn/image/image_ExifBuilder.h>

#include "detail/image_JpegConfig.h"
#include "detail/image_ExifParser.h"

namespace nn { namespace image {

namespace
{
// 各 IFD に記録可能なタグの数 (現行使用で最大でこれだけ、という意味)
//  - 対応するタグを増やした場合には増やさないといけない。
//  - その場合はワークメモリの量も増えるので注意。
const uint16_t Max0thTiffElementNum = 11u;
const uint16_t Max0thExifElementNum = 8u;
const uint16_t Max1stTiffElementNum = 6u;

// 既定のタグの値
const char DefaultMake[] = "Nintendo co., ltd"; // メーカー名(任天堂)
const detail::Rational DefaultXReso = {72, 1}; // X方向画像解像度(72/1)
const detail::Rational DefaultYReso = {72, 1}; // Y方向画像解像度(72/1)
const uint16_t DefaultResoUnit = 2; // 解像度の単位(inch=dpi)
const uint16_t DefaultYccPos = 1; // YCbCrのコンポジション位置(default=中央)
const uint8_t DefaultExifVer[] = {'0', '2', '3', '0'}; // Exifのバージョン(v2.3.0)
const uint8_t DefaultCompoConf[] = {1, 2, 3, 0}; // 色空間(YCbCr)
const uint8_t DefaultFlashpix[] = {'0', '1', '0', '0'}; // 互換性のあるFlashpixのバージョン(v1.0.0)
const uint16_t DefaultColorSpace = 1; // 元画像の色空間(sRGB)
const uint16_t DefaultCompression = 6; // サムネイルの圧縮形式(JPEG)

// @brief 設定可能な情報を保持しておく構造体
struct EmbeddedData
{
    // 0th TIFF IFD.
    const char *software;   // ソフトウェア名
    uint16_t softwareSize;  // ソフトウェア名の終端含む大きさ
    uint16_t orientation;   // 画像の方向
    const char *dateTime;   // 日付
    uint32_t offsetExifIfd; // ExifIFDへのオフセット

    // 0th EXIF IFD.
    const uint8_t *note;    // メーカーノート
    uint16_t noteSize;      // メーカーノートの大きさ
    uint16_t pixelX;        // 実効画像幅
    uint16_t pixelY;        // 実効画像高さ
    const char *uniqueId;   // ユニーク ID

    // 1st TIFF IFD.
    const uint8_t *thumbnail;   // サムネイル
    uint32_t thumbnailOffset;   // サムネイルへのオフセット
    uint32_t thumbnailSize;     // サムネイルのサイズ
};

// @brief コンテキスト構造体。ワークメモリを割り当てる。
struct EmbedContext
{
    detail::TiffHeader tiffHeader; //!< TIFF ヘッダ
    struct
    {
        detail::IfdHeader tiff;   //!< 0th TIFF IFD
        detail::ValuedIfdTag tiffTags[Max0thTiffElementNum];
        detail::IfdHeader exif;   //!< 0th Exif IFD
        detail::ValuedIfdTag exifTags[Max0thExifElementNum];
    } ifd0;
    struct
    {
        detail::IfdHeader tiff;   //!< 1st TIFF IFD
        detail::ValuedIfdTag tiffTags[Max1stTiffElementNum];
    } ifd1;

    EmbeddedData data;
};

// @brief ワークメモリの領域から内部コンテキストを取得する
NN_FORCEINLINE EmbedContext* GetEmbedContext(void *workBuf) NN_NOEXCEPT
{
    return reinterpret_cast<EmbedContext*>(workBuf);
}

/**
    @brief 現在の設定値で 0th TIFF IFD のタグの情報を構成する
 */
uint16_t Setup0thTiffTags(EmbedContext *pContext) NN_NOEXCEPT
{
    detail::ValuedIfdTag *tags = pContext->ifd0.tiffTags;
    uint16_t tagIndex = 0u;

    tags[tagIndex].tag = detail::IfdTagMake;
    tags[tagIndex].tag.valueCount = sizeof(DefaultMake) / sizeof(char);
    tags[tagIndex].value.vAscii = DefaultMake;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Maker:          %s\n", DefaultMake);
    tagIndex ++;

    if (pContext->data.orientation >= 1)
    {
        // 画像方向は、設定されていれば構成に含める。
        NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
        tags[tagIndex].tag = detail::IfdTagOrientaion;
        tags[tagIndex].value.vShort = &pContext->data.orientation;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Orientation:    %u\n", pContext->data.orientation);
        tagIndex ++;
    }

    NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagXResolution;
    tags[tagIndex].value.vRational = &DefaultXReso;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - XResolution:    %u/%u\n", DefaultXReso.numerator, DefaultXReso.denominator);
    tagIndex ++;

    NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagYResolution;
    tags[tagIndex].value.vRational = &DefaultYReso;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - YResolution:    %u/%u\n", DefaultYReso.numerator, DefaultYReso.denominator);
    tagIndex ++;

    NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagResolutionUnit;
    tags[tagIndex].value.vShort = &DefaultResoUnit;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - ResolutionUnit: %u\n", DefaultResoUnit);
    tagIndex ++;

    if (pContext->data.software != nullptr)
    {
        // 作成ソフトウェア名は、設定されていれば構成に含める。
        NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
        tags[tagIndex].tag = detail::IfdTagSoftware;
        tags[tagIndex].tag.valueCount = pContext->data.softwareSize;
        tags[tagIndex].value.vAscii = pContext->data.software;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Software:       %s\n", pContext->data.software);
        tagIndex ++;
    }

    if (pContext->data.dateTime != nullptr)
    {
        // 作成日時は、設定されていれば構成に含める。
        NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
        tags[tagIndex].tag = detail::IfdTagDateTime;
        tags[tagIndex].value.vAscii = pContext->data.dateTime;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - DateTime:       %s\n", pContext->data.dateTime);
        tagIndex ++;
    }

    NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagYccPosition;
    tags[tagIndex].value.vShort = &DefaultYccPos;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - YCbCrPosition:  %u\n", DefaultYccPos);
    tagIndex ++;
    NN_SDK_ASSERT(tagIndex < Max0thTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagExifIfd;
    tags[tagIndex].value.vLong = &pContext->data.offsetExifIfd;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - ExifIfdPointer: <not determined>\n");
    tagIndex ++;

    return tagIndex;
}

/**
    @brief 現在の設定値で 0th Exif IFD のタグの情報を構成する
 */
uint16_t Setup0thExifTags(EmbedContext *pContext) NN_NOEXCEPT
{
    detail::ValuedIfdTag *tags = pContext->ifd0.exifTags;
    uint16_t tagIndex = 0u;

    tags[tagIndex].tag = detail::IfdTagExifVersion;
    tags[tagIndex].value.vUndefined = DefaultExifVer;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(
        " - ExifVer:        %c%c%c%c\n",
        DefaultExifVer[0], DefaultExifVer[1], DefaultExifVer[2], DefaultExifVer[3]);
    tagIndex ++;

    NN_SDK_ASSERT(tagIndex < Max0thExifElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagCompConfig;
    tags[tagIndex].value.vUndefined = DefaultCompoConf;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(
        " - ComponentConf:  %u%u%u%u\n",
        DefaultCompoConf[0], DefaultCompoConf[1], DefaultCompoConf[2], DefaultCompoConf[3]);
    tagIndex ++;

    if (pContext->data.note != nullptr)
    {
        // メーカーノートは、設定されていれば構成に含める
        NN_SDK_ASSERT(tagIndex < Max0thExifElementNum, "Tag overflows\n");
        tags[tagIndex].tag = detail::IfdTagMakerNote;
        tags[tagIndex].tag.valueCount = pContext->data.noteSize;
        tags[tagIndex].value.vUndefined = pContext->data.note;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > MakerNote:      size=%u\n", pContext->data.noteSize);
        tagIndex ++;
    }

    NN_SDK_ASSERT(tagIndex < Max0thExifElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagFlashpixVer;
    tags[tagIndex].value.vUndefined = DefaultFlashpix;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(
        " - FlashpixVer:    %c%c%c%c\n",
        DefaultFlashpix[0], DefaultFlashpix[1], DefaultFlashpix[2], DefaultFlashpix[3]);
    tagIndex ++;

    NN_SDK_ASSERT(tagIndex < Max0thExifElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagColorSpace;
    tags[tagIndex].value.vShort = &DefaultColorSpace;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - ColorSpace:     %u\n", DefaultColorSpace);
    tagIndex ++;

    // この時点では実効画像幅は設定されていないが、サイズ計算のためポインタを登録しておく
    NN_SDK_ASSERT(tagIndex < Max0thExifElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagPxXDimension;
    tags[tagIndex].tag.type = detail::IfdTagType_Short;
    tags[tagIndex].value.vShort = &pContext->data.pixelX;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - PixelXDim:      <not determined>\n");
    tagIndex ++;

    // この時点では実効画像高さは設定されていないが、サイズ計算のためポインタを登録しておく
    NN_SDK_ASSERT(tagIndex < Max0thExifElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagPxYDimension;
    tags[tagIndex].tag.type = detail::IfdTagType_Short;
    tags[tagIndex].value.vShort = &pContext->data.pixelY;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - PixelYDim:      <not determined>\n");
    tagIndex ++;

    if (pContext->data.uniqueId != nullptr)
    {
        // ユニーク ID は、設定されていれば構成に含める
        NN_SDK_ASSERT(tagIndex < Max0thExifElementNum, "Tag overflows\n");
        tags[tagIndex].tag = detail::IfdTagUniqueId;
        tags[tagIndex].value.vAscii = pContext->data.uniqueId;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - UniqueID:       %s\n", pContext->data.uniqueId);
        tagIndex ++;
    }

    return tagIndex;
}

/**
    @brief 現在の設定値で 1st TIFF IFD のタグの情報を構成する
    @details
    この関数はサムネイルが存在する場合にのみ呼ばれ、必ず既定のフィールドすべてを書き込む必要がある。
 */
uint16_t Setup1stTiffTags(EmbedContext *pContext) NN_NOEXCEPT
{
    detail::ValuedIfdTag *tags = pContext->ifd1.tiffTags;
    uint16_t tagIndex = 0u;

    tags[tagIndex].tag = detail::IfdTagCompression;
    tags[tagIndex].value.vShort = &DefaultCompression;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Compression:    %u\n", DefaultCompression);
    tagIndex ++;
    NN_SDK_ASSERT(tagIndex < Max1stTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagXResolution;
    tags[tagIndex].value.vRational = &DefaultXReso;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - XResolution:    %u/%u\n", DefaultXReso.numerator, DefaultXReso.denominator);
    tagIndex ++;
    NN_SDK_ASSERT(tagIndex < Max1stTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagYResolution;
    tags[tagIndex].value.vRational = &DefaultYReso;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - YResolution:    %u/%u\n", DefaultYReso.numerator, DefaultYReso.denominator);
    tagIndex ++;
    NN_SDK_ASSERT(tagIndex < Max1stTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagResolutionUnit;
    tags[tagIndex].value.vShort = &DefaultResoUnit;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - ResolutionUnit: %u\n", DefaultResoUnit);
    tagIndex ++;
    NN_SDK_ASSERT(tagIndex < Max1stTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagJpegFormat;
    tags[tagIndex].value.vLong = &pContext->data.thumbnailOffset;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - JPEGFormat:     <not determined>\n");
    tagIndex ++;
    NN_SDK_ASSERT(tagIndex < Max1stTiffElementNum, "Tag overflows\n");
    tags[tagIndex].tag = detail::IfdTagJpegFormatLen;
    tags[tagIndex].value.vLong = &pContext->data.thumbnailSize;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - JPEGFormatLen:  %u\n", pContext->data.thumbnailSize);
    tagIndex ++;

    return tagIndex;
}
}

size_t ExifBuilder::GetWorkBufferSize() NN_NOEXCEPT
{
    return sizeof(EmbedContext);
}

ExifBuilder::ExifBuilder(void *workBuf, const size_t kWorkBufSize) NN_NOEXCEPT :
    m_Stage(Stage_Registered),
    m_WorkBuf(workBuf)
{
    NN_STATIC_ASSERT(NN_ALIGNOF(EmbedContext) <= NN_ALIGNOF(std::max_align_t));
    NN_SDK_REQUIRES(
        workBuf != nullptr && kWorkBufSize >= this->GetWorkBufferSize() &&
        reinterpret_cast<uintptr_t>(workBuf) % NN_ALIGNOF(std::max_align_t) == 0,
        "Valid workbuffer is required.\n");
    NN_UNUSED(kWorkBufSize);

    EmbedContext *pContext = GetEmbedContext(m_WorkBuf);
    /* 0th TIFF IFD を初期化 */
    pContext->data.software = nullptr;
    pContext->data.softwareSize = 0u;
    pContext->data.orientation = 0u;
    pContext->data.dateTime = nullptr;
    pContext->data.offsetExifIfd = 0ul;
    /* 0th EXIF IFD を初期化 */
    pContext->data.note = nullptr;
    pContext->data.noteSize = 0u;
    pContext->data.pixelX = 0ul;
    pContext->data.pixelY = 0ul;
    pContext->data.uniqueId = nullptr;
    /* 0th TIFF IFD を初期化 */
    pContext->data.thumbnail = nullptr;
    pContext->data.thumbnailOffset = 0ul;
    pContext->data.thumbnailSize = 0ul;
}

ExifBuilder::~ExifBuilder() NN_NOEXCEPT {}

void ExifBuilder::SetOrientation(const ExifOrientation kOrientation) NN_NOEXCEPT
{
#if 0
    switch (kOrientation)
    {
    case ExifOrientation_Normal:
    case ExifOrientation_FlipHorizontal:
    case ExifOrientation_Rotate180:
    case ExifOrientation_FlipVertical:
    case ExifOrientation_FlipTopRightToLeftBottom:
    case ExifOrientation_Rotate270:
    case ExifOrientation_FlipTopLeftToRightBottom:
    case ExifOrientation_Rotate90:
        break;
    default: // 未定義
        NN_SDK_REQUIRES(false, "Unknown orientation is specified\n");
    }
#else
    NN_SDK_REQUIRES(kOrientation >= 1 && kOrientation <= 8, "Unknown orientation is specified");
#endif

    GetEmbedContext(m_WorkBuf)->data.orientation = static_cast<uint16_t>(kOrientation);
    m_Stage = Stage_Registered;
}
void ExifBuilder::SetSoftware(const char *software, const size_t kSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        software == nullptr || (kSize > 0 && kSize <= UINT16_MAX),
        "Valid software name is required if specified.\n");

    // ASCII 文字列 + '\0' かどうかの検査
    if (software != nullptr)
    {
        for (size_t i = 0; i < kSize - 1; i++)
        {
            NN_SDK_REQUIRES((software[i] & 0x80) == 0, "Invalid string (non-ascii)\n");
            NN_SDK_REQUIRES(software[i] != '\0', "Invalid string (length != %d)\n", kSize);
        }
        NN_SDK_REQUIRES(software[kSize - 1] == '\0', "Invalid string (not-terminated)\n");
    }

    EmbedContext *pContext = GetEmbedContext(m_WorkBuf);
    pContext->data.software = software;
    pContext->data.softwareSize = (software == nullptr? 0: static_cast<uint16_t>(kSize));
    m_Stage = Stage_Registered;
}
void ExifBuilder::SetDateTime(const char *dateTime, const size_t kSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        dateTime == nullptr || kSize >= 20,
        "Valid date-time is required if specified.\n");
    NN_UNUSED(kSize);

    // ASCII 19 文字 + '\0' かどうかの検査
    if (dateTime != nullptr)
    {
        for (int i = 0; i < 20 - 1; i++)
        {
            NN_SDK_REQUIRES((dateTime[i] & 0x80) == 0, "Invalid string (non-ascii)\n");
            NN_SDK_REQUIRES(dateTime[i] != '\0', "Invalid string (length != 19)\n");
        }
        NN_SDK_REQUIRES(dateTime[20 - 1] == '\0', "Invalid string (not-terminated)\n");
    }

    GetEmbedContext(m_WorkBuf)->data.dateTime = dateTime;
    m_Stage = Stage_Registered;
}

void ExifBuilder::SetMakerNote(const void *note, const size_t kSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        note == nullptr || (kSize > 0 && kSize <= UINT16_MAX),
        "Valid maker note is required if specified.\n");

    EmbedContext *pContext = GetEmbedContext(m_WorkBuf);
    pContext->data.note = static_cast<const uint8_t*>(note);
    pContext->data.noteSize = (note == nullptr? 0: static_cast<uint16_t>(kSize));
    m_Stage = Stage_Registered;
}
void ExifBuilder::SetUniqueId(const char *uniqueId, const size_t kSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        uniqueId == nullptr || kSize >= 33,
        "Valid unique ID is required if specified.\n");
    NN_UNUSED(kSize);

    // ASCII 32 文字 + '\0' かどうかの検査
    if (uniqueId != nullptr)
    {
        for (int i = 0; i < 33 - 1; i++)
        {
            NN_SDK_ASSERT(std::isxdigit(uniqueId[i]) != 0, "Invalid string (non-hex)\n");
            NN_SDK_ASSERT(uniqueId[i] != '\0', "Invalid string (length != 32)\n");
        }
        NN_SDK_ASSERT(uniqueId[33 - 1] == '\0', "Invalid string (not-terminated)\n");
    }

    GetEmbedContext(m_WorkBuf)->data.uniqueId = uniqueId;
    m_Stage = Stage_Registered;
}

void ExifBuilder::SetThumbnail(const void *thumbnail, const size_t kSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(
        thumbnail == nullptr || kSize > 0,
        "Valid thumbnail is required if specified.\n");
    NN_SDK_REQUIRES(kSize <= UINT32_MAX);

    EmbedContext *pContext = GetEmbedContext(m_WorkBuf);
    pContext->data.thumbnail = reinterpret_cast<const uint8_t*>(thumbnail);
    pContext->data.thumbnailSize = (thumbnail == nullptr? 0: static_cast<uint32_t>(kSize));
    m_Stage = Stage_Registered;
}

JpegStatus ExifBuilder::Analyze() NN_NOEXCEPT
{
    // TIFF ヘッダを BIG エンディアンで初期化する (註: BIG/ LITTLE にこだわりはない)
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Creating TIFF header\n");
    EmbedContext *pContext = GetEmbedContext(m_WorkBuf);
    detail::InitializeTiffHeader(&pContext->tiffHeader, detail::Endian_Big);

    /* ------------------------------------------------------------------------------------
     * 0th TIFF の設定 */

    // 0th TIFF IFD の構築のための情報を収集する
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Checking 0th TIFF tags\n");
    uint16_t tagNum = Setup0thTiffTags(pContext);

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Initializing 0th TIFF IFD header\n");
    bool isSuccess = detail::InitializeIfdHeader(
        &pContext->ifd0.tiff,
        pContext->ifd0.tiffTags,
        tagNum,
        pContext->tiffHeader.offset0thIfd);
    if (!isSuccess)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("0th TIFF IFD exceeds size limitation of APP1\n");
        return JpegStatus_OutputOverabounds;
    }

    // Exif IFD へのオフセットには、TIFF IFD の「次の IFD のオフセット」を利用する。
    // (後述するが、1st TIFF IFD のオフセットには Exif IFD の「次のオフセット」を利用。)
    pContext->data.offsetExifIfd = pContext->ifd0.tiff.nextOffset;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > ExifIfdPointer: 0x%04lx\n", pContext->data.offsetExifIfd);


    // 0th Exif IFD の構築のための情報を収集する
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Checking 0th EXIF tags\n");
    tagNum = Setup0thExifTags(pContext);

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Initializing 0th EXIF IFD header\n");
    isSuccess = detail::InitializeIfdHeader(
        &pContext->ifd0.exif,
        pContext->ifd0.exifTags,
        tagNum,
        static_cast<uint16_t>(pContext->data.offsetExifIfd));
    if (!isSuccess)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("0th EXIF IFD exceeds size limitation of APP1\n");
        return JpegStatus_OutputOverabounds;
    }

    // 1st TIFF IFD へのオフセットに、Exif IFD の「次の IFD のオフセット」を利用する。
    // Exif IFD の次の IFD は存在しないため、0 を設定。
    pContext->ifd0.tiff.nextOffset = pContext->ifd0.exif.nextOffset;
    pContext->ifd0.exif.nextOffset = detail::IfdTagState_NotSpecified;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > IFD0.TIFF.Next: 0x%04lx\n", pContext->ifd0.tiff.nextOffset);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > IFD0.EXIF.Next: 0x%04lx\n", pContext->ifd0.exif.nextOffset);

    /* ------------------------------------------------------------------------------------
     * 1st TIFF の設定 */
    if(pContext->data.thumbnail != nullptr)
    {
        // サムネイルが存在 => 1st TIFF IFD を作成

        NN_SDK_ASSERT(pContext->data.thumbnailSize > 0, "Invalid thumbnail size\n");

        NN_DETAIL_IMAGE_JPEG_LOG_INFO("Checking 1st TIFF tags\n");
        tagNum = Setup1stTiffTags(pContext);

        NN_DETAIL_IMAGE_JPEG_LOG_INFO("Initializing 1st TIFF IFD header\n");
        isSuccess = detail::InitializeIfdHeader(
            &pContext->ifd1.tiff,
            pContext->ifd1.tiffTags,
            tagNum,
            pContext->ifd0.tiff.nextOffset);
        if (!isSuccess)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("1st TIFF IFD exceeds size limitation of APP1\n");
            return JpegStatus_OutputOverabounds;
        }

        // サムネイルの終端が Exif のサイズ上限を超えないか検査する
        uint32_t thumbnailEnd = pContext->ifd1.tiff.nextOffset + pContext->data.thumbnailSize;
        if (!(thumbnailEnd <= detail::ExifProperty_SizeMax))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Thumbnail image exceeds size limitation of APP1\n");
            return JpegStatus_OutputOverabounds;
        }
        pContext->data.thumbnailOffset = pContext->ifd1.tiff.nextOffset;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > JPEGFormat:     0x%04x\n", pContext->data.thumbnailOffset);

        // 1st TIFF IFD の次の IFD は存在しないため、0 を設定。
        pContext->ifd1.tiff.nextOffset = detail::IfdTagState_NotSpecified;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > IFD1.TIFF.Next: 0x%04lx\n", pContext->ifd1.tiff.nextOffset);

        m_OutputSize = static_cast<uint16_t>(thumbnailEnd);
    }
    else
    {
        // サムネイルが存在しない => 1st TIFF IFD を作成しない

        NN_SDK_ASSERT(
            pContext->ifd0.tiff.nextOffset <= detail::ExifProperty_SizeMax,
            "Output Exif binary overbounds the Exif size limitation\n");
        m_OutputSize = static_cast<uint16_t>(pContext->ifd0.tiff.nextOffset);

        NN_DETAIL_IMAGE_JPEG_LOG_INFO("Skipping 1st TIFF IFD header\n");
        pContext->ifd0.tiff.nextOffset = detail::IfdTagState_NotSpecified;
        pContext->ifd1.tiff.offset = detail::IfdTagState_NotSpecified;
        pContext->ifd1.tiff.tagCount = 0;
        pContext->ifd1.tiff.nextOffset = detail::IfdTagState_NotSpecified;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > IFD0.TIFF.Next: 0x%04lx\n", pContext->ifd0.tiff.nextOffset);
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > IFD1.TIFF.Next: 0x%04lx\n", pContext->ifd1.tiff.nextOffset);
    }
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Estimated resulting EXIF binary size: %u\n", m_OutputSize);

    m_Stage = Stage_Analyzed;
    return JpegStatus_Ok;
}

size_t ExifBuilder::GetAnalyzedOutputSize() const NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_Stage >= Stage_Analyzed, "Embeded data is not analyzed.\n");
    return m_OutputSize;
}

void ExifBuilder::Build(
    void *exifBuf,
    const size_t kExifBufSize,
    const ::nn::image::Dimension &kDimension) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Stage >= Stage_Analyzed, "Embedded data is not analyzed.\n");

    NN_SDK_REQUIRES(
        exifBuf != nullptr && kExifBufSize >= m_OutputSize,
        "Valid output buffer is required.\n");
    NN_SDK_REQUIRES(
        kDimension.width > 0 && kDimension.width <= 0xFFFF &&
        kDimension.height > 0 && kDimension.height <= 0xFFFF,
        "Valid pixel size is required.\n");

    EmbedContext *pContext = GetEmbedContext(m_WorkBuf);

    // 与えられた実行画像幅,高さを設定する
    // (ここでワークメモリが汚染されるため、const 関数にできない。)
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > PixelXDim:      %u\n", kDimension.width);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" > PixelYDim:      %u\n", kDimension.height);
    // ポインタは先行して Setup0thExifTags() で登録済み
    pContext->data.pixelX = static_cast<uint16_t>(kDimension.width);
    pContext->data.pixelY = static_cast<uint16_t>(kDimension.height);

    // 出力
    uint8_t *data = reinterpret_cast<uint8_t*>(exifBuf);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Building TIFF header\n");
    const uint16_t kBufSizeU16 = (kExifBufSize > UINT16_MAX? UINT16_MAX: static_cast<uint16_t>(kExifBufSize));
    detail::WriteTiffHeader(data, kBufSizeU16, pContext->tiffHeader);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Building 0th TIFF IFD (@0x%04x)\n", pContext->ifd0.tiff.offset);
    detail::WriteIfd(
        data,
        kBufSizeU16,
        pContext->ifd0.tiffTags,
        pContext->ifd0.tiff,
        pContext->tiffHeader.io);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Building 0th EXIF IFD (@0x%04x)\n", pContext->ifd0.exif.offset);
    detail::WriteIfd(
        data,
        kBufSizeU16,
        pContext->ifd0.exifTags,
        pContext->ifd0.exif,
        pContext->tiffHeader.io);
    if (pContext->ifd1.tiff.offset != detail::IfdTagState_NotSpecified)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_INFO("Building 1st TIFF IFD (@0x%04x)\n", pContext->ifd1.tiff.offset);
        detail::WriteIfd(
            data,
            kBufSizeU16,
            pContext->ifd1.tiffTags,
            pContext->ifd1.tiff,
            pContext->tiffHeader.io);

        if (pContext->data.thumbnailOffset != detail::IfdTagState_NotSpecified)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_INFO("Building Thumbnail (@0x%04x)\n", pContext->data.thumbnailOffset);
            std::memcpy(
                data + pContext->data.thumbnailOffset,
                pContext->data.thumbnail,
                pContext->data.thumbnailSize);
        }
    }
}

}}
