﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/image/image_JpegCommon.h>

#define JPEG_INTERNALS          // NOLINT(name/macro)
#include "include/jpeglib.h"
#undef JPEG_INTERNALS

#include "image_JpegConfig.h"

/**
    @file JPEG ヘッダの解析/構築のための諸々を定義するファイル
 */

namespace nn { namespace image { namespace detail {

/**
    @brief JPEG ヘッダのパーサとその利用者のための各種定数
 */
enum JpegProperty : uint32_t
{
    JpegProperty_YccComponentNum = 3u,          //!< YCbCr 色空間のバイト数 (JPEG仕様)
    JpegProperty_AcceptableComponentNum = 4u,   //!< 入力画像の色空間(RGBA)のバイト数 (SDK仕様)

    JpegProperty_HuffTableBitNum = 16u,         //!< ハフマンテーブルビット配分数のスロット数 (JPEG仕様)
    JpegProperty_MarkerLength = 2u,             //!< JPEG マーカーの長さ (JPEG仕様)
    JpegProperty_SegmentSizeLength = 2u,        //!< セグメント長のフィールドの長さ (JPEG仕様)
    JpegProperty_AdobeMarkerLength = 7u,        //!< Adobe APP14 マーカー長 (Adobe仕様)
    JpegProperty_AdobeTransformOffset = 6u,     //!< Adove APP14 マーカー中の Transform へのオフセット (Adobe仕様)
    JpegProperty_ExifAppHeaderSize = 10         //!< Exif APP1 マーカーのヘッダ長 (Exif仕様)
};

/**
    @brief JPEG マーカー定義
 */
enum JpegMarker: uint16_t
{
    JpegMarker_Sof = 0xFF,      //!< JPEG マーカー全般を検出するのにも使用する。
    JpegMarker_Sof0 = 0xFFC0,   //!< SOFx は JPEG の圧縮タイプを判定するのに使用。4, 8, 12, 16 は存在しない。
    JpegMarker_Sof1 = 0xFFC1,
    JpegMarker_Sof2 = 0xFFC2,
    JpegMarker_Sof3 = 0xFFC3,
    JpegMarker_Sof5 = 0xFFC5,
    JpegMarker_Sof6 = 0xFFC6,
    JpegMarker_Sof7 = 0xFFC7,
    JpegMarker_Sof9 = 0xFFC9,
    JpegMarker_Sof10 = 0xFFCA,
    JpegMarker_Sof11 = 0xFFCB,
    JpegMarker_Sof13 = 0xFFCD,
    JpegMarker_Sof14 = 0xFFCE,
    JpegMarker_Sof15 = 0xFFCF,

    JpegMarker_Dht = 0xFFC4,

    JpegMarker_Soi = 0xFFD8,
    JpegMarker_Eoi = 0xFFD9,
    JpegMarker_Sos = 0xFFDA,
    JpegMarker_Dqt = 0xFFDB,

    JpegMarker_App0 = 0xFFE0,
    JpegMarker_App1 = 0xFFE1,
    JpegMarker_App14 = 0xFFEE,
    JpegMarker_AppMask = 0xFFF0 //!< marker & AppMask == APP0 で判定する。
};

/**
    @brief 一つの色コンポーネントを表す構造体
 */
typedef struct
{
    jpeg::JOCTET cn;    // component ID
    jpeg::JOCTET hn;    // horizontal sampling value
    jpeg::JOCTET vn;    // vertical sampling value
    jpeg::JOCTET tqn;   // Correspondence quantization table number
    uint32_t dctHScaledSize;
    uint32_t dctVScaledSize;
    uint32_t widthInBlocks;
    uint32_t heightInBlocks;
} JpegComponentInfo;

/**
    @brief SOF セグメントの情報
 */
typedef struct
{
    uint16_t sof;       // merker code
    uint16_t lf;        // segment length
    jpeg::JOCTET p;     // number of components
    uint16_t y;         // vertical size of image
    uint16_t x;         // horizontal size of image
    jpeg::JOCTET nf;    // number of components
    uint16_t maxHn;     // Max horizontal sampling value
    uint16_t maxVn;     // Max vertical sampling value
    JpegComponentInfo componentInfo[JpegProperty_AcceptableComponentNum]; // components info
} JpegSofInfo;

/**
    @brief 単一のハフマンテーブルを表す情報
 */
typedef struct
{
    jpeg::JOCTET csn;       // component ID
    jpeg::JOCTET tdnTan;    // DC/AC component huffman table No.
} JpegHuffmanTableInfo;

/**
    @brief SOS セグメントの情報
 */
typedef struct
{
    uint16_t sos;       // Start Of Scan
    uint16_t ls;        // segment length
    jpeg::JOCTET ns;    // number of components
    JpegHuffmanTableInfo huffman[JpegProperty_AcceptableComponentNum];  // Huffman table info
    jpeg::JOCTET ss;    // quantized coefficient start No.
    jpeg::JOCTET se;    // quantized coefficient end No.
    jpeg::JOCTET ah;    // split coefficient shift(last scan)
    jpeg::JOCTET al;    // split coefficient shift
} JpegSosInfo;

/**
    @brief JPEG データの情報
 */
typedef struct
{
    JpegSofInfo sof;
    uint32_t    blockSize;
    uint32_t    sosCount;
    uint32_t    dhtCount;
    uint32_t    dhtMask;
    uint32_t    dqtCount;
    uint32_t    dqtMask;
    uint32_t    minDctHScaledSize;
    uint32_t    minDctVScaledSize;
    uint32_t    outputWidth;
    uint32_t    outputHeight;
    uint32_t    limSe;
    uint32_t    derivedTableNum;
    uint32_t    outColorComponents; // # of color components in outColorSpace
    bool        isEnabledMergeUpSample;
    jpeg::J_COLOR_SPACE outColorSpace;
    jpeg::J_COLOR_SPACE jpegColorSpace; // colorspace of JPEG image
    bool        hasJfifMarker;      // TRUE iff a JFIF APP0 marker was found
    bool        hasAdobeMarker;     // TRUE iff an Adobe APP14 marker was found
    bool        isEnabledFancyUpSample;  // TRUE=apply fancy upsampling
    bool        hasMultipleScans;   // True if file has multiple scans
    jpeg::JOCTET adobeTransform;    // Color transform code from Adobe marker
    jpeg::JOCTET firstNs;           // Ns in 1st SOS Info
} JpegInfo;
typedef JpegInfo *JpegInfoPtr;

//! @name JPEG データ解析のための関数
//! @{

/**
    @brief      JPEG ヘッダが SOI から始まるかを検査する。
 */
::nn::image::JpegStatus CheckSoi(
    const uint8_t *jpeg,
    const size_t jpegSize) NN_NOEXCEPT;

/**
    @brief      JPEG ヘッダを解析し、画像の情報を取得する。
 */
::nn::image::JpegStatus ReadJpegHeader(
    JpegInfoPtr pinfo,
    const jpeg::JOCTET *jpeg,
    const size_t jpegSize,
    const uint32_t resolutionDenom) NN_NOEXCEPT;

/**
    @brief      JPEG ヘッダを解析し、Exif 情報バイナリの位置とサイズを取得する。
 */
bool GetExifInfo(
    const void **pData,
    uint16_t *pSize,
    const jpeg::JOCTET *jpeg,
    const size_t kJpegSize) NN_NOEXCEPT;

//! @brief 与えられた JPEG データの要約がプログレッシブ JPEG によるものかを判定する。
NN_FORCEINLINE bool IsProgressive(const JpegInfoPtr pinfo) NN_NOEXCEPT
{
    return (pinfo)->sof.sof != JpegMarker_Sof0;
}
//! @}

//! @name JPEG データ作成のための関数
//! @{
/**
    @brief JPEG ヘッダを初期化する。
 */
void SetJpegHeader(
    JpegInfoPtr pinfo,
    const uint16_t width,
    const uint16_t height,
    const ::nn::image::JpegSamplingRatio kSample) NN_NOEXCEPT;

/**
    @brief 指定したアドレスに Exif 情報の書き込みのための APP1 ヘッダを記録する。
 */
void WriteExifHeader(void *outBuf, const size_t kOutputSize) NN_NOEXCEPT;
//! @}

}}}
