﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <climits>
#include <cstring>
#include <algorithm>

#include <nn/nn_SdkAssert.h>

#include <nn/image/image_JpegCommon.h>

#include "image_JpegConfig.h"
#include "image_JpegParser.h"

#define JPEG_INTERNALS          // NOLINT(name/macro)
#include "include/jconfig.h"
#undef JPEG_INTERNALS

namespace nn { namespace image { namespace detail {

namespace
{
// 汎用インライン関数
NN_FORCEINLINE uint8_t Read1Byte(const void *ptr) NN_NOEXCEPT
{
    return GETJOCTET(*reinterpret_cast<const jpeg::JOCTET*>(ptr));
}
NN_FORCEINLINE uint16_t Read2Bytes(const void *ptr) NN_NOEXCEPT
{
    return ((static_cast<uint16_t>(Read1Byte(ptr)) << 8)
        + GETJOCTET(*(reinterpret_cast<const jpeg::JOCTET*>(ptr) + 1)));
}
NN_FORCEINLINE void Write1Byte(void *ptr, const uint8_t value) NN_NOEXCEPT
{
    (*reinterpret_cast<jpeg::JOCTET*>(ptr) = static_cast<jpeg::JOCTET>(value & 0xFF));
}
NN_FORCEINLINE void Write2Bytes(void *ptr, const uint16_t value) NN_NOEXCEPT
{
    Write1Byte(ptr, static_cast<uint8_t>((value & 0xFF00) >> 8));
    *(reinterpret_cast<jpeg::JOCTET*>(ptr) + 1) = static_cast<jpeg::JOCTET>(value & 0xFF);
}

// ログ関係マクロ
#if NN_DETAIL_IMAGE_JPEG_CONFIG_DEBUG_PRINT >= 1
#define NN_DETAIL_IMAGE_JPEG_PRINT_IF_APPX(marker, pos, size, nextAddr) \
    do \
    { \
        if (((marker) & JpegMarker_AppMask) == JpegMarker_App0) \
        { \
            NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - APP%d: Pos: %08x, size: %04x, NextSOF: %08x\n", ((marker) & 0xF), (pos), (size), (nextAddr)); \
        } \
    } while (NN_STATIC_CONDITION(false))

void PrintSof(const JpegSofInfo &sof) NN_NOEXCEPT
{
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("SOF: 0x%02X\n", sof.sof);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Lf: 0x%02X\n", sof.lf);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - P : 0x%02X\n", sof.p);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Y : 0x%02X\n", sof.y);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - X : 0x%02X\n", sof.x);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Nf: 0x%02X\n", sof.nf);
}
void PrintSofComponent(const JpegComponentInfo &comp) NN_NOEXCEPT
{
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("   - Component: %d\n", comp.cn);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("     - Hn : 0x%01X\n", comp.hn);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("     - Vn : 0x%01X\n", comp.vn);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("     - Tqn: 0x%01X\n", comp.tqn);
}
void PrintSos(const JpegSosInfo &sos) NN_NOEXCEPT
{
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("SOS: 0x%02X\n", sos.sos);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Ls: 0x%02X\n", sos.ls);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Ns: 0x%01X\n", sos.ns);
    for (uint8_t i = 0; i < sos.ns; i++)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("   - Component: %d\n", sos.huffman[i].csn);
        NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("     - Tdn:0x%01X\n", (sos.huffman[i].tdnTan >> 4) & 0xF);
        NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("     - Tan:0x%01X\n", (sos.huffman[i].tdnTan     ) & 0xF);
    }
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Ss: 0x%01X\n", sos.ss);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Se: 0x%01X\n", sos.se);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Ah: 0x%01X\n", sos.ah);
    NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - Al: 0x%01X\n", sos.al);
}
#else
#define NN_DETAIL_IMAGE_JPEG_PRINT_IF_APPX(...)
inline void PrintSof(const JpegSofInfo &sof) NN_NOEXCEPT
{
    NN_UNUSED(sof);
}
inline void PrintSofComponent(const JpegComponentInfo &comp) NN_NOEXCEPT
{
    NN_UNUSED(comp);
}
inline void PrintSos(const JpegSosInfo &sos) NN_NOEXCEPT
{
    NN_UNUSED(sos);
}
#endif

// JPEG フォーマット解析用
#define NN_DETAIL_IMAGE_GET_AC_TABLE_NO(val)    ((val) & 0x0F)
#define NN_DETAIL_IMAGE_GET_DC_TABLE_NO(val)    (((val) >> 4) & 0x0F)

// libjpeg 仕様対応
#define NN_DETAIL_IMAGE_GET_SCALED_SIZE(a_size, a_blockSize, a_scaleParam) \
    jpeg::jdiv_round_up((a_size) * (a_scaleParam), (a_blockSize))

// JFIF APP0 セグメントのシグニチャ
const uint8_t AppJfifIdentity[] = {'J', 'F', 'I', 'F', '\0'};
// Adobe APP14 セグメントのシグニチャ
const uint8_t AppAdobeIdentity[] = {'A', 'd', 'o', 'b', 'e'};
// Exif APP1 セグメントのシグニチャ
const uint8_t AppExifIdentity[] = {'E', 'x', 'i', 'f', '\0', '\0'};

/**
    @brief マーカー種別
 */
typedef enum MarkerType
{
    MarkerType_MarkOnly,    //!< マーカーのみのマーカー。(e.g. SOI)
    MarkerType_Segment,     //!< セグメント系マーカー。(e.g. SOFx, DHT, APPx, ...)
    MarkerType_Skip1Byte,   //!< 0xFFFF は 0xFF に圧縮する仕様。
    MarkerType_Reserved,    //!< 予約済みのセグメント系マーカー。(無視するために分けている。)
    MarkerType_Unsupported, //!< nn::image::jpeg では非対応のマーカー。あればエラー。

    MarkerType_Undefined    //!< 2バイトスキップ
} MarkerType;

/**
    @brief マーカーの値域と対応するマーカー種別を定義する構造体
 */
typedef struct MarkerDef
{
    uint16_t start;     //!< 値域の先頭
    uint16_t end;       //!< 値域の末尾(これも含む)
    MarkerType type;    //!< マーカー種別
} MarkerDef;

/**
    @brief マーカー定義
 */
const MarkerDef DefinedMarker[] =
{
    // 0xFF00           -> value 0xFF
    {0xFF01, 0xFF01, MarkerType_Segment},       // Temporary segment for arithmatic coding
    {0xFF02, 0xFF4E, MarkerType_Reserved},      // 予約済み
    {0xFF4F, 0xFFBF, MarkerType_Unsupported},   // 未定義 or JPEG2000 => エラー
    // {0xFF4F, 0xFF53, MarkerType_Unsupported},// JPEG2000
    // 0xFF54           -> NONE
    // {0xFF55, 0xFF55, MarkerType_Unsupported},// JPEG2000
    // 0xFF56           -> NONE
    // {0xFF57, 0xFF58, MarkerType_Unsupported},// JPEG2000
    // 0xFF59 - 0xFF5B  -> NONE
    // {0xFF5C, 0xFF61, MarkerType_Unsupported},// JPEG2000
    // 0xFF62           -> NONE
    // {0xFF63, 0xFF64, MarkerType_Unsupported},// JPEG2000
    // 0xFF65 - 0xFF8F  -> NONE
    // {0xFF90, 0xFF93, MarkerType_Unsupported},// JPEG2000
    // 0xFF94 - 0xFFBF  -> NONE
    {0xFFC0, 0xFFC7, MarkerType_Segment},       // SOF + DHT
    {0xFFC8, 0xFFC8, MarkerType_Reserved},      // 予約済み
    {0xFFC9, 0xFFCF, MarkerType_Segment},       // SOF + Arithmatic code table
    {0xFFD0, 0xFFD9, MarkerType_MarkOnly},      // Restart marker + SOI + EOI
    {0xFFDA, 0xFFEF, MarkerType_Segment},       // SOS + DQT + DNL + DRI + DHP + EXP + APPx
    {0xFFF0, 0xFFFD, MarkerType_Reserved},      // 予約済み
    {0xFFFE, 0xFFFE, MarkerType_Segment},       // COMマーカー
    {0xFFFF, 0xFFFF, MarkerType_Skip1Byte},     // 0xFFFF -> 0xFF に変換
};

/**
    @brief 与えられたマーカーコードからマーカー種別を取得する。
 */
MarkerType GetMarkerType(const uint16_t marker) NN_NOEXCEPT
{
    for (size_t s = 0; s < sizeof(DefinedMarker) / sizeof(MarkerDef) && marker >= DefinedMarker[s].start; s++)
    {
        if (marker <= DefinedMarker[s].end)
        {
            return DefinedMarker[s].type;
        }
    }
    return MarkerType_Undefined;
}

/**
    @brief 指定されたマーカーコードを、JPEG データから検索する。

    @return あれば true 、なければ false 。
 */
bool FindMarkerCoder(
    size_t* pOutOffsetToMarker,
    const uint8_t *jpeg,
    const size_t jpegSize,
    const size_t start,
    const uint16_t markerCode) NN_NOEXCEPT
{
    const uint8_t *pos = nullptr;
    size_t offset = 0x00;
    uint16_t marker = 0x00;
    size_t length = 0;

    *pOutOffsetToMarker = 0;
    while (offset < jpegSize && start + offset < jpegSize)
    {
        size_t skipSize = 0;

        length = jpegSize - (start + offset);

        // memchr で length の範囲から探す。
        pos = reinterpret_cast<const uint8_t*>(std::memchr(jpeg + start + offset, JpegMarker_Sof, length));
        if (!(pos != nullptr && static_cast<size_t>(pos - jpeg) + JpegProperty_MarkerLength <= jpegSize))
        {
            // 発見できず or データ末尾 => エラー
            break;
        }

        marker = Read2Bytes(pos);
        if (marker == 0xFF00)
        {
            // 0xFF00 はただの 0xFF であるのでスキップ
            offset = (pos + JpegProperty_MarkerLength) - (jpeg + start);
            continue;
        }

        // マーカー種別と、読み飛ばし時のスキップ長を取得
        bool isValid = false;
        MarkerType markerType = GetMarkerType(marker);
        NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("Marker: %04x (type=%d) @%08x\n", marker, markerType, (pos - jpeg));
        switch (markerType)
        {
        case MarkerType_Segment:
            NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - segment\n");
            isValid = false;
            if (static_cast<size_t>(pos - jpeg) + JpegProperty_MarkerLength + JpegProperty_SegmentSizeLength <= jpegSize)
            {
                // セグメント長の取得
                uint16_t segSize = Read2Bytes(pos + JpegProperty_MarkerLength);
                NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("   - size: 0x%x\n", segSize);
                skipSize = segSize + JpegProperty_MarkerLength;

                if (skipSize >= JpegProperty_MarkerLength + JpegProperty_SegmentSizeLength &&
                    static_cast<size_t>(pos - jpeg) + skipSize <= jpegSize)
                {
                    NN_DETAIL_IMAGE_JPEG_PRINT_IF_APPX(marker, (pos - jpeg), skipSize, (pos - jpeg) + skipSize);
                    isValid = true;
                }
                // else: セグメントが短すぎ or あふれる => エラー
            }
            // else: セグメントサイズを取得できない => エラー
            if (!isValid)
            {
                NN_DETAIL_IMAGE_JPEG_LOG_ERROR("   -> Invalid segment marker\n");
                return false;
            }
            break;

        case MarkerType_Unsupported:
            // 非サポートのマーカーを検出した => エラー
            NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - unsupported\n");
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("   -> Unsupported JPEG data\n");
            return false;

        case MarkerType_Skip1Byte:
        case MarkerType_MarkOnly:
        case MarkerType_Reserved:
        default:
            NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(" - common\n");
            // 再検索の場合は 1 or 2 バイトスキップ
            skipSize = (markerType == MarkerType_Skip1Byte ? 1: JpegProperty_MarkerLength);
            break;
        }

        if (marker == markerCode)
        {
            // 目的のマーカーを発見したので探索を終了 (OK)
            NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("   -> Target found\n");
            *pOutOffsetToMarker = (pos - jpeg);
            return true;
        }
        else if (marker == JpegMarker_Eoi)
        {
            // 予期せず EOI にぶつかったので、探索を終了 (NG)
            NN_DETAIL_IMAGE_JPEG_LOG_DETAIL("[Reached end of image (EOI)]\n");
            return false;
        }

        // 取得したスキップ長ぶんスキップして再検索
        offset = (pos + skipSize) - (jpeg + start);
    }

    // 探索終了
    return false;
}

//! @brief 指定された SOFx を検索
bool GetSofInfo(
    JpegSofInfo* pOutSof,
    const uint8_t *jpeg,
    const size_t jpegSize,
    const JpegInfoPtr pinfo,
    const uint16_t markerCode) NN_NOEXCEPT
{
    size_t offset = 0;
    if (!FindMarkerCoder(&offset, jpeg, jpegSize, 0, markerCode))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("SOF not found\n");
        return false;
    }

    // SOFx を読んでも溢れないかの検査
    if (!(offset + 2 * 2 <= jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("SOF segment overflows given Jpeg data\n");
        return false;
    }

    // 各メンバを取得する
    pOutSof->sof = Read2Bytes(jpeg + offset);
    offset += 2;

    pOutSof->lf = Read2Bytes(jpeg + offset);
    // Sof セグメント長 (P + Nf) + (Lf + Y + X) の正当性検査
    if (!(pOutSof->lf >= 2 * 1 + 3 * 2 && offset + pOutSof->lf <= jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOF segment length\n");
        return false;
    }
    offset += 2;

    pOutSof->p = Read1Byte(jpeg + offset);
    offset += 1;
    pOutSof->y = Read2Bytes(jpeg + offset);
    offset += 2;
    pOutSof->x = Read2Bytes(jpeg + offset);
    offset += 2;
    pOutSof->nf = Read1Byte(jpeg + offset);
    offset += 1;

    // デバッグ用に SOFx をダンプ
    PrintSof(*pOutSof);

    // SOFx の内容の正当性を検査
    // 1) 色数は 1 ～ 4
    // 2) Lf+P+X+Y+Nf+Nf*(Cn+HnVn+Tqn) = 2+1+2+2+1+3Nf
    if (!(pOutSof->nf >= 1 && pOutSof->nf <= JpegProperty_AcceptableComponentNum &&
        pOutSof->lf == (pOutSof->nf * 3 + 2) * 1 + 3 * 2))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOF segment\n");
        return false;
    }

    // Hn, Vn の最大値の調査
    pOutSof->maxHn = 0;
    pOutSof->maxVn = 0;
    for (uint8_t i = 0; i < pOutSof->nf; i++)
    {
        // Cn の取得
        pOutSof->componentInfo[i].cn = Read1Byte(jpeg + offset);
        offset += 1;

        // Hn, Vn の取得
        uint8_t samplingValue = Read1Byte(jpeg + offset);
        offset += 1;
        pOutSof->componentInfo[i].hn = (samplingValue >> 4) & 0xF;
        pOutSof->componentInfo[i].vn = (samplingValue     ) & 0xF;
        if (!(pOutSof->componentInfo[i].hn != 0 && pOutSof->componentInfo[i].vn != 0))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOF segment (2)\n");
            return false;
        }

        // 最大値の更新
        pOutSof->maxHn = std::max(pOutSof->maxHn, static_cast<uint16_t>(pOutSof->componentInfo[i].hn));
        pOutSof->maxVn = std::max(pOutSof->maxVn, static_cast<uint16_t>(pOutSof->componentInfo[i].vn));

        // 参照する DQT の存在チェック
        pOutSof->componentInfo[i].tqn = Read1Byte(jpeg + offset);
        offset += 1;
        if ((pinfo->dqtMask & (1 << pOutSof->componentInfo[i].tqn)) == 0)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Undefined DQT ID\n");
            return false;
        }

        // デバッグのために SOFx の component info をダンプ
        PrintSofComponent(pOutSof->componentInfo[i]);
    }
    if (!(pOutSof->maxHn > 0 && pOutSof->maxVn > 0))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOF segment (3)\n");
        return false;
    }

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Maximum Hn,Vn = %02X,%02x\n", pOutSof->maxHn, pOutSof->maxVn);
    return true;
}

//! @brief SOS の情報を取得
bool GetSosInfo(
    JpegSosInfo *pOutSos,
    size_t *pOutEndOffset,
    const uint8_t *jpeg,
    const size_t jpegSize,
    const size_t start,
    const JpegInfoPtr pinfo) NN_NOEXCEPT
{
    size_t offset = 0;
    if (!FindMarkerCoder(&offset, jpeg, jpegSize, start, JpegMarker_Sos))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("SOS not found\n");
        return false;
    }
    // SOS を読んでも溢れないかの検査
    if (!(offset + 2 * 2 <= jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of SOS segment\n");
        return false;
    }

    // 各メンバを取得する
    pOutSos->sos = Read2Bytes(jpeg + offset);
    offset += 2;

    pOutSos->ls = Read2Bytes(jpeg + offset);
    // SOS のセグメント長の検査 (Ls + Ns + ... + Ss + Se + AhAl = 2+1+...+1+1+1)
    if (!(pOutSos->ls >= 4 * 1 + 2 && offset + pOutSos->ls <= jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of SOS segment (2)");
        return false;
    }
    offset += 2;

    pOutSos->ns = Read1Byte(jpeg + offset);
    offset += 1;

    // SOS の正当性検査
    // 1) 色の数は4まで。
    // 2) Ls+Ns+Ns*(Csn+TdnTan)+Ss+Se+AhAl = 2+1+2Ns+1+1+1
    if (!(pOutSos->ns > 0 && pOutSos->ns <= JpegProperty_AcceptableComponentNum &&
        pOutSos->ls == (pOutSos->ns * 2 + 4) * 1 + 2))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOS\n");
        return false;
    }

    // 参照するハフマンテーブルの存在を検査
    for (uint8_t i = 0; i < pOutSos->ns; i++)
    {
        pOutSos->huffman[i].csn = Read1Byte(jpeg + offset);
        offset += 1;

        bool isValid = false;
        for (uint8_t j = 0; j < pinfo->sof.nf && !isValid; j++)
        {
            isValid = (pinfo->sof.componentInfo[j].cn == pOutSos->huffman[i].csn);
        }
        if (!isValid)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOS (2): Undefined color component\n");
            return false;
        }

        pOutSos->huffman[i].tdnTan = Read1Byte(jpeg + offset);
        if (!(NN_DETAIL_IMAGE_GET_DC_TABLE_NO(pOutSos->huffman[i].tdnTan) < NUM_HUFF_TBLS &&
            NN_DETAIL_IMAGE_GET_AC_TABLE_NO(pOutSos->huffman[i].tdnTan) < NUM_HUFF_TBLS))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOS (3): Invalid reference to Huffmantable\n");
            return false;
        }
        offset += 1;
    }

    pOutSos->ss = Read1Byte(jpeg + offset);
    offset += 1;
    pOutSos->se = Read1Byte(jpeg + offset);
    offset += 1;
    uint8_t ahAl = Read1Byte(jpeg + offset);
    offset += 1;
    pOutSos->ah = (ahAl >> 4) & 0xF;
    pOutSos->al = (ahAl     ) & 0xF;
    // Ss, Se, Ah, Al の検査
    // 1) Ss, Se の値域: [0, 63]
    // 2) Ah, Al の値域: [0x0, 0xD]
    if (!(pOutSos->ss <= 63 && pOutSos->se <= 63 && pOutSos->ah <= 0xD && pOutSos->al <= 0xD))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong SOS segment");
        return false;
    }

    *pOutEndOffset = offset;

    // デバッグ用に SOS をダンプ
    PrintSos(*pOutSos);
    return true;
}

//! @brief ハフマンテーブルの数を取得
bool GetHuffTableNum(JpegInfoPtr pinfo, const uint8_t *jpeg, const size_t jpegSize) NN_NOEXCEPT
{
    uint16_t length = 0;
    size_t lengthRemain = 0;
    size_t start = 0;
    size_t offset = 0;
    uint8_t TcnThn = 0;
    uint8_t Ln = 0;
    uint32_t tableNum = 0;
    uint32_t tableFlag = 0;

    while (FindMarkerCoder(&offset, jpeg, jpegSize, start, JpegMarker_Dht))
    {
        // データ溢れの検査
        if (!(offset + JpegProperty_MarkerLength + JpegProperty_SegmentSizeLength <= jpegSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of DHT segment\n");
            return false;
        }
        offset += JpegProperty_MarkerLength;

        // セグメント長の取得と検査
        length = Read2Bytes(jpeg + offset);
        if (!(length >= JpegProperty_SegmentSizeLength && offset + length <= jpegSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of DHT segment (2)\n");
            return false;
        }
        offset += JpegProperty_SegmentSizeLength;
        lengthRemain = length - JpegProperty_SegmentSizeLength;

        while (lengthRemain > 0) // for one DHT Marker Code
        {
            size_t tableSize = 0;
            uint8_t Tcn = 0;
            uint8_t Thn = 0;

            // DHTx のサイズチェック
            if (!(offset + 1 + JpegProperty_HuffTableBitNum <= jpegSize))
            {
                NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid DHT segment\n");
                return false;
            }

            // DHTx の Tcn, Thn の取得、検査
            TcnThn = Read1Byte(jpeg + offset);
            Tcn = (TcnThn >> 4) & 0x0F;
            Thn = TcnThn & 0x0F;
            if (!((Tcn & ~0x01) == 0x00 && Thn < NUM_HUFF_TBLS))
            {
                NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid DHT segment (2)\n");
                return false;
            }

            // テーブルの存在を記録 (↑の検査より、TcnThn の値は 0x13 まで)
            if ((tableFlag & (0x01 << TcnThn)) == 0x00)
            {
                tableNum ++;
                tableFlag = tableFlag | (0x01 << TcnThn);
            }

            // DHTx のサイズは ヘッダ長 + Ln の和
            tableSize = 1 + JpegProperty_HuffTableBitNum;
            for (uint32_t i = 0; i < JpegProperty_HuffTableBitNum; i ++)
            {
                Ln = Read1Byte(jpeg + offset + 1 + i * 1);
                tableSize += Ln;
            }

            offset += tableSize;
            lengthRemain -= tableSize;
        }

        if (!(lengthRemain == 0))
        {
            // 不正な DHT サイズ
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid DHT\n");
            return false;
        }
        start = offset;
    }
    if (!(tableNum > 0))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong DHT segment\n");
        return false;
    }
    pinfo->dhtCount = tableNum;
    pinfo->dhtMask = tableFlag;

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Huffman table num: %d\n", pinfo->dhtCount);
    return true;
}

// @brief ブロックサイズの取得
bool GetBlockSize(JpegInfoPtr pinfo, const uint8_t *jpeg, const size_t jpegSize) NN_NOEXCEPT
{
    // 最初の SOS を解析
    JpegSosInfo sos;
    size_t endOfSos = 0;
    if (!GetSosInfo(&sos, &endOfSos, jpeg, jpegSize, 0, pinfo))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("SOS not found\n");
        return false;
    }
    pinfo->firstNs = sos.ns;
    // SOS の Ns が SOF の Nf より小さければ、スキャンは複数あるはず。
    pinfo->hasMultipleScans = (sos.ns < pinfo->sof.nf);

    uint32_t blockSize = 0;
    if (sos.ns > 0)
    {
        blockSize = DCTSIZE;
        pinfo->limSe = DCTSIZE2 - 1;
    }
    else
    {
        for (int i = 1; i < 16; i ++)
        {
            if (sos.se == (i * i - 1))
            {
                blockSize = i;
                pinfo->limSe = (i < 8)? sos.se: DCTSIZE2 - 1;
                break;
            }
        }
    }
    if (!(blockSize > 0))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong SOS segment\n");
        return false;
    }
    pinfo->blockSize = blockSize;

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Block size: %u\n", pinfo->blockSize);
    return true;
}

//! @brief libjpeg で導出されるハフマンテーブルの数を取得
bool GetDerivedTableNum(JpegInfoPtr pinfo, const uint8_t *jpeg, const size_t jpegSize) NN_NOEXCEPT
{
    uint32_t tableNum = 0;

    size_t start = 0;
    size_t next = 0;
    JpegSosInfo sos = {};
    uint8_t tableFlag = 0;
    uint8_t tablePos = 0;
    if (IsProgressive(pinfo))
    {
        while (GetSosInfo(&sos, &next, jpeg, jpegSize, start, pinfo))
        {
            for (uint8_t i = 0; i < sos.ns; i++)
            {
                if (sos.ss == 0)
                {
                    // DC Table
                    if (sos.ah == 0)
                    {
                        tablePos = NN_DETAIL_IMAGE_GET_DC_TABLE_NO(sos.huffman[i].tdnTan);
                        if ((tableFlag & (0x01 << tablePos)) == 0x00)
                        {
                            tableNum ++;
                            tableFlag |= 0x01 << tablePos;
                        }
                    }
                }
                else
                {
                    // AC Table
                    tablePos = NN_DETAIL_IMAGE_GET_AC_TABLE_NO(sos.huffman[i].tdnTan) + NUM_HUFF_TBLS;
                    if ((tableFlag & (0x01 << tablePos)) == 0x00)
                    {
                        tableNum ++;
                        tableFlag |= 0x01 << tablePos;
                    }
                }
            }
            std::memset(&sos, 0, sizeof(JpegSosInfo));
            start = next;
        }
    }
    else
    {
        while (GetSosInfo(&sos, &next, jpeg, jpegSize, start, pinfo))
        {
            for (uint8_t i = 0; i < sos.ns; i++)
            {
                // DC Table
                tablePos = NN_DETAIL_IMAGE_GET_DC_TABLE_NO(sos.huffman[i].tdnTan);
                if ((tableFlag & (0x01 << tablePos)) == 0x00)
                {
                    tableNum ++;
                    tableFlag |= 0x01 << tablePos;
                }

                if (pinfo->limSe)
                {
                    // AC Table
                    tablePos = NN_DETAIL_IMAGE_GET_AC_TABLE_NO(sos.huffman[i].tdnTan) + NUM_HUFF_TBLS;
                    if ((tableFlag & (0x01 << tablePos)) == 0x00)
                    {
                        tableNum ++;
                        tableFlag |= 0x01 << tablePos;
                    }
                }
            }
            std::memset(&sos, 0, sizeof(JpegSosInfo));
            start = next;
        }
    }
    if (!(tableNum > 0))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOS\n");
        return false;
    }
    pinfo->derivedTableNum = tableNum;

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Derived table num: %d\n", pinfo->derivedTableNum);
    return true;
}

//! @brief libjpeg で導出される量子化テーブルの数を取得
bool GetQuantTableNum(JpegInfoPtr pinfo, const uint8_t *jpeg, const size_t jpegSize) NN_NOEXCEPT
{
    uint16_t length = 0;
    size_t lengthRemain = 0;
    size_t start = 0;
    size_t offset = 0;
    uint32_t tableNum = 0;
    uint32_t tableFlag = 0;
    uint8_t tablePos = 0;
    uint8_t tableLength = 0;
    uint8_t tableValue = 0;

    while (FindMarkerCoder(&offset, jpeg, jpegSize, start, JpegMarker_Dqt))
    {
        // セグメント長を取得できるか検査
        if (!(offset + JpegProperty_MarkerLength + JpegProperty_SegmentSizeLength <= jpegSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of DQT segment\n");
            return false;
        }
        offset += JpegProperty_MarkerLength;

        // セグメント長の取得と検査
        length = Read2Bytes(jpeg + offset);
        if (!(length >= JpegProperty_SegmentSizeLength && offset + length <= jpegSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of DQT segment (2)\n");
            return false;
        }
        offset += JpegProperty_SegmentSizeLength;
        lengthRemain = length - JpegProperty_SegmentSizeLength;

        while (lengthRemain > 0)
        {
            // 各 DQT ごとにテーブルサイズを計算
            tableValue = Read1Byte(jpeg + offset);
            tableLength = (tableValue >> 4) & 0x0F;
            tablePos = tableValue & 0x0F;
            if (!(tablePos < NUM_QUANT_TBLS))
            {
                NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid DQT pos\n");
                return false;
            }

            // テーブルを記録
            if ((tableFlag & (0x01 << tablePos)) == 0x00)
            {
                tableNum ++;
                tableFlag |= (0x01 << tablePos);
            }

            size_t tableSize = 1 + DCTSIZE2 * (tableLength + 1);
            offset += tableSize;
            lengthRemain -= tableSize;
        }
        if (!(lengthRemain == 0))
        {
            // 不正な DQT サイズ
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid DQT\n");
            return false;
        }

        start = offset;
    }
    if (!(tableNum > 0))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Empty DQT segment\n");
        return false;
    }

    pinfo->dqtCount = tableNum;
    pinfo->dqtMask = tableFlag;

    NN_DETAIL_IMAGE_JPEG_LOG_INFO("Number of quantization table: %d\n", pinfo->dqtCount);
    return true;
}

// @brief 各色コンポーネントの情報を取得
void CalcComponentInfo(JpegInfoPtr pinfo, int32_t scaleParam) NN_NOEXCEPT
{
    // 出力値の初期化
    pinfo->minDctHScaledSize = pinfo->minDctVScaledSize = scaleParam;

    JpegComponentInfo *pCompInfo;
    for (uint8_t i = 0; i < pinfo->sof.nf; i++)
    {
        pCompInfo = &pinfo->sof.componentInfo[i];

        // 出力する画素数を計算
        uint8_t ssize = 1;
        while (!(pinfo->minDctHScaledSize * ssize > DCTSIZE / 2 ||
            (pinfo->sof.maxHn % (pCompInfo->hn * ssize * 2)) != 0))
        {
            ssize = ssize * 2;
        }
        pCompInfo->dctHScaledSize = pinfo->minDctHScaledSize * ssize;
        ssize = 1;
        while (!(pinfo->minDctVScaledSize * ssize > DCTSIZE / 2 ||
            (pinfo->sof.maxVn % (pCompInfo->vn * ssize * 2)) != 0))
        {
            ssize = ssize * 2;
        }
        pCompInfo->dctVScaledSize = pinfo->minDctVScaledSize * ssize;

        // 上界値: 大きい方の値は小さい方の値の2倍の大きさまで。
        if (pCompInfo->dctHScaledSize > pCompInfo->dctVScaledSize * 2)
        {
            pCompInfo->dctHScaledSize = pCompInfo->dctVScaledSize * 2;
        }
        else if (pCompInfo->dctVScaledSize > pCompInfo->dctHScaledSize * 2)
        {
            pCompInfo->dctVScaledSize = pCompInfo->dctHScaledSize * 2;
        }

        // ブロック数の計算
        pCompInfo->widthInBlocks = jpeg::jdiv_round_up(
            pinfo->sof.x * pCompInfo->hn,
            pinfo->sof.maxHn * pinfo->blockSize);
        pCompInfo->heightInBlocks = jpeg::jdiv_round_up(
            pinfo->sof.y * pCompInfo->vn,
            pinfo->sof.maxVn * pinfo->blockSize);
    }

#ifdef UPSAMPLE_MERGING_SUPPORTED
    if (!(pinfo->jpegColorSpace == jpeg::JCS_YCbCr && pinfo->sof.nf == 3 &&
        pinfo->outColorSpace == jpeg::JCS_RGB && pinfo->outColorComponents == RGB_PIXELSIZE &&
        pinfo->sof.componentInfo[0].hn == 2 &&
        pinfo->sof.componentInfo[1].hn == 1 &&
        pinfo->sof.componentInfo[2].hn == 1 &&
        pinfo->sof.componentInfo[0].vn <=  2 &&
        pinfo->sof.componentInfo[1].vn == 1 &&
        pinfo->sof.componentInfo[2].vn == 1 &&
        pinfo->sof.componentInfo[0].dctHScaledSize == pinfo->minDctHScaledSize &&
        pinfo->sof.componentInfo[1].dctHScaledSize == pinfo->minDctHScaledSize &&
        pinfo->sof.componentInfo[2].dctHScaledSize == pinfo->minDctHScaledSize &&
        pinfo->sof.componentInfo[0].dctVScaledSize == pinfo->minDctVScaledSize &&
        pinfo->sof.componentInfo[1].dctVScaledSize == pinfo->minDctVScaledSize &&
        pinfo->sof.componentInfo[2].dctVScaledSize == pinfo->minDctVScaledSize))
    {
        // 入力色空間=YCbCr, 入力色数=3, 出力色空間=RGB, 出力色数=3 じゃなければ upsample 無効
        // サンプリング比が 4:2:0 or 4:2:2 じゃなければ upsample 無効
        // InverseDCT のスケーリングが、コンポーネントごとに等しくなければ upsample 無効
        pinfo->isEnabledMergeUpSample = FALSE;
    }
    else
    {
        pinfo->isEnabledMergeUpSample = TRUE;
    }
#else
    pinfo->isEnabledMergeUpSample = FALSE;
#endif //#if UPSAMPLE_MERGING_SUPPORTED
}

//! @brief JPEG の種類 (= SOFx) を調べる
bool GetJpegType(uint16_t *pOutSofMarker, const uint8_t *jpeg, const size_t jpegSize) NN_NOEXCEPT
{
    static const uint16_t SofList[] = {
        JpegMarker_Sof0, JpegMarker_Sof2, JpegMarker_Sof6, JpegMarker_Sof10, JpegMarker_Sof14};
    size_t offset = 0;
    for (size_t s = 0; s < sizeof(SofList) / sizeof(uint16_t); s++)
    {
        // SOFx を探す
        if (FindMarkerCoder(&offset, jpeg, jpegSize, 0, SofList[s]))
        {
            *pOutSofMarker = SofList[s];
            return true;
        }
    }
    return false;
}

//! @brief 1ブロックをスケーリングしたあとのサイズを調べる
int GetScaleParam(const int blockSize, const int scaleNum, const int scaleDenom) NN_NOEXCEPT
{
    auto scaleParam = jpeg::jdiv_round_up(scaleNum * blockSize, scaleDenom);
    NN_SDK_ASSERT(scaleParam <= INT_MAX);
    return (scaleParam > 16? 16: (scaleParam > 1? scaleParam: 1));
}

//! @brief JPEGデータの色空間を調べる
bool GetJpegColorSpace(
    jpeg::J_COLOR_SPACE *pPutColorSpace,
    const JpegSofInfo &sof,
    const bool sawJfif,
    const bool sawAdobe,
    const uint8_t kAdobeTransform) NN_NOEXCEPT
{
    jpeg::J_COLOR_SPACE colorSpace = jpeg::JCS_YCbCr;

    // 色数で分岐
    switch (sof.nf)
    {
    case 1:
        // 1色 = グレースケール
        colorSpace = jpeg::JCS_GRAYSCALE;
        break;
    case 3:
        if (sawJfif)
        {
            // 3色で JFIF APP0 があれば、YCbCr 固定 (JFIF 仕様)
            colorSpace = jpeg::JCS_YCbCr;
        }
        else if (sawAdobe)
        {
            // 3色で Adobe APP14 があれば、色空間変換のパラメータを見る
            switch (kAdobeTransform)
            {
            case 0:
                colorSpace = jpeg::JCS_RGB;
                break;
            case 1:
            default:
                colorSpace = jpeg::JCS_YCbCr;
                break;
            }
        }
        else
        {
            // 3色で特別なマーカーがなければ、コンポーネントIDの慣例に従う。 (libjpeg の仕様)
            uint32_t cid0 = sof.componentInfo[0].cn;
            uint32_t cid1 = sof.componentInfo[1].cn;
            uint32_t cid2 = sof.componentInfo[2].cn;

            if (cid0 == 1 && cid1 == 2 && cid2 == 3)
            {
                // JFIF っぽい
                colorSpace = jpeg::JCS_YCbCr;
            }
            else if (cid0 == 82 && cid1 == 71 && cid2 == 66)
            {
                // ASCII で 'R','G','B'
                colorSpace = jpeg::JCS_RGB;
            }
            else
            {
                // 標準は YCbCr
                colorSpace = jpeg::JCS_YCbCr;
            }
        }
        break;
    case 4:
        if (sawAdobe)
        {
            // 3色で Adobe APP14 があれば、色空間変換のパラメータを見る
            switch (kAdobeTransform)
            {
            case 0:
                colorSpace = jpeg::JCS_CMYK;
                break;
            case 2:
            default:
                colorSpace = jpeg::JCS_YCCK;
                break;
            }
        }
        else
        {
            // 3色で特別なマーカーがなければ、CMYK を仮定する。
            colorSpace = jpeg::JCS_CMYK;
        }
        break;

    default:
        // 1, 3, 4 色以外は非サポート
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Unknown color space: %02X\n", sof.nf);
        return false;
    }

    *pPutColorSpace = colorSpace;
    return true;
}

//! @brief JPEG画像の色情報を調べる
bool SetOutColorInfo(JpegInfoPtr pinfo, const uint8_t* jpeg, const size_t jpegSize) NN_NOEXCEPT
{
    // JFIF APP0 を探す
    size_t start = 0;
    size_t offset = 0;
    pinfo->hasJfifMarker = false;
    if (FindMarkerCoder(&offset, jpeg, jpegSize, start, JpegMarker_App0))
    {
        offset += JpegProperty_MarkerLength;

        // セグメント長を調べる
        uint16_t segmentLength = Read2Bytes(jpeg + offset);
        if (!(segmentLength >= JpegProperty_SegmentSizeLength && offset + segmentLength <= jpegSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of APP0 segment\n");
            return false;
        }
        offset += JpegProperty_SegmentSizeLength;
        segmentLength -= JpegProperty_SegmentSizeLength;

        if (segmentLength >= sizeof(AppJfifIdentity) &&
            std::memcmp(jpeg + offset, AppJfifIdentity, sizeof(AppJfifIdentity)) == 0)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_INFO("APP0 with JFIF extension found\n");
            pinfo->hasJfifMarker = true;
        }

        offset += segmentLength;
    }

    // Adobe APP14 を探す
    start = offset;
    offset = 0;
    pinfo->hasAdobeMarker = false;
    while (FindMarkerCoder(&offset, jpeg, jpegSize, start, JpegMarker_App14) &&
            !pinfo->hasAdobeMarker)
    {
        offset += JpegProperty_MarkerLength;

        // セグメント長を調べる
        uint16_t segmentLength = Read2Bytes(jpeg + offset);
        if (!(segmentLength >= JpegProperty_SegmentSizeLength && offset + segmentLength <= jpegSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of APP14 segment\n");
            return false;
        }
        offset += JpegProperty_SegmentSizeLength;
        segmentLength -= JpegProperty_SegmentSizeLength;

        // Adobe マーカーの長さ: 12 bytes
        // ID+Ver+F0+F1+Transform = 5+2+2+2+1
        if (segmentLength >= (sizeof(AppAdobeIdentity) + JpegProperty_AdobeMarkerLength) &&
            std::memcmp(jpeg + offset, AppAdobeIdentity, sizeof(AppAdobeIdentity)) == 0)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_INFO("APP14 with Adobe extension found\n");
            pinfo->hasAdobeMarker = true;
            // transform info は adobe[11] に記録されている。 (see examine_app14 (@jdmarker.c))
            pinfo->adobeTransform = Read1Byte(
                jpeg + offset + sizeof(AppAdobeIdentity) + JpegProperty_AdobeTransformOffset);
        }

        start = offset + segmentLength;
    }

    if (!GetJpegColorSpace(
            &pinfo->jpegColorSpace,
            pinfo->sof,
            pinfo->hasJfifMarker,
            pinfo->hasAdobeMarker, pinfo->adobeTransform))
    {
        return false;
    }

    pinfo->outColorSpace = jpeg::JCS_RGB;
    pinfo->outColorComponents = RGB_PIXELSIZE;
    pinfo->isEnabledFancyUpSample = false;
    return true;
}
} // Anonymous namespace

JpegStatus CheckSoi(
    const uint8_t *jpeg,
    const size_t jpegSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(jpegSize >= JpegProperty_MarkerLength);
    NN_UNUSED(jpegSize);
    if (!(Read2Bytes(jpeg) == JpegMarker_Soi))
    {
        return JpegStatus_WrongFormat;
    }
    return JpegStatus_Ok;
}

JpegStatus ReadJpegHeader(
    JpegInfoPtr pinfo,
    const uint8_t *jpeg,
    const size_t jpegSize,
    const uint32_t resolution) NN_NOEXCEPT
{
    size_t eoiOffset = 0;
    if (!FindMarkerCoder(&eoiOffset, jpeg, jpegSize, 0, JpegMarker_Eoi))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("EOI not found, broken\n");
        return JpegStatus_WrongFormat;
    }

    if (!GetQuantTableNum(pinfo, jpeg, jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid DQT\n");
        return JpegStatus_WrongFormat;
    }

    uint16_t sofMarker = 0x0000;
    if (!GetJpegType(&sofMarker, jpeg, jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Unknown type of SOF\n");
        return JpegStatus_UnsupportedFormat;
    }

    if (!GetSofInfo(&pinfo->sof, jpeg, jpegSize, pinfo, sofMarker))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid SOF\n");
        return JpegStatus_WrongFormat;
    }

    if (!SetOutColorInfo(pinfo, jpeg, jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Failed to setup output color\n");
        return JpegStatus_WrongFormat;
    }

    if (!GetHuffTableNum(pinfo, jpeg, jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Invalid DHT\n");
        return JpegStatus_WrongFormat;
    }

    if (!GetBlockSize(pinfo, jpeg, jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Failed to calc block size\n");
        return JpegStatus_WrongFormat;
    }

    if (!GetDerivedTableNum(pinfo, jpeg, jpegSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Failed to derive number of table\n");
        return JpegStatus_WrongFormat;
    }

    // 出力サイズの計算
    int32_t scaleParam = GetScaleParam(
        pinfo->blockSize,
        1, (resolution < pinfo->blockSize)? resolution: pinfo->blockSize);
    pinfo->outputWidth = NN_DETAIL_IMAGE_GET_SCALED_SIZE(pinfo->sof.x, pinfo->blockSize, scaleParam);
    pinfo->outputHeight = NN_DETAIL_IMAGE_GET_SCALED_SIZE(pinfo->sof.y, pinfo->blockSize, scaleParam);
    CalcComponentInfo(pinfo, scaleParam);

    return JpegStatus_Ok;
}

void SetJpegHeader(
    JpegInfoPtr pinfo,
    const uint16_t width,
    const uint16_t height,
    const JpegSamplingRatio kSample) NN_NOEXCEPT
{
#define NN_DETAIL_IMAGE_SET_COMP(index,id,hsamp,vsamp) \
    (pCompInfo = &pinfo->sof.componentInfo[index], pCompInfo->cn = (id), pCompInfo->hn = (hsamp), pCompInfo->vn = (vsamp))

    JpegComponentInfo *pCompInfo;
    // 画像サイズの設定
    pinfo->sof.x = width;
    pinfo->sof.y = height;

    // Component Count = YCbCr (=3) @ jpeg_set_colorspace()
    pinfo->sof.nf = JpegProperty_YccComponentNum;

    // BlockSize = DCTSIZE (=8) @ jpeg_CreateCompress()
    pinfo->blockSize = DCTSIZE;
    pinfo->limSe = DCTSIZE2 - 1;

    // コンポーネント設定 @ jpeg_set_colorspace()
    switch (kSample)
    {
    case JpegSamplingRatio_444:
        NN_DETAIL_IMAGE_SET_COMP(0, 1, 1, 1);
        NN_DETAIL_IMAGE_SET_COMP(1, 2, 1, 1);
        NN_DETAIL_IMAGE_SET_COMP(2, 3, 1, 1);
        pinfo->sof.maxHn = 1;
        pinfo->sof.maxVn = 1;
        break;
    case JpegSamplingRatio_422:
        NN_DETAIL_IMAGE_SET_COMP(0, 1, 2, 1);
        NN_DETAIL_IMAGE_SET_COMP(1, 2, 1, 1);
        NN_DETAIL_IMAGE_SET_COMP(2, 3, 1, 1);
        pinfo->sof.maxHn = 2;
        pinfo->sof.maxVn = 1;
        break;
    case JpegSamplingRatio_420:
        NN_DETAIL_IMAGE_SET_COMP(0, 1, 2, 2);
        NN_DETAIL_IMAGE_SET_COMP(1, 2, 1, 1);
        NN_DETAIL_IMAGE_SET_COMP(2, 3, 1, 1);
        pinfo->sof.maxHn = 2;
        pinfo->sof.maxVn = 2;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    // その他のコンポーネント情報を計算
    CalcComponentInfo(pinfo, DCTSIZE);

#undef NN_DETAIL_IMAGE_SET_COMP
}

bool GetExifInfo(
    const void **pData,
    uint16_t *pSize,
    const uint8_t *jpeg,
    const size_t kJpegSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        pData != nullptr && pSize != nullptr && jpeg != nullptr && kJpegSize >= JpegProperty_SegmentSizeLength,
        "Bad argument");

    bool isFound = false;
    size_t offset = 0;
    if (FindMarkerCoder(&offset, jpeg, kJpegSize, 0, JpegMarker_App1))
    {
        offset += JpegProperty_MarkerLength;

        uint16_t length = Read2Bytes(jpeg + offset);
        if (!(length >= JpegProperty_SegmentSizeLength && offset + length <= kJpegSize))
        {
            NN_DETAIL_IMAGE_JPEG_LOG_ERROR("Wrong size of APP0 segment\n");
            return false;
        }
        offset += JpegProperty_SegmentSizeLength;

        length -= JpegProperty_SegmentSizeLength;
        if (length >= sizeof(AppExifIdentity) &&
            std::memcmp(jpeg + offset, AppExifIdentity, sizeof(AppExifIdentity)) == 0)
        {
            NN_DETAIL_IMAGE_JPEG_LOG_INFO("APP1 with Exif extension isFound\n");
            isFound = true;
            // シグニチャを飛ばして TIFF ヘッダの先頭を返す。
            *pData = reinterpret_cast<const void*>(jpeg + offset + 6);
            *pSize = length - 6;
        }
    }

    return isFound;
}

void WriteExifHeader(void *outBuf, const size_t kOutputSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        kOutputSize >= JpegProperty_ExifAppHeaderSize && kOutputSize - 2 <= 0xFFFF,
        "Invalid size of APP1(EXIF) buffer\n");

    jpeg::JOCTET *bytes = reinterpret_cast<jpeg::JOCTET*>(outBuf);
    Write2Bytes(bytes, JpegMarker_App1);
    Write2Bytes(bytes + 2, static_cast<uint16_t>(kOutputSize - 2));

    std::memcpy(bytes + 4, AppExifIdentity, sizeof(AppExifIdentity));
}

}}}
