﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>

#include "image_JpegConfig.h"
#include "image_ExifParser.h"
#include "image_JpegParser.h"

namespace nn { namespace image { namespace detail {

namespace
{
//! @brief BigEndian 用 2 バイト読み込み
uint16_t ReadBe2Bytes(const uint8_t *data) NN_NOEXCEPT
{
    return
        ((static_cast<uint16_t>(data[0])) << 8)
        + (static_cast<uint16_t>(data[1]));
}
//! @brief LittleEndian 用 2 バイト読み込み
uint16_t ReadLe2Bytes(const uint8_t *data) NN_NOEXCEPT
{
    return
        (static_cast<uint16_t>(data[0]))
        + ((static_cast<uint16_t>(data[1])) << 8);
}
//! @brief BigEndian 用 4 バイト読み込み
uint32_t ReadBe4Bytes(const uint8_t *data) NN_NOEXCEPT
{
    return
        ((static_cast<uint32_t>(data[0])) << 24)
        + ((static_cast<uint32_t>(data[1])) << 16)
        + ((static_cast<uint32_t>(data[2])) << 8)
        + ((static_cast<uint32_t>(data[3])));
}
//! @brief LittleEndian 用 4 バイト読み込み
uint32_t ReadLe4Bytes(const uint8_t *data) NN_NOEXCEPT
{
    return
        ((static_cast<uint32_t>(data[0])))
        + ((static_cast<uint32_t>(data[1])) << 8)
        + ((static_cast<uint32_t>(data[2])) << 16)
        + ((static_cast<uint32_t>(data[3])) << 24);
}

//! @brief 1 バイト書き込み
NN_FORCEINLINE void Write1Byte(uint8_t *data, uint8_t value) NN_NOEXCEPT
{
    *(data) = (value);
}
//! @brief 非 ASCII を空白に置換しつつ 1 バイト書き込み
NN_FORCEINLINE void WriteAscii(uint8_t *data, uint8_t value) NN_NOEXCEPT
{
    *(data) = (((value) & 0x80) != 0)? 0x20: (value);
}
//! @brief BigEndian 用 2 バイト書き込み
void WriteBe2Bytes(uint8_t *data, const uint16_t v) NN_NOEXCEPT
{
    data[0] = static_cast<uint8_t>((v & 0xFF00) >> 8);
    data[1] = static_cast<uint8_t>((v & 0x00FF));
}
//! @brief LittleEndian 用 2 バイト書き込み
void WriteLe2Bytes(uint8_t *data, const uint16_t v) NN_NOEXCEPT
{
    data[0] = static_cast<uint8_t>((v & 0x00FF));
    data[1] = static_cast<uint8_t>((v & 0xFF00) >> 8);
}
//! @brief BigEndian 用 4 バイト書き込み
void WriteBe4Bytes(uint8_t *data, const uint32_t v) NN_NOEXCEPT
{
    data[0] = static_cast<uint8_t>((v & 0xFF000000) >> 24);
    data[1] = static_cast<uint8_t>((v & 0x00FF0000) >> 16);
    data[2] = static_cast<uint8_t>((v & 0x0000FF00) >> 8);
    data[3] = static_cast<uint8_t>((v & 0x000000FF));
}
//! @brief LittleEndian 用 4 バイト書き込み
void WriteLe4Bytes(uint8_t *data, const uint32_t v) NN_NOEXCEPT
{
    data[0] = static_cast<uint8_t>((v & 0x000000FF));
    data[1] = static_cast<uint8_t>((v & 0x0000FF00) >> 8);
    data[2] = static_cast<uint8_t>((v & 0x00FF0000) >> 16);
    data[3] = static_cast<uint8_t>((v & 0xFF000000) >> 24);
}

//! @brief タグごとの単位サイズを取得
uint16_t GetTagSizeUnit(const IfdTagType kType) NN_NOEXCEPT
{
    switch (kType)
    {
    case IfdTagType_Byte:
    case IfdTagType_Ascii:
    case IfdTagType_Undefined:
        return 1;

    case IfdTagType_Short:
        return 2;

    case IfdTagType_Long:
#ifdef NN_DETAIL_IMAGE_JPEG_CONFIG_EXIF_SIGNED_VALUE_SUPPORTED
    case IfdTagType_SignedLong:
#endif
        return 4;

    case IfdTagType_Rational:
#ifdef NN_DETAIL_IMAGE_JPEG_CONFIG_EXIF_SIGNED_VALUE_SUPPORTED
    case IfdTagType_SignedRational:
#endif
        return 8;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

//! @brief BigEndian 用バイナリ読み書き環境を取得
void GetBigEndianIo(BinaryIo *pBinaryIo) NN_NOEXCEPT
{
    pBinaryIo->read2 = ReadBe2Bytes;
    pBinaryIo->read4 = ReadBe4Bytes;
    pBinaryIo->write2 = WriteBe2Bytes;
    pBinaryIo->write4 = WriteBe4Bytes;
}
//! @brief LittleEndian 用バイナリ読み書き環境を取得
void GetLittleEndianIo(BinaryIo *pBinaryIo) NN_NOEXCEPT
{
    pBinaryIo->read2 = ReadLe2Bytes;
    pBinaryIo->read4 = ReadLe4Bytes;
    pBinaryIo->write2 = WriteLe2Bytes;
    pBinaryIo->write4 = WriteLe4Bytes;
}

NN_FORCEINLINE bool CheckIfdOffset(uint32_t kOffset, uint16_t kSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT(kSize >= ExifProperty_IfdSizeMin);
    return (
        kOffset >= ExifProperty_TiffHeaderSize &&
        kOffset <= static_cast<uint32_t>(kSize - ExifProperty_IfdSizeMin));
}
} /* Anonymous namespace */


bool ReadTiffHeader(
    BinaryIo * const pBinaryIo,
    uint16_t *pOffset0thIfd,
    const uint8_t *exifData,
    const uint16_t kExifSize) NN_NOEXCEPT
{
    if (!(kExifSize >= ExifProperty_TiffHeaderSize + ExifProperty_IfdSizeMin &&
        kExifSize <= ExifProperty_SizeMax))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("TIFF overruns the bound.\n");
        return false;
    }

    // エンディアンと IO を取得
    switch (ReadBe2Bytes(exifData))
    {
    case 0x4949:
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - LittleEndian\n");
        GetLittleEndianIo(pBinaryIo);
        break;

    case 0x4D4D:
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - BigEndian\n");
        GetBigEndianIo(pBinaryIo);
        break;

    default:
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("TIFF header is recorded with unknown endian type.\n");
        return false;
    }

    // マジックナンバーをオフセット 2 から取得
    uint32_t offset = 2;
    if (!(pBinaryIo->read2(exifData + offset) == 0x002A))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("TIFF header countains unknown magic number.\n");
        return false;
    }

    // 0th TIFF IFD へのオフセットをオフセット 4 から取得
    offset += 2;
    uint32_t ifdOffset = pBinaryIo->read4(exifData + offset);
    if (!CheckIfdOffset(ifdOffset, kExifSize))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR(
            "TIFF: IFD offset overruns the bound (size=0x%04x, next=0x%04x).\n",
            kExifSize, ifdOffset);
        return false;
    }

    *pOffset0thIfd = static_cast<uint16_t>(ifdOffset);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - 0th TIFF IFD offset: 0x%04x.\n", *pOffset0thIfd);
    return true;
}

bool ReadIfdHeader(
    IfdHeader *pIfdHeader,
    const uint16_t kIfdOffset,
    const ExifBinary &kExif) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        kIfdOffset != IfdTagState_NotSpecified,
        "Reading unspecified IFD\n");

    // オフセットの初期化 (16bit -> 32bit)
    uint32_t offset = kIfdOffset;
    if (!CheckIfdOffset(offset, kExif.size))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("IFD@0x%04x: IFD overruns the bound.\n", kIfdOffset);
        return false;
    }

    // タグ数を取得
    uint16_t count = kExif.io.read2(kExif.data + offset);
    if (!(offset + ExifProperty_IfdSizeMin + count * ExifProperty_IfdTagSize <= kExif.size))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("IFD@0x%04x: IFD tags overruns the bound.\n", kIfdOffset);
        return false;
    }

    // 次の IFD へのオフセットを取得
    offset += 2 + count * ExifProperty_IfdTagSize;
    uint32_t nextOffset = kExif.io.read4(kExif.data + offset);
    if (!(nextOffset == 0 || CheckIfdOffset(nextOffset, kExif.size)))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR(
            "IFD@0x%04x: Next IFD offset overruns the bound (size=0x%04x, next=0x%04x).\n",
            kIfdOffset, kExif.size, nextOffset);
        return false;
    }

    pIfdHeader->offset = kIfdOffset;
    pIfdHeader->tagCount = count;
    pIfdHeader->nextOffset = static_cast<uint16_t>(nextOffset);

    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Offset:    0x%04x\n",  pIfdHeader->offset);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Tag count: %u\n",  pIfdHeader->tagCount);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Next IFD:  0x%04x\n", pIfdHeader->nextOffset);
    return true;
}

bool SearchIfdTags(
    IfdTag tags[], // Must be sorted by tag ID (asc)
    const uint16_t kNumTags,
    const IfdHeader &kIfd,
    const ExifBinary &kExif) NN_NOEXCEPT
{
    NN_SDK_ASSERT(
        kIfd.offset != IfdTagState_NotSpecified,
        "Reading unspecified IFD\n");

#define NN_DETAIL_IMAGE_GET_IFD_FIELF_OFFSET(ifdOffset, tagIndex) ((ifdOffset) + 2 + (tagIndex) * ExifProperty_IfdTagSize)

    uint32_t arrayIndex = 0;
    uint32_t ifdIndex = 0;
    uint32_t count = 0;
    uint32_t valueOffset = IfdTagState_NotSpecified;

    while (arrayIndex < kNumTags && ifdIndex < kIfd.tagCount)
    {
        const uint8_t *dataPos = kExif.data + NN_DETAIL_IMAGE_GET_IFD_FIELF_OFFSET(kIfd.offset, ifdIndex);

        /* ID check */
        IfdTag &tag = tags[arrayIndex];
        uint16_t dataId = kExif.io.read2(dataPos);
        if (dataId < tag.id)
        {
            /* 探しているのと違うタグが見つかった */
            NN_DETAIL_IMAGE_JPEG_LOG_DETAIL(
                "   - Skip:  %04x (type=%u, count=%lu)\n",
                dataId, kExif.io.read2(dataPos + 2), kExif.io.read4(dataPos + 4));
            ifdIndex ++;
            continue;
        }

        bool isExpected = false;
        if (dataId == tag.id)
        {
            do
            {
                NN_DETAIL_IMAGE_JPEG_LOG_INFO(
                    "   - Found: %04x (type=%u, count=%lu)\n",
                    tag.id, tag.type, tag.valueCount);

                /* Type check */
                uint16_t type = kExif.io.read2(dataPos + 2);
                if (tag.type == IfdTagType_ShortOrLong)
                {
                    // Short or Long のタグはどちらかであれば OK
                    if (!(type == IfdTagType_Short || type == IfdTagType_Long))
                    {
                        NN_DETAIL_IMAGE_JPEG_LOG_WARN(
                            "Tag type (%u) is unexpected (%u or %u)\n",
                            type, IfdTagType_Short, IfdTagType_Long);
                        break;
                    }
                    tag.type = (type == IfdTagType_Short? IfdTagType_Short: IfdTagType_Long);
                }
                else if (type != tag.type)
                {
                    NN_DETAIL_IMAGE_JPEG_LOG_WARN("Tag type (%u) is unexpected (%u)\n", type, tag.type);
                    break;
                }

                /* Count check */
                count = kExif.io.read4(dataPos + 4);
                if (!(count < 0xFFFF &&
                     (count == tag.valueCount || tag.valueCount == IfdTagState_NotSpecified)))
                {
                    NN_DETAIL_IMAGE_JPEG_LOG_WARN("Tag count (%u) is unexpected (%u)\n", count, tag.valueCount);
                    break;
                }

                /* Offset check */
                uint32_t valueSize = count * GetTagSizeUnit(tag.type);
                valueOffset =
                    (valueSize <= 4)? // 単位x数 が 4byte 以下なら...
                        (NN_DETAIL_IMAGE_GET_IFD_FIELF_OFFSET(kIfd.offset, ifdIndex) + 8): // 値が offset のフィールドにある。 (0 個でも。)
                        (kExif.io.read4(dataPos + 8)); // 4byte 以上なら、offset のフィールドを読む。
                if (!(valueOffset < kExif.size && valueOffset + valueSize <= kExif.size))
                {
                    // 値が変な場所に書いてあることになっているとエラー。
                    NN_DETAIL_IMAGE_JPEG_LOG_ERROR(
                        "Value offset (0x%lx-0x%lx) overruns the bound.\n",
                        valueOffset, valueOffset + valueSize);
                    return false;
                }

                NN_DETAIL_IMAGE_JPEG_LOG_INFO("     - Count:  %u\n", count);
                NN_DETAIL_IMAGE_JPEG_LOG_INFO("     - Offset: 0x%04x\n", valueOffset);
                isExpected = true;
                ifdIndex ++;
            } while (NN_STATIC_CONDITION(false));
        }

        if (!isExpected)
        {
            // タグが期待した内容ではなかったら、存在しないことにする。
            NN_DETAIL_IMAGE_JPEG_LOG_WARN(
                "Tag not found: %04x (type=%u, count=%lu)\n",
                tag.id, tag.type, tag.valueCount);
            count = 0;
            valueOffset = IfdTagState_NotSpecified;
        }

        tag.valueCount = static_cast<uint16_t>(count);
        tag.valueOffset = static_cast<uint16_t>(valueOffset);
        arrayIndex ++;
    }

    // 調べるまでもなかったタグは明示的に掃除する。
    for (; arrayIndex < kNumTags; arrayIndex++)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_WARN(
            "Tag not found: %04x (type=%u, count=%lu)\n",
            tags[arrayIndex].id, tags[arrayIndex].type, tags[arrayIndex].valueCount);
        tags[arrayIndex].valueCount = 0;
        tags[arrayIndex].valueOffset = IfdTagState_NotSpecified;
    }

    return true;

#undef NN_DETAIL_IMAGE_GET_IFD_FIELF_OFFSET
}

void InitializeTiffHeader(
    TiffHeader *pTiffHeader,
    const Endian kEndian) NN_NOEXCEPT
{
    // 与えられたエンディアンで IO を初期化
    switch (kEndian)
    {
    case Endian_Little:
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - LittleEndian\n");
        GetLittleEndianIo(&pTiffHeader->io);
        break;
    case Endian_Big:
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - BigEndian\n");
        GetBigEndianIo(&pTiffHeader->io);
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }

    pTiffHeader->endian = kEndian;
    pTiffHeader->offset0thIfd = 0x0008;
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - 0th TIFF IFD offset: 0x%04x.\n", pTiffHeader->offset0thIfd);
}

void WriteTiffHeader(
    uint8_t *exifData,
    const uint16_t kExifSize,
    const TiffHeader &tiffHeader) NN_NOEXCEPT
{
    NN_SDK_ASSERT(kExifSize >= ExifProperty_TiffHeaderSize + ExifProperty_IfdSizeMin, "");
    NN_UNUSED(kExifSize);

#if NN_DETAIL_IMAGE_JPEG_CONFIG_DEBUG_PRINT >= 1
    switch (tiffHeader.endian)
    {
    case Endian_Big:
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - BigEndian\n");
        break;
    case Endian_Little:
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - LittleEndian\n");
        break;
    default:
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - (unknown endian type: 0x%04x)\n", tiffHeader.endian);
        break;
    }
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - 0th TIFF IFD offset: 0x%04x.\n", tiffHeader.offset0thIfd);
#endif

    // エンディアン書き込み
    NN_SDK_ASSERT(
        tiffHeader.endian == Endian_Big || tiffHeader.endian == Endian_Little,
        "");
    tiffHeader.io.write2(exifData, tiffHeader.endian);

    // マジックナンバー
    uint32_t offset = 2;
    tiffHeader.io.write2(exifData + offset, 0x002A);

    // 0th TIFF IFD へのオフセット
    offset += 2;
    tiffHeader.io.write4(exifData + offset, tiffHeader.offset0thIfd);
}

bool InitializeIfdHeader(
    IfdHeader *pIfdHeader,
    const ValuedIfdTag *tags,
    const uint16_t kNumTags,
    const uint16_t kIfdOffset) NN_NOEXCEPT
{
    // 必ず消費するサイズ + タグ数*タグサイズ
    uint32_t offset = kIfdOffset + ExifProperty_IfdSizeMin + kNumTags * ExifProperty_IfdTagSize;
    if (!(offset <= ExifProperty_SizeMax))
    {
        NN_DETAIL_IMAGE_JPEG_LOG_ERROR("IFD overruns the bound: 0x%lx\n", offset);
        return false;
    }

    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Offset:    0x%04x\n", kIfdOffset);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - IFD end:   0x%04lx\n", offset);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Tags[%u]:\n", kNumTags);

    // 各タグの内容を検査する。
    for (uint16_t i = 0; i < kNumTags; i++)
    {
        NN_DETAIL_IMAGE_JPEG_LOG_INFO(
            "   - Tag: %04x (type=%u, count=%lu)\n",
            tags[i].tag.id, tags[i].tag.type, tags[i].tag.valueCount);

        uint32_t size = tags[i].tag.valueCount * GetTagSizeUnit(tags[i].tag.type);
        if (size > 4)
        {
            // 単位x数 が 4byte 以上のタグについて、値のサイズも IFD の大きさに含める。
            NN_DETAIL_IMAGE_JPEG_LOG_INFO(
                "     - Additional data size: %ux%u=%u\n",
                tags[i].tag.valueCount,
                GetTagSizeUnit(tags[i].tag.type),
                size);
            if (!(offset + size <= ExifProperty_SizeMax))
            {
                NN_DETAIL_IMAGE_JPEG_LOG_ERROR("IFD value overruns the bound: 0x%lx\n", offset);
                return false;
            }

            offset += size;
        }
    }

    pIfdHeader->offset = kIfdOffset;
    pIfdHeader->tagCount = kNumTags;
    pIfdHeader->nextOffset = static_cast<uint16_t>(offset);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - Next IFD:  0x%04x\n", pIfdHeader->nextOffset);
    return true;
}

void WriteIfd(
    uint8_t *exifData,
    const uint16_t kExifSize,
    const ValuedIfdTag *tags,
    const IfdHeader &ifdHeader,
    const BinaryIo &io) NN_NOEXCEPT
{
#define NN_DETAIL_IMAGE_FOR_EACH_DO(aData, aLength, aUnit, aOp, data, aOffset) \
    do \
    { \
        for (uint16_t dataIndex = 0; dataIndex < (aLength); dataIndex ++) \
        { \
            aOp((data) + (aOffset) + dataIndex * (aUnit), (aData)[dataIndex]); \
        } \
    } while (NN_STATIC_CONDITION(false))

    uint32_t offset = ifdHeader.offset;
    uint32_t dataOffset = offset + ExifProperty_IfdSizeMin + ifdHeader.tagCount * ExifProperty_IfdTagSize;
    NN_SDK_ASSERT(offset < kExifSize && dataOffset <= kExifSize, "");
    NN_UNUSED(kExifSize);

    // タグ数を先頭に書く
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - [0x%04x] Tag count: %u \n", offset, ifdHeader.tagCount);
    io.write2(exifData + offset, ifdHeader.tagCount);

    offset += 2;
    for (uint16_t i = 0; i < ifdHeader.tagCount; i++)
    {
        NN_SDK_ASSERT(offset + ExifProperty_IfdTagSize <= kExifSize, "");

        const ValuedIfdTag &vtag = tags[i];

        NN_DETAIL_IMAGE_JPEG_LOG_INFO(
            " - [0x%04x] Tag: %04x (type=%u, count=%lu)\n",
            offset, vtag.tag.id, vtag.tag.type, vtag.tag.valueCount);
        io.write2(exifData + offset, vtag.tag.id);
        io.write2(exifData + offset + 2, static_cast<uint16_t>(vtag.tag.type));
        io.write4(exifData + offset + 4, static_cast<uint32_t>(vtag.tag.valueCount));

        uint16_t unit = GetTagSizeUnit(vtag.tag.type);
        uint32_t dataSize = vtag.tag.valueCount * unit;
        uint32_t valueOffset = IfdTagState_NotSpecified;
        NN_DETAIL_IMAGE_JPEG_LOG_INFO("            - DataSize: %u\n", dataSize);
        if (dataSize <= 4)
        {
            // offset の領域に値が収まるケース

            valueOffset = ((vtag.tag.valueCount != 0)? offset + 8: IfdTagState_NotSpecified);
            // 一度クリアしておく。
            io.write4(exifData + offset + 8, 0x00000000);
        }
        else
        {
            // IFD の後ろに値を書くケース

            NN_SDK_ASSERT(dataSize < ExifProperty_SizeMax, "");
            valueOffset = dataOffset;
            dataOffset += dataSize;

            NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - [0x%04x] Data\n", valueOffset);
            io.write4(exifData + offset + 8, valueOffset);
        }
        NN_SDK_ASSERT(valueOffset + dataSize <= kExifSize, "");

        if (valueOffset != IfdTagState_NotSpecified)
        {
            // 値を、タグ種別ごとに異なる方法で書き込む

            switch (vtag.tag.type)
            {
            case IfdTagType_Byte:
            case IfdTagType_Undefined:
                NN_DETAIL_IMAGE_FOR_EACH_DO(vtag.value.vBytes, vtag.tag.valueCount, unit, Write1Byte, exifData, valueOffset);
                break;

            case IfdTagType_Ascii:
                /* Write user input ASCII data with sanitizing. */
                NN_DETAIL_IMAGE_FOR_EACH_DO(vtag.value.vAscii, vtag.tag.valueCount - 1, unit, WriteAscii, exifData, valueOffset);
                *(exifData + valueOffset + (vtag.tag.valueCount - 1) * unit) = '\0'; // Force terminate
                break;

            case IfdTagType_Short:
                NN_DETAIL_IMAGE_FOR_EACH_DO(vtag.value.vShort, vtag.tag.valueCount, unit, io.write2, exifData, valueOffset);
                break;

            case IfdTagType_Long:
                NN_DETAIL_IMAGE_FOR_EACH_DO(vtag.value.vLong, vtag.tag.valueCount, unit, io.write4, exifData, valueOffset);
                break;

            case IfdTagType_Rational:
                // Rational は分母と分子をそれぞれ Long で書く
                for (uint16_t j = 0; j < vtag.tag.valueCount; j++)
                {
                    io.write4(
                        exifData + valueOffset + j * unit,
                        vtag.value.vRational[j].numerator);
                    io.write4(
                        exifData + valueOffset + j * unit + 4,
                        vtag.value.vRational[j].denominator);
                }
                break;

#ifdef NN_DETAIL_IMAGE_JPEG_CONFIG_EXIF_SIGNED_VALUE_SUPPORTED
            case IfdTagType_SignedLong:
                // SignedLong はビットパターンを維持して Long に変換して書く
                NN_DETAIL_IMAGE_FOR_EACH_DO(vtag.value.vSignedLong, vtag.tag.valueCount, unit, io.write4, exifData, valueOffset);
                break;

            case IfdTagType_SignedRational:
                // SignedRational は分母と分子をそれぞれを SignedLong で書く
                for (uint16_t j = 0; j < vtag.tag.valueCount; j++)
                {
                    io.write4(
                        exifData + valueOffset + j * unit,
                        vtag.value.vSignedRational[j].numerator);
                    io.write4(
                        exifData + valueOffset + j * unit + 4,
                        vtag.value.vSignedRational[j].denominator);
                }
                break;
#endif
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
        offset += ExifProperty_IfdTagSize;
    }

    // 次の IFD へのオフセットを書いて終了
    NN_SDK_ASSERT(
        offset + 4 <= kExifSize &&
        (offset + 4u == static_cast<uint32_t>(ifdHeader.offset) + ExifProperty_IfdSizeMin + ifdHeader.tagCount * ExifProperty_IfdTagSize),
        "'offset' vioration: %d\n", offset);
    NN_DETAIL_IMAGE_JPEG_LOG_INFO(" - [0x%04x] Next IFD:  %u \n", offset, ifdHeader.nextOffset);
    io.write4(exifData + offset, static_cast<uint32_t>(ifdHeader.nextOffset));

#undef NN_DETAIL_IMAGE_FOR_EACH_DO
}

}}}
