﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/edid/edid_Lib.h>
#include <nn/edid/edid_Edid.h>
#include "detail/edid_Edid.h"

static bool IsBlockChecksumValid(const std::uint8_t* pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pData);

    std::uint8_t checksum = 0;

    for(const std::uint8_t* ptr = pData; ptr != pData + size; ++ptr)
    {
        checksum += *ptr;
    }

    return checksum == 0;
}

nn::edid::Error nn::edid::OpenEdid(Edid* pOutEdid, const void* pData, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEdid);
    NN_SDK_REQUIRES_NOT_NULL(pData);

    detail::Initialize(pOutEdid);

    if( size == 0 || size % detail::BlockSize != 0 )
    {
        return nn::edid::Error_InvalidSize;
    }

    const std::uint8_t* pEdid = static_cast<const std::uint8_t*>(pData);

    // validate checksums
    for( const std::uint8_t* ptr = pEdid; ptr != pEdid + size; ptr += detail::BlockSize )
    {
        if( !IsBlockChecksumValid(pEdid, detail::BlockSize) )
        {
            return nn::edid::Error_Corrupt;
        }
    }

    // validate header
    if( pEdid[0] != 0x00 ||
        pEdid[1] != 0xFF ||
        pEdid[2] != 0xFF ||
        pEdid[3] != 0xFF ||
        pEdid[4] != 0xFF ||
        pEdid[5] != 0xFF ||
        pEdid[6] != 0xFF ||
        pEdid[7] != 0x00 )
    {
        return nn::edid::Error_Corrupt;
    }

    // TODO: any other checks?
    // TODO: check block maps vs. size

    pOutEdid->_data = pData;
    pOutEdid->_size = size;

    return nn::edid::Error_None;
}

void nn::edid::CloseEdid(Edid* pEdid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEdid);

    pEdid->_data = nullptr;
    pEdid->_size = 0;
}

void nn::edid::PrintEdid(const Edid* pEdid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEdid);

    NN_UNUSED(pEdid);
}
