﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SdkAssert.h>
#include <nn/edid/edid_DisplayMode.h>
#include <nn/edid/edid_Edid.h>
#include <nn/edid/edid_DisplayModeInfo.h>
#include <nn/edid/edid_DisplayTimingInfo.h>
#include <nn/edid/edid_ImageSizeInfo.h>
#include "detail/edid_Serialize.h"
#include "detail/edid_DisplayModeInfo.h"
#include "detail/edid_Edid.h"
#include "cea861/edid_Cea861.h"

const std::uint8_t EstablishedTimingIOffset   = 0x23;
const std::uint8_t EstablishedTimingIIOffset  = 0x24;
const std::uint8_t EstablishedTimingIIIOffset = 0x25;

const std::uint8_t StandardTimingOffset = 0x26;
const std::uint8_t DetailedTimingOffset = 0x36;

void nn::edid::VisitDisplayModes(const Edid* pEdid, DisplayModeVisitor visitor, void* pUserData) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEdid);
    NN_SDK_REQUIRES_NOT_NULL(visitor);

    if( !detail::IsValid(pEdid) )
    {
        return;
    }

    const std::uint8_t* pData = static_cast<const std::uint8_t*>(pEdid->_data);

    // Established Timings
    detail::GetEstablishedTimingI(visitor, pUserData, pData[EstablishedTimingIOffset]);
    detail::GetEstablishedTimingII(visitor, pUserData, pData[EstablishedTimingIIOffset]);
    detail::GetEstablishedTimingIII(visitor, pUserData, pData[EstablishedTimingIIIOffset]);

    // Standard Timings
    const int StandardTimingStride = 2;
    for( const std::uint8_t* pStandardTiming = pData + StandardTimingOffset;
         pStandardTiming < pData + DetailedTimingOffset;
         pStandardTiming += StandardTimingStride )
    {
        // first byte is 0 indicates reserved
        // 0x0101 indicates unused
        if( *pStandardTiming == 0 || (*pStandardTiming == 0x01 && pStandardTiming[1] == 0x01) )
        {
            continue;
        }

        DisplayModeInfo mode;
        detail::GetStandardTiming(&mode, *pStandardTiming, pStandardTiming[1]);

        visitor(&mode, nullptr, nullptr, pUserData);
    }

    for( const std::uint8_t* pDetailedTiming = pData + DetailedTimingOffset;
         pDetailedTiming < (pData + DetailedTimingOffset) + detail::DetailedTimingStride * 4;
         pDetailedTiming += detail::DetailedTimingStride )
    {
        DisplayModeInfo mode;
        DisplayTimingInfo timing;
        ImageSizeInfo image;

        if( detail::GetDetailedTiming(&mode, &timing, &image, pDetailedTiming, detail::DetailedTimingStride) )
        {
            visitor(&mode, &timing, &image, pUserData);
        }
    }

    detail::DetailedTimingVisitorData visitorData;
    visitorData.visitor = visitor;
    visitorData.pUserData = pUserData;

    cea861::VisitBlocks(pEdid, detail::ModeExtensionBlockVisitor, &visitorData);
}

void nn::edid::GetDisplayModePreference(DisplayModeInfo* pOutMode, DisplayTimingInfo* pOutTiming, ImageSizeInfo* pOutSize, const Edid* pEdid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutMode);
    NN_SDK_REQUIRES_NOT_NULL(pOutTiming);
    NN_SDK_REQUIRES_NOT_NULL(pOutSize);
    NN_SDK_REQUIRES_NOT_NULL(pEdid);

    NN_UNUSED(pOutMode);
    NN_UNUSED(pOutTiming);
    NN_UNUSED(pOutSize);
    NN_UNUSED(pEdid);
}
