﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>
#include <nn/util/util_StringView.h>
#include <nn/util/util_IntrusiveDic.h>
#include <nn/utilTool/utilTool_XmlParser.h>
#include <nn/utilTool/utilTool_BinarizerContext.h>

namespace nn { namespace utilTool {

//! @brief コンバータユーティリティ
class ConverterUtil
{
public:
    // !@brief 文字列から値へ変換します。
    static void Parse( float* pData, const char* nptr, char** endptr = NULL ) NN_NOEXCEPT;

    // !@brief 文字列から10進数の値へ変換します。
    static void Parse( int* pData, const char* nptr, char** endptr = NULL ) NN_NOEXCEPT;

    // !@brief 文字列から16進数の値へ変換します。
    static void Parse( char* pData, const char* nptr, char** endptr = NULL ) NN_NOEXCEPT;

    // !@brief 文字列から値へ変換します。
    static void Parse( bool* pData, const char* nptr, char** endptr = NULL ) NN_NOEXCEPT;

    //! @brief 文字列から値を読み込みます。
    template< typename T >
    static void Read( ConverterContext* /*pCtx*/, T* pData, const nn::util::string_view& str ) NN_NOEXCEPT
    {
        Parse( pData, str.data() );
    }

    // !@brief 文字列から値の配列を読み込みます。
    template< typename T >
    static void ReadArray( ConverterContext* pCtx, T* pData, const nn::util::string_view& str, int count );

    //! @brief 属性から値を読み込みます。
    template< typename T >
    static void ReadAttribute( ConverterContext* pCtx, T* pTarget, const XmlNode* pNode, const nn::util::string_view& name );

    //! @brief 属性から値を読み込みます。
    template< typename T, int N >
    static void ReadAttribute( ConverterContext* pCtx, T* pTarget, const XmlNode* pNode, const nn::util::string_view& name, const nn::util::string_view ( &table )[ N ] );

private:
    static const nn::util::string_view g_Spaces;
};

//--------------------------------------------------------------------------------------------------

// クラスメンバの定義
//! @cond

template<>
inline
void ConverterUtil::Read( ConverterContext* /*pCtx*/, nn::util::string_view* pData, const nn::util::string_view& str ) NN_NOEXCEPT
{
    *pData = str;
}

template<>
inline
void ConverterUtil::Read( ConverterContext* /*pCtx*/, std::string* pData, const nn::util::string_view& str ) NN_NOEXCEPT
{
    pData->assign( str.data(), str.length() );
}

template< typename T >
inline
void ConverterUtil::ReadArray( ConverterContext* /*pCtx*/, T* pData, const nn::util::string_view& str, int count )
{
    const char* pCur = str.empty() ? "" : str.data();
    int index = 0;
    char* endptr = NULL;
    T value;

    while ( index < count )
    {
        Parse( &value, pCur, &endptr );
        if ( pCur == endptr )
        {
            break;
        }
        pCur = endptr;
        pData[ index++ ] = value;
    }

    if ( index < count )
    {
        // 値が足りなかった場合は例外を投げます。
        // TODO: ContextのStream
        throw "Error: Too little count of values.";
    }
    else if ( nn::util::string_view( pCur ).find_first_not_of( g_Spaces ) != nn::util::string_view::npos )
    {
        // 値が余った場合は例外を投げます。
        // TODO: ContextのStream
        throw "Error: Too much count of values.";
    }
}

template< typename T >
inline
void ConverterUtil::ReadAttribute( ConverterContext* pCtx, T* pTarget, const XmlNode* pNode, const nn::util::string_view& name )
{
    nn::util::string_view attr = pNode->FindAttribute( name );
    if ( attr.data() ) // empty() は長さ 0 の属性が存在する場合も含みます。
    {
        Read( pCtx, pTarget, attr );
    }
    else
    {
        // TODO: ContextのStream
        throw "Error: Attribute not found.";
    }
}

template< typename T, int N >
inline
void ConverterUtil::ReadAttribute(
    ConverterContext* pCtx,
    T* pTarget,
    const XmlNode* pNode,
    const nn::util::string_view& name,
    const nn::util::string_view ( &table )[ N ] )
{
    nn::util::string_view attr;
    ReadAttribute( pCtx, &attr, pNode, name );
    const nn::util::string_view* found = std::find( table.begin(), table.end(), attr );
    if ( found == table.end() )
    {
        // TODO: ContextのStream
        throw "Error: Enum not found";
    }
    *pTarget = static_cast< T >( std::distance( table.begin(), found ) );
}

//! @endcond

}} // namespace nn::utilTool
