﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/cstd/cstd_CStdArg.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/nn_Log.h>

namespace nn {
    namespace utilTool {

        enum CommandLogLevel
        {
            CommandLogLevel_Debug,
            CommandLogLevel_Verbose,
            CommandLogLevel_Message,
            CommandLogLevel_Warning,
            CommandLogLevel_Error,
            CommandLogLevel_Force
        };

        struct CommandLogMetaInfo
        {
            static const int MAX_TAGS = 12;

            CommandLogLevel level;
            const char* fileName;
            int lineNumber;
            int numTags;
            const char* tags[MAX_TAGS];
        };

        struct CommandLogMessage
        {
            const char* format;
            std::va_list* args;
        };

        class ICommandLogObserver : public nn::util::IntrusiveListBaseNode<ICommandLogObserver>
        {
        public:
            virtual ~ICommandLogObserver() {}
            virtual bool NeedLog(CommandLogLevel level) = 0;
            virtual void SendLog(const CommandLogMetaInfo& metaInfo, const CommandLogMessage& message) = 0;
        };

        bool NeedCommandLog(CommandLogLevel level) NN_NOEXCEPT;
        void SendCommandLogV(const CommandLogMetaInfo& logMetaInfo, const char* format, std::va_list formatArg) NN_NOEXCEPT;
        void SendCommandLog(const CommandLogMetaInfo& logMetaInfo, const char* format, ...) NN_NOEXCEPT;
        void RegisterCommandLogObserver(ICommandLogObserver &observer);
        void UnregisterCommandLogObserver(ICommandLogObserver &observer);

        class DefaultCommandLogObserver : public ICommandLogObserver
        {
        public:
            DefaultCommandLogObserver() : m_CurrentLevel(CommandLogLevel_Warning), m_IsEnabledPositionDisplay(false), m_IsEnabledTagDisplay(true) {}
            virtual ~DefaultCommandLogObserver() {}
            virtual bool NeedLog(CommandLogLevel level);
            virtual void SendLog(const CommandLogMetaInfo& metaInfo, const CommandLogMessage& message);
            void SetLevel(CommandLogLevel level);
            void SetPositionDisplay(bool isEnabled);
            void SetTagDisplay(bool isEnabled);

        private:
            CommandLogLevel m_CurrentLevel;
            bool m_IsEnabledPositionDisplay;
            bool m_IsEnabledTagDisplay;
        };
    }
}

#define NN_UTILTOOL_LOG(logLevel, ...) \
    do                                                                  \
    {                                                                   \
        if (::nn::utilTool::NeedCommandLog(logLevel))                   \
        {                                                               \
            ::nn::utilTool::CommandLogMetaInfo logMetaInfo = {};        \
            logMetaInfo.level = (logLevel);                             \
            logMetaInfo.fileName = __FILE__;                            \
            logMetaInfo.lineNumber = __LINE__;                          \
            ::nn::utilTool::SendCommandLog(logMetaInfo, __VA_ARGS__);   \
        }                                                               \
    } while (NN_STATIC_CONDITION(0))

#define NN_UTILTOOL_LOG_TAG1(logLevel, tag1, ...) \
    do                                                                  \
    {                                                                   \
        if (::nn::utilTool::NeedCommandLog(logLevel))                   \
        {                                                               \
            ::nn::utilTool::CommandLogMetaInfo logMetaInfo = {};        \
            logMetaInfo.level = (logLevel);                             \
            logMetaInfo.fileName = __FILE__;                            \
            logMetaInfo.lineNumber = __LINE__;                          \
            logMetaInfo.numTags = 1;                                    \
            logMetaInfo.tags[0] = tag1;                                 \
            ::nn::utilTool::SendCommandLog(logMetaInfo, __VA_ARGS__);   \
        }                                                               \
    } while (NN_STATIC_CONDITION(0))

#define NN_UTILTOOL_LOG_DEBUG(...) NN_UTILTOOL_LOG_TAG1(::nn::utilTool::CommandLogLevel_Debug, "DEBUG", __VA_ARGS__)
#define NN_UTILTOOL_LOG_VERBOSE(...) NN_UTILTOOL_LOG_TAG1(::nn::utilTool::CommandLogLevel_Verbose, "VERBOSE", __VA_ARGS__)
#define NN_UTILTOOL_LOG_PROGRESS(...) NN_UTILTOOL_LOG_TAG1(::nn::utilTool::CommandLogLevel_Verbose, "PROGRESS", __VA_ARGS__)
#define NN_UTILTOOL_LOG_WARNING(...) NN_UTILTOOL_LOG_TAG1(::nn::utilTool::CommandLogLevel_Warning, "WARNING", __VA_ARGS__)
#define NN_UTILTOOL_LOG_MESSAGE(...) NN_UTILTOOL_LOG(::nn::utilTool::CommandLogLevel_Message, __VA_ARGS__)
#define NN_UTILTOOL_LOG_ERROR(...) NN_UTILTOOL_LOG_TAG1(::nn::utilTool::CommandLogLevel_Error, "ERROR", __VA_ARGS__)
#define NN_UTILTOOL_LOG_FATAL(...) NN_UTILTOOL_LOG_TAG1(::nn::utilTool::CommandLogLevel_Error, "FATAL", __VA_ARGS__)
#define NN_UTILTOOL_LOG_FORCE(...) NN_UTILTOOL_LOG_TAG1(::nn::utilTool::CommandLogLevel_Force, "FORCE", __VA_ARGS__)
