﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/utilTool/detail/utilTool_BinarizerContextImpl.h>

namespace nn { namespace utilTool {

//! @brief バイナリコンバータコンテクスト
class BinarizerContext2 : public ConverterContext
{
    NN_DISALLOW_COPY( BinarizerContext2 );
public:

    //! @name 構築/破棄
    //! @{

    //! @brief コンストラクタです。
    BinarizerContext2() NN_NOEXCEPT;

    //! @brief デストラクタです。
    ~BinarizerContext2() NN_NOEXCEPT;

    //! @brief      指定したセクション数でコンテクストを初期化します。
    //! @param[in]  sectionCount    バイナリファイルのセクションの数です。
    void Initialize( int sectionCount ) NN_NOEXCEPT;

    //! @brief コンテクストを再利用できるように再設定します。
    void Reset() NN_NOEXCEPT;

    //! @}

    //! @name 取得
    //! @{

    //! @brief  構築中のバイナリファイルの先頭のポインタを返します。
    //! @return 結果を返します。
    void* GetPtr() NN_NOEXCEPT;

    //! @brief  バイナリファイルのサイズを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref Calculate() の実行後に取得することができます。
    size_t GetSize() const NN_NOEXCEPT;

    //! @brief  ファイルを配置するのに必要なアライメントを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref Calculate() の実行後に取得することができます。
    size_t GetAlignment() const NN_NOEXCEPT;

    //! @}

    //! @name 事前計算
    //! @{

    //! @brief      指定したセクションにメモリブロックを登録します。
    //! @param[in]  sectionIndex    指定するセクションのインデックスです。
    //! @param[in]  pBlock          登録するメモリブロックです。
    //! @details    @ref nn::util::ResDic 用のメモリブロックを登録しないでください。@ref AddResDicMemoryBlock() を使用して下さい。
    void AddMemoryBlock(int sectionIndex, nn::util::MemorySplitter::MemoryBlock* pBlock) NN_NOEXCEPT;

    //! @brief      BinPtr を登録します。
    //! @param[in]  pFrom           BinPtr が存在するメモリブロックのポインタです。
    //! @param[in]  fromOffset      BinPtr が存在するメモリブロックの先頭から BinPtr が存在する位置のオフセットです。
    //! @param[in]  pTarget         BinPtr が指す対象のメンバのメモリブロックのポインタです。
    //! @pre        @ref Calculate() より前に呼び出す必要があります。
    void RegisterBinPtr(const nn::util::MemorySplitter::MemoryBlock* pFrom, ptrdiff_t fromOffset, const nn::util::MemorySplitter::MemoryBlock* pTarget) NN_NOEXCEPT;

    //! @brief      BinPtr を登録します。
    //! @param[in]  pFrom           BinPtr が存在するメモリブロックのポインタです。
    //! @param[in]  fromOffset      BinPtr が存在するメモリブロックの先頭から BinPtr が存在する位置のオフセットです。
    //! @param[in]  pTarget         BinPtr が指す対象のメンバのメモリブロックのポインタです。
    //! @param[in]  targetOffset    BinPtr が指す対象のメンバのメモリブロックの先頭から、そのメンバが存在する位置のオフセットです。
    //! @pre        @ref Calculate() より前に呼び出す必要があります。
    void RegisterBinPtr(const nn::util::MemorySplitter::MemoryBlock* pFrom, ptrdiff_t fromOffset, const nn::util::MemorySplitter::MemoryBlock* pTarget, ptrdiff_t targetOffset) NN_NOEXCEPT;

    //! @brief      バイナリファイルの名前を設定します。
    //! @param[in]  name    バイナリファイルの名前です。文字列は内部でコピーされるため、保持する必要はありません。
    //! @pre        @ref CalculateStringPool() より前に呼び出す必要があります。
    void SetName(const nn::util::string_view& name) NN_NOEXCEPT;

    //! @brief      指定したセクションに nn::util::ResDic の構築に使用するメモリブロックを登録します。
    //! @param[in]  sectionIndex    指定するセクションのインデックスです。
    //! @param[in]  pBlock          登録する @ref nn::util::ResDic 用のメモリブロックです。
    //! @param[in]  pKeys           ResDic のキー文字列配列です。文字列は内部でコピーされるため、保持する必要はありません。
    //! @param[in]  entryCount      ResDic の要素数です。キー文字列配列の要素数と一致する必要があります。
    //! @pre        @ref CalculateStringPool() より前に呼び出す必要があります。
    //! @details    内部で @ref AddMemoryBlock() を行うため、別途 @ref AddMemoryBlock() を呼び出す必要はありません。
    void AddResDicMemoryBlock(int sectionIndex, nn::util::MemorySplitter::MemoryBlock* pBlock, const nn::util::string_view* pKeys, int entryCount) NN_NOEXCEPT;

    //! @brief      BinPtrToString と文字列を登録します。
    //! @param[in]  pFrom       BinPtr が存在するメモリブロックのポインタです。
    //! @param[in]  fromOffset  BinPtr が存在するメモリブロックの先頭から BinPtr が存在する位置のオフセットです。
    //! @param[in]  str         登録する文字列です。
    //! @pre        @ref CalculateStringPool() より前に呼び出す必要があります。
    void RegisterBinPtrToString(const nn::util::MemorySplitter::MemoryBlock* pFrom, ptrdiff_t fromOffset, const nn::util::string_view& str) NN_NOEXCEPT;

    //! @brief      登録済みの文字列を元に文字列プールの変換準備を行います。
    //! @pre        @ref Calculate() より前に呼び出す必要があります。
    //! @return     文字列プールのメモリブロックを返します。このメモリブロックは @ref AddMemoryBlock() で追加する必要があります。
    //! @details    この関数の呼び出し以降、@ref SetName() @ref AddResDicMemoryBlock() @ref RegisterBinPtrToString() を使用することはできません。
    nn::util::MemorySplitter::MemoryBlock* CalculateStringPool() NN_NOEXCEPT;

    //! @brief      指定したセクションのメモリブロックをアライメントの降順に並び替えます。
    //! @param[in]  sectionIndex    指定するセクションのインデックスです。
    void SortByAlignment(int sectionIndex) NN_NOEXCEPT;

    //! @}

    //! @name メモリ領域計算
    //! @{

    //! @brief  登録済みのメモリブロックを元に事前計算を行います。@n
    //!         呼び出し後 @ref GetSize() と @ref GetAlignment() の値を取得することができます。
    void Calculate() NN_NOEXCEPT;

    //! @brief      確保済みのバッファを割り当てます。
    //! @param[in]  buffer      割り当てるバッファです。
    //! @param[in]  bufferSize  割り当てるバッファのサイズです。
    //! @pre    事前に @ref Calculate() を呼び出す必要があります。
    //! @pre    バッファのサイズは @ref Calculate() 後に @ref GetSize() で取得できる値以上である必要があります。
    //! @pre    @ref SetPtr() を複数回実行することはできません。
    void SetPtr(void* buffer, size_t bufferSize) NN_NOEXCEPT;

    //! @}

    //! @name 変換
    //! @{

    //! @brief      バイナリブロックヘッダを追加します。
    //! @param[in]  pHeader 追加するバイナリブロックヘッダです。
    void AddHeader( nn::util::BinaryBlockHeader* pHeader ) NN_NOEXCEPT;

    //! @brief  文字列プール・リロケーションテーブル・各ヘッダの変換を行います。@n
    //!         ファイルヘッダのシグネチャとバージョンは設定されません。
    //! @return ファイルヘッダを返します。
    nn::util::BinaryFileHeader* Convert() NN_NOEXCEPT;

    //! @}

private:

    class PtrStringNode
    {
    public:
        PtrStringNode(detail::MemberPosition from, const nn::util::string_view str)
            : from(from)
            , str(str)
        {
        }

        detail::MemberPosition from;
        const nn::util::string_view str;
    };

    struct ResDicNode
    {
        NN_DISALLOW_COPY( ResDicNode );
    public:
        ResDicNode()
            : pBlock()
            , keyList()
        {
        }

        nn::util::MemorySplitter::MemoryBlock* pBlock;
        std::vector< nn::util::string_view > keyList;
    };

    typedef std::vector< ResDicNode* >   ResDicNodeList;
    typedef std::vector< PtrStringNode > PtrStringNodeList;

    // 登録された ResDic のデータを埋めます。Impl.ConvertBinPtr() の呼び出し後である必要があります。
    void ConvertResDic() NN_NOEXCEPT;

private:
    detail::BinarizerContextImpl    m_Impl;
    ResDicNodeList                  m_ResDicNodeList;
    PtrStringNodeList               m_PtrStringNodeList;
};

}} // namespace nn::utilTool
