﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/utilTool/detail/utilTool_BinarizerContextImpl.h>

namespace nn { namespace utilTool {

//! @brief バイナリコンバータコンテクスト
class BinarizerContext : public ConverterContext
{
    NN_DISALLOW_COPY( BinarizerContext );
public:

    //! @name 構築/破棄
    //! @{

    //! @brief コンストラクタです。
    BinarizerContext() NN_NOEXCEPT;

    //! @brief デストラクタです。
    ~BinarizerContext() NN_NOEXCEPT;

    //! @brief      指定したセクション数でコンテクストを初期化します。
    //! @param[in]  sectionCount    バイナリファイルのセクションの数です。
    void Initialize( int sectionCount ) NN_NOEXCEPT;

    //! @brief コンテクストを再利用できるように再設定します。
    void Reset() NN_NOEXCEPT;

    //! @}

    //! @name 取得
    //! @{

    //! @brief  リロケーションテーブルを除く部分のポインタを取得します。@n
    //!         リロケーションテーブルのポインタは別途取得する必要があります。
    //! @return 結果を返します。
    void* GetBasePtr() NN_NOEXCEPT;

    //! @brief  リロケーションテーブルを除く部分のサイズを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref CalculateBase() か @ref AllocateBase() の実行後に取得することができます。
    size_t GetBaseSize() const NN_NOEXCEPT;

    //! @brief  ファイルを配置するのに必要なアライメントを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref CalculateBase() か @ref AllocateBase() の実行後に取得することができます。
    size_t GetAlignment() const NN_NOEXCEPT;

    //! @brief  リロケーションテーブル部分のポインタを取得します。
    //! @return 結果を返します。
    void* GetRelocationTablePtr() NN_NOEXCEPT;

    //! @brief  リロケーションテーブル部分のサイズを取得します。
    //! @return 結果を返します。
    //! @pre    値は @ref AllocateRelocationTable() か @ref ConvertRelocationTable() の実行後に取得することができます。
    size_t GetRelocationTableSize() const NN_NOEXCEPT;

    //! @}

    //! @name 事前計算
    //! @{

    //! @brief      バイナリファイルの名前を設定します。
    //! @param[in]  name    バイナリファイルの名前です。
    //! @pre        @ref BuildStringPool() より前に呼び出す必要があります。
    //! @details    内部で @ref InsertString() を行うため、別途登録する必要はありません。
    void SetName( const nn::util::string_view& name ) NN_NOEXCEPT;

    //! @brief      文字列プールに文字列を登録します。
    //! @param[in]  str 登録する文字列です。
    //! @details    登録する文字列はコピーされるため、保持する必要はありません。
    void InsertString( const nn::util::string_view& str ) NN_NOEXCEPT;

    //! @brief      指定したセクションにメモリブロックを登録します。
    //! @param[in]  sectionIndex    指定するセクションのインデックスです。
    //! @param[in]  pBlock          登録するメモリブロックです。
    void AddMemoryBlock( int sectionIndex, nn::util::MemorySplitter::MemoryBlock* pBlock ) NN_NOEXCEPT;

    //! @brief  文字列プールを構築して変換準備を行います。
    //! @pre    呼び出し前に、使用する文字列は @ref InsertString() を用いて登録済みである必要があります。
    //! @pre    @ref ConvertString() より前に呼び出す必要があります。
    nn::util::MemorySplitter::MemoryBlock* BuildStringPool() NN_NOEXCEPT;

    //! @brief      指定したセクションのメモリブロックをアライメントの降順に並び替えます。
    //! @param[in]  sectionIndex    指定するセクションのインデックスです。
    void SortByAlignment( int sectionIndex ) NN_NOEXCEPT;

    //! @}

    //! @name 変換
    //! @{

    //! @brief  文字列プールを変換します。
    //! @pre    @ref LinkString() より前に呼び出す必要があります。
    void ConvertStringPool() NN_NOEXCEPT;

    //! @brief      ポインタに文字列プール内の指定文字列への参照を書き込みます。
    //! @param[in]  pBinPtr 参照を書き込むポインタです。
    //! @param[in]  str     指定する文字列です。
    //! @pre        @ref ConvertStringPool() より後に呼び出す必要があります。
    //! @pre        指定する文字列は @ref InsertString() により登録済みである必要があります。
    void LinkString( nn::util::BinPtrToString* pBinPtr, const nn::util::string_view& str ) NN_NOEXCEPT;

    //! @brief      @ref BintTPtr にファイル内領域へのオフセットを書き込みます。
    //! @tparam     T       @ref BintTPtr が指す要素の型です。
    //! @param[in]  pBinPtr オフセットを書き込むポインタです。
    //! @param[in]  ptr     ポインタに書き込む先のアドレスです。
    template< typename T >
    void LinkPtr( nn::util::BinTPtr< T >* pBinPtr, void* ptr ) NN_NOEXCEPT
    {
        void* pBase = m_Impl.GetBasePtr();
        NN_SDK_ASSERT_NOT_NULL( pBase );

        pBinPtr->SetOffset( pBase, ptr ); // ここでセットしておく。Impl.ConvertBinPtr は不要。
        uint32_t position = static_cast< uint32_t >( util::BytePtr( pBase ).Distance( pBinPtr ) );
        uint32_t target = static_cast< uint32_t >( pBinPtr->GetOffset() );
        m_Impl.RegisterBinPtr( detail::MemberPosition( position ), detail::MemberPosition( target ) );
    }

    //! @brief      バイナリブロックヘッダを追加します。
    //! @param[in]  pHeader 追加するバイナリブロックヘッダです。
    void AddHeader( nn::util::BinaryBlockHeader* pHeader ) NN_NOEXCEPT;

    //! @}

    //! @name 事後計算
    //! @{

    //! @brief  リロケーションテーブルを変換します。
    //! @pre    @ref ConvertHeaders() より前に呼び出す必要があります。
    void ConvertRelocationTable() NN_NOEXCEPT;

    //! @brief  ファイルヘッダと各ブロックヘッダを変換します。ファイルヘッダのシグネチャとバージョンは設定されません。
    //! @return 変換済みのバイナリファイルヘッダを返します。
    //! @pre    @ref ConvertRelocationTable() より後に呼び出す必要があります。
    nn::util::BinaryFileHeader* ConvertHeaders() NN_NOEXCEPT;

    //! @}

    //! @name メモリ領域計算
    //! @{

    //! @brief  登録済みのメモリブロックを元に、リロケーションテーブルを除く部分の事前計算を行います。@n
    //!         呼び出し後 @ref GetBaseSize() と @ref GetAlignment() の値を取得することができます。
    void CalculateBase() NN_NOEXCEPT;

    //! @brief  リロケーションテーブルを除く部分の事前計算を行い、バッファを確保します。@n
    //!         @ref CalculateBase() を呼び出す必要はありません。
    //! @pre    @ref SetBasePtr() を利用する場合、この関数を実行することはできません。
    void AllocateBase() NN_NOEXCEPT;

    //! @brief      確保済みのバッファを、リロケーションテーブルを除く部分を配置するバッファに割り当てます。
    //! @param[in]  buffer      割り当てるバッファです。
    //! @param[in]  bufferSize  割り当てるバッファのサイズです。
    //! @pre    事前に @ref CalculateBase() を呼び出す必要があります。
    //! @pre    バッファのサイズは @ref CalculateBase() 後に @ref GetBaseSize() で取得できる値以上である必要があります。
    //! @pre    @ref AllocateBase() を利用する場合、この関数を実行することはできません。
    //! @pre    @ref SetBasePtr() を複数回実行することはできません。
    void SetBasePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT;

    //! @brief リロケーションテーブルのサイズを計算してバッファを確保します。
    //! @pre    @ref SetRelocationTablePtr() を利用する場合、この関数を実行することはできません。
    void AllocateRelocationTable() NN_NOEXCEPT;

    //! @brief      確保済みのバッファを、リロケーションテーブルを配置するバッファに割り当てます。
    //! @param[in]  buffer      割り当てるバッファです。
    //! @param[in]  bufferSize  割り当てるバッファのサイズです。
    //! @pre        必要なバッファのサイズは @ref nn::util::RelocationTable::CalculateSize() で計算することができます。@n
    //!             また、エントリーの数は登録したポインタの数を超えることはありません。
    //! @pre        @ref AllocateRelocationTable() を利用する場合、この関数を実行することはできません。
    //! @pre        @ref SetRelocationTablePtr() を複数回実行することはできません。
    void SetRelocationTablePtr( void* buffer, size_t bufferSize ) NN_NOEXCEPT;

    //! @}

private:
    detail::BinarizerContextImpl m_Impl;
    void* m_pAllocatedBase;
    void* m_pAllocatedRelTable;
};

}} // namespace nn::utilTool
