﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace util {

/*!
    @brief      指定したバイト数の文字列を検査します。

    @param[in]  string      文字列。
    @param[in]  byteLength  文字列のバイト数。

    @return     妥当な文字列かどうか。

    @pre
        - string != nullptr
        - byteLength > 0

    @details
                本関数は、途中で終端文字を検知しても指定したバイト数分の検査を行います。
*/
bool VerifyUtf8String(const char* string, size_t byteLength) NN_NOEXCEPT;

/*!
    @brief      指定したバイト数の文字列からコードポイント数を取得します。

    @param[in]  string      文字列。
    @param[in]  byteLength  文字列のバイト数。

    @return     コードポイント数。

    @pre
        - string != nullptr
        - byteLength > 0

    @details
                本関数は、途中で終端文字を検知しても指定したバイト数分の検査を行います。@n
                終端文字もカウントに含まれるので注意してください。

                妥当な文字列を指定しなかった場合、 -1 が返ります。
*/
int GetCodePointCountOfUtf8String(const char* string, size_t byteLength) NN_NOEXCEPT;

/*!
    @brief      バッファに収まる範囲内で、指定した最大コードポイント数分の文字列をコピーします。

    @param[out] output          コピー先のバッファ。
    @param[in]  outputSize      コピー先のバッファサイズ。
    @param[in]  input           コピー元。
    @param[in]  inputByteLength コピー元のバイト数。
    @param[in]  maxCount        コピーする最大コードポイント数。

    @return     コピーしたバイト数。

    @pre
        - output != nullptr
        - outputSize > 0
        - input != nullptr
        - inputByteLength > 0
        - maxCount > 0

    @details
                本関数は、正しいシーケンスと認識できるところまでコピーします。@n
                出力した文字列は、終端文字で終端していることを保証します。

                途中で終端文字を検知しても指定したバイト数分の検査・コピーを行います。@n
                終端文字以降もコピーされるので注意してください。

                例：{'A', '\0', '\xE3', '\x81', '\x82', '\xE3', '\x81'}; // 「A」＋「終端文字」＋「あ」＋「い」の 2 バイト目まで。

                上記の例は途中に終端文字を含み、末尾が途切れている文字列ですが、5 バイトはコピーされます。
*/
size_t CopyUtf8String(char* output, size_t outputSize, const char* input, size_t inputByteLength, int maxCount) NN_NOEXCEPT;

}}
