﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_InPlace.h> // in_place_t
#include <type_traits>
#include <utility> // std::forward
#include <new> // placement new
#include <memory> // std::addressof
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_MacroForVariadic.h>

namespace nn { namespace util {

namespace detail {

template <typename Error, typename T>
class OptionalObjectHolderBase
{
    static_assert(std::is_trivially_destructible<Error>::value, "");
    static_assert(std::is_destructible<T>::value, "");
public:
    void* GetObjectAddress() NN_NOEXCEPT
    {
        return &m_Storage;
    }
    const void* GetObjectAddress() const NN_NOEXCEPT
    {
        return &m_Storage;
    }
    Error& RefError() NN_NOEXCEPT
    {
        return m_Error;
    }
    Error GetError() const NN_NOEXCEPT
    {
        return m_Error;
    }
private:
    Error m_Error;
    typename std::aligned_storage<sizeof(T), std::alignment_of<T>::value>::type m_Storage;
};

template <typename Error, typename T, typename Enabled = void>
struct OptionalObjectHolder
    : public OptionalObjectHolderBase<Error, T>
{
public:
    void DestroyImpl() NN_NOEXCEPT
    {
        if (this->GetError().IsSuccess())
        {
            reinterpret_cast<T*>(this->GetObjectAddress())->T::~T();
        }
    }

    ~OptionalObjectHolder() NN_NOEXCEPT
    {
        DestroyImpl();
    }
};

template <typename Error, typename T>
struct OptionalObjectHolder<Error, T, typename std::enable_if<std::is_trivially_destructible<T>::value>::type>
     : public OptionalObjectHolderBase<Error, T>
{
public:
    void DestroyImpl() NN_NOEXCEPT {}
};

template <typename Error, typename T>
class OptionalBase
{
public:

    OptionalBase() {}

    explicit OptionalBase(const T& x)
    {
        Construct(x);
    }

    explicit OptionalBase(T&& x)
    {
        Construct(std::move(x));
    }

    explicit OptionalBase(in_place_t)
    {
        Construct();
    }

    explicit OptionalBase(const Error& errorValue)
    {
        OverwriteError(errorValue);
    }

    OptionalBase(const OptionalBase& x)
    {
        if (x.IsEngaged())
        {
            Construct(x.GetObject());
        }
        else
        {
            OverwriteError(x);
        }
    }
    OptionalBase(OptionalBase&& x)
    {
        if (x.IsEngaged())
        {
            Construct(std::move(x.GetObject()));
        }
        else
        {
            OverwriteError(x);
        }
    }

    #define NN_UTIL_OPTIONAL_DETAIL_OPTIONAL_BASE_DEFINE_IN_PLACE_CONSTRUCTOR(n) \
        template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (T)> \
        OptionalBase(in_place_t, NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (T, x)) { Construct(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (T, x)); }

        NN_UTIL_VARIADIC_DEFINE_MACROS(NN_UTIL_OPTIONAL_DETAIL_OPTIONAL_BASE_DEFINE_IN_PLACE_CONSTRUCTOR)

    #undef NN_UTIL_OPTIONAL_DETAIL_OPTIONAL_BASE_DEFINE_IN_PLACE_CONSTRUCTOR

    bool IsEngaged() const NN_NOEXCEPT
    {
        return GetError().IsSuccess();
    }

    void AssignError() NN_NOEXCEPT
    {
        Destroy();
    }

    void Assign(const OptionalBase& rhs)
    {
        if (IsEngaged())
        {
            if (rhs.IsEngaged())
            {
                this->GetObject() = rhs.GetObject();
            }
            else
            {
                Destroy();
                this->OverwriteError(rhs);
            }
        }
        else
        {
            if (rhs.IsEngaged())
            {
                Construct(rhs.GetObject());
            }
            else
            {
                this->OverwriteError(rhs);
            }
        }
    }

    void Assign(OptionalBase&& rhs)
    {
        if (IsEngaged())
        {
            if (rhs.IsEngaged())
            {
                this->GetObject() = std::move(rhs.GetObject());
            }
            else
            {
                Destroy();
                this->OverwriteError(rhs);
            }
        }
        else
        {
            if (rhs.IsEngaged())
            {
                Construct(std::move(rhs.GetObject()));
            }
            else
            {
                this->OverwriteError(rhs);
            }
        }
    }

    void AssignValue(const T& value)
    {
        if (IsEngaged())
        {
            this->GetObject() = value;
        }
        else
        {
            Construct(value);
        }
    }

    void AssignValue(T&& value)
    {
        if (IsEngaged())
        {
            this->GetObject() = std::move(value);
        }
        else
        {
            Construct(std::move(value));
        }
    }

    void Emplace()
    {
        Destroy();
        Construct();
    }

    #define NN_UTIL_OPTIONAL_DETAIL_BASE_DEFINE_EMPLACE(n) \
        template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (T)> \
        void Emplace(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (T, x)) \
        { \
            Destroy(); \
            Construct(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (T, x)); \
        }

        NN_UTIL_VARIADIC_DEFINE_MACROS(NN_UTIL_OPTIONAL_DETAIL_BASE_DEFINE_EMPLACE)

    #undef NN_UTIL_OPTIONAL_DETAIL_BASE_DEFINE_EMPLACE

    void Swap(OptionalBase& other)
    {
        if (this->IsEngaged())
        {
            if (other.IsEngaged())
            {
                using std::swap;
                swap(this->GetObject(), other.GetObject());
            }
            else
            {
                auto errorValue(other.GetError());
                other.Construct(std::move(this->GetObject()));
                this->Destroy();
                this->OverwriteError(errorValue);
            }
        }
        else
        {
            if (other.IsEngaged())
            {
                auto errorValue(this->GetError());
                this->Construct(std::move(other.GetObject()));
                other.Destroy();
                other.OverwriteError(errorValue);
            }
            else
            {
                this->SwapError(other);
            }
        }
    }

    T& GetObject() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(this->IsEngaged());
        return *reinterpret_cast<T*>(m_Holder.GetObjectAddress());
    }

    const T& GetObject() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(this->IsEngaged());
        return *reinterpret_cast<const T*>(m_Holder.GetObjectAddress());
    }

    T* GetObjectPointer() NN_NOEXCEPT
    {
        return std::addressof(GetObject());
    }
    const T* GetObjectPointer() const NN_NOEXCEPT
    {
        return std::addressof(GetObject());
    }

    Error GetError() const
    {
        return m_Holder.GetError();
    }

    template <class U>
    T ValueOr(U&& value) const
    {
        if (this->IsEngaged())
        {
            return this->GetObject();
        }
        else
        {
            return static_cast<T>(std::forward<U&&>(value));
        }
    }

private:

    typedef OptionalObjectHolder<Error, T> Holder;
    Holder m_Holder;
    static_assert(std::is_trivially_destructible<T>::value == std::is_trivially_destructible<Holder>::value, "");

    void Construct()
    {
        NN_SDK_ASSERT(!IsEngaged());
        new (m_Holder.GetObjectAddress()) T();
        m_Holder.RefError().SetSuccess();
        NN_SDK_REQUIRES(IsEngaged());
    }

    #define NN_UTIL_OPTIONAL_DETAIL_OPTIONAL_BASE_DEFINE_CONSTRUCT(n) \
        template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (T)> \
        void Construct(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (T, x)) \
        { \
            NN_SDK_ASSERT(!IsEngaged()); \
            new (m_Holder.GetObjectAddress()) T(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (T, x)); \
            m_Holder.RefError().SetSuccess(); \
            NN_SDK_REQUIRES(IsEngaged()); \
        }

        NN_UTIL_VARIADIC_DEFINE_MACROS(NN_UTIL_OPTIONAL_DETAIL_OPTIONAL_BASE_DEFINE_CONSTRUCT)

    #undef NN_UTIL_OPTIONAL_DETAIL_OPTIONAL_BASE_DEFINE_CONSTRUCT

    void Destroy() NN_NOEXCEPT
    {
        m_Holder.DestroyImpl();
        m_Holder.RefError().SetInvalid();
        NN_SDK_REQUIRES(!IsEngaged());
    }

    void OverwriteError(const Error& other)
    {
        NN_SDK_ASSERT(!other.IsSuccess());
        NN_SDK_ASSERT(!GetError().IsSuccess());
        m_Holder.RefError().OverwriteError(other);
    }

    void OverwriteError(const OptionalBase& other)
    {
        OverwriteError(other.GetError());
    }

    void SwapError(OptionalBase& other)
    {
        NN_SDK_ASSERT(!other.GetError().IsSuccess());
        NN_SDK_ASSERT(!GetError().IsSuccess());
        m_Holder.RefError().SwapError(other.m_Holder.RefError());
    }
};

} // namespace detail

}}
