﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_OptionalBase.h>
#include <type_traits>
#include <utility>
#include <nn/util/util_InPlace.h> // in_place_t
#include <nn/util/util_MacroForVariadic.h>

namespace nn { namespace util {

namespace detail {

struct NulloptHelper
{
};

}

//! @{
//! @name optional 関連型

/**
    @brief 無効な optional であることを表す型です。
*/
typedef nn::util::detail::NulloptHelper* nullopt_t;

//! @}

//! @{
//! @name optional 関連定数

/**
    @brief nullopt_t のインスタンスです。
*/
const nullopt_t nullopt = nullptr;

//! @}

NN_PRAGMA_PUSH_WARNINGS
NN_DISABLE_WARNING_SHADOW

/**
    @brief 無効状態を持つことのできる、オブジェクトを保持するためのクラスです。

    @tparam T 内部オブジェクトの型を表します。
*/
template <typename T>
class optional
{
    static_assert(!std::is_same<typename std::decay<T>::type, nullopt_t>::value, "T must not be nullopt_t");
    static_assert(!std::is_same<typename std::decay<T>::type, in_place_t>::value, "T must not be in_place_t");
    static_assert(!std::is_reference<T>::value, "not implemented when T is reference");

public:

    /**
        @brief 無効状態で optional を構築するコンストラクタです。
    */
    optional() NN_NOEXCEPT {}

    /**
        @brief 無効状態で optional を構築するコンストラクタです。
        @param nullopt nullopt を指定します。
    */
    NN_IMPLICIT optional(nullopt_t nullopt) NN_NOEXCEPT
    {
        NN_UNUSED(nullopt);
    }

    /**
        @brief コピーコンストラクタです。
        @param other コピー元の optional オブジェクトです。
    */
    optional(const optional& other) : m_Base(other.m_Base) {}

    /**
        @brief ムーブコンストラクタです。
        @param other ムーブ元の optional オブジェクトです。
    */
    optional(optional&& other) : m_Base(std::move(other.m_Base)) {}

    /**
        @brief 指定したオブジェクトのコピーによって optional を構築するコンストラクタです。
        @param x コピー元となるオブジェクトです。
    */
    NN_IMPLICIT optional(const T& x) : m_Base(x) {}

    /**
        @brief 指定したオブジェクトからムーブして optional を構築するコンストラクタです。
        @param x ムーブ元となるオブジェクトです。
    */
    NN_IMPLICIT optional(T&& x) : m_Base(std::move(x)) {}

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
    /**
        @brief 指定した引数を内部オブジェクトのコンストラクタに転送し、直接構築するコンストラクタです。
        @tparam Args 内部オブジェクトのコンストラクタの引数の型を表します。
        @param in_place 直接構築を表すタグです。
        @param args 内部オブジェクトのコンストラクタに渡す引数です。
    */
    template <typename... Args>
    explicit optional(in_place_t in_place, Args... args);
#else
    explicit optional(in_place_t in_place) : m_Base(in_place) {}

    #define NN_UTIL_OPTIONAL_DEFINE_IN_PLACE_CONSTRUCTOR(n) \
        template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (T)> \
        optional(in_place_t in_place, NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (T, x)) : m_Base(in_place, NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (T, x)) {}

        NN_UTIL_VARIADIC_DEFINE_MACROS(NN_UTIL_OPTIONAL_DEFINE_IN_PLACE_CONSTRUCTOR)

    #undef NN_UTIL_OPTIONAL_DEFINE_IN_PLACE_CONSTRUCTOR
#endif

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
    /**
        @brief デストラクタです。
    */
    ~optional() NN_NOEXCEPT = default;
#endif

    /**
        @brief optional を無効状態にする代入演算子です。
        @param nullopt nullopt を指定します。
        @return *this を返します。
    */
    optional& operator=(nullopt_t nullopt) NN_NOEXCEPT
    {
        NN_UNUSED(nullopt);
        m_Base.AssignError();
        return *this;
    }

    /**
        @brief 指定されたオブジェクトを内部オブジェクトに再代入する代入演算子です。
        @param rhs 代入元のオブジェクトです。
        @return *this を返します。
    */
    optional& operator=(const optional& rhs)
    {
        m_Base.Assign(rhs.m_Base);
        return *this;
    }

    /**
        @brief 指定されたオブジェクトを内部オブジェクトにムーブする代入演算子です。
        @param rhs ムーブ元のオブジェクトです。
        @return *this を返します。
    */
    optional& operator=(optional&& rhs)
    {
        m_Base.Assign(std::move(rhs.m_Base));
        return *this;
    }

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
    /**
        @brief 指定されたオブジェクトを内部オブジェクトに転送する代入演算子です。
        @param rhs 代入元のオブジェクトです。
        @return *this を返します。
    */
    template <typename U> optional& operator=(U&& rhs);
#else
    template <typename U>
    typename std::enable_if<std::is_same<typename std::remove_reference<U>::type, T>::value, optional&>::type operator=(U&& value)
    {
        m_Base.AssignValue(std::forward<U&&>(value));
        return *this;
    }
#endif

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
    /**
        @brief 内部オブジェクトのコンストラクタに与えられた引数を渡して再構築します。
        @tparam Args 内部オブジェクトのコンストラクタの引数の型を表します。
        @param args 内部オブジェクトのコンストラクタに渡す引数です。
    */
    template <typename... Args>
    void emplace(Args... args);
#else
    void emplace()
    {
        m_Base.Emplace();
    }

    #define NN_UTIL_OPTIONAL_DEFINE_EMPLACE(n) \
        template <NN_UTIL_VARIADIC_TEMPLATE_TEMPLATE_ARGUMENTS_##n (T)> \
        void emplace(NN_UTIL_VARIADIC_TEMPLATE_ARGUMENT_LIST_##n (T, x)) { m_Base.Emplace(NN_UTIL_VARIADIC_TEMPLATE_FORWARD_LIST_##n (T, x)); }

        NN_UTIL_VARIADIC_DEFINE_MACROS(NN_UTIL_OPTIONAL_DEFINE_EMPLACE)

    #undef NN_UTIL_OPTIONAL_DEFINE_EMPLACE
#endif

    /**
        @brief 対象のオブジェクトと値を交換します。
        @param other 交換対象のオブジェクトを表します。
    */
    void swap(optional& other)
    {
        m_Base.Swap(other.m_Base);
    }

    /**
        @brief 内部オブジェクトが有効であるかどうかを返します。
        @return 内部オブジェクトが有効なときは true を、無効状態のときは false を返します。
    */
    NN_EXPLICIT_OPERATOR bool() const NN_NOEXCEPT
    {
        return m_Base.IsEngaged();
    }

    /**
        @brief 内部オブジェクトへの参照を返します。
        @return 内部オブジェクトへの参照を返します。
    */
    T& operator*() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(*this);
        return m_Base.GetObject();
    }

    /**
        @brief 内部オブジェクトへの参照を返します。
        @return 内部オブジェクトへの参照を返します。
    */
    const T& operator*() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(*this);
        return m_Base.GetObject();
    }

    /**
        @brief 内部オブジェクトへのポインタを返します。
        @return 内部オブジェクトへのポインタを返します。
    */
    T* operator->() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(*this);
        return m_Base.GetObjectPointer();
    }

    /**
        @brief 内部オブジェクトへのポインタを返します。
        @return 内部オブジェクトへのポインタを返します。
    */
    const T* operator->() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(*this);
        return m_Base.GetObjectPointer();
    }

    /**
        @brief 内部オブジェクトへの参照を返します。
        @return 内部オブジェクトへの参照を返します。
    */
    T& value() NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(*this);
        return m_Base.GetObject();
    }

    /**
        @brief 内部オブジェクトへの参照を返します。
        @return 内部オブジェクトへの参照を返します。
    */
    const T& value() const NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(*this);
        return m_Base.GetObject();
    }

    /**
        @brief 内部オブジェクトが有効であれば内部オブジェクトを、無効状態のときは与えられた引数を返します。
        @tparam U 無効状態のときに取る値の型
        @param value 無効状態のときに返す値です。
        @return 内部オブジェクトが有効であれば static_cast<T>(*this) を返し、無効状態のときは value を返します。
    */
    template <class U>
    T value_or(U&& value) const
    {
        return m_Base.ValueOr(std::forward<U&&>(value));
    }

private:
    struct BoolHolder
    {
    private:
        bool m_Success;
    public:
        BoolHolder() NN_NOEXCEPT : m_Success(false) {}
        void SetSuccess() NN_NOEXCEPT
        {
            this->m_Success = true;
        }
        void SetInvalid() NN_NOEXCEPT
        {
            this->m_Success = false;
        }
        bool IsSuccess() const NN_NOEXCEPT
        {
            return m_Success;
        }
        void OverwriteError(const BoolHolder&) NN_NOEXCEPT {}
        void SwapError(BoolHolder&) NN_NOEXCEPT {}
    };
    typename nn::util::detail::OptionalBase<BoolHolder, T> m_Base;
    static_assert(std::is_trivially_destructible<T>::value == std::is_trivially_destructible<decltype(m_Base)>::value, "");
};

NN_PRAGMA_POP_WARNINGS

//! @{
//! @name optional 関連関数

/**
    @brief optional の値を入れ替えます。
    @tparam T optional のテンプレート引数です。
    @param x 入れ替える一つ目の参照です。
    @param y 入れ替える二つ目の参照です。
*/
template <typename T>
inline void swap(optional<T>& x, optional<T>& y)
{
    x.swap(y);
}

/**
    @brief optional を作成するためのユーティリティ関数です。
    @tparam T optional のテンプレート引数です。
    @param value 内部オブジェクトの初期化に使用する値です。
    @return 作成された optional オブジェクトを返します。
*/
template <typename T>
inline optional<typename std::decay<T>::type> make_optional(T&& value)
{
    return optional<typename std::decay<T>::type>(std::forward<T&&>(value));
}

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

/**
    @brief optional 同士の等値演算子です。同様に、operator!= も定義されます。
    @tparam T optional のテンプレート引数です。
    @param x 比較の一つ目の引数です。
    @param y 比較の二つ目の引数です。
    @return 比較の結果を返します。
*/
template <typename T>
bool operator==(const optional<T>& x, const optional<T>& y);

/**
    @brief optional 同士の比較演算子です。同様に、operator>, operator<=, operator>= も定義されます。
    @tparam T optional のテンプレート引数です。
    @param x 比較の一つ目の引数です。
    @param y 比較の二つ目の引数です。
    @return 比較の結果を返します。
*/
template <typename T>
bool operator<(const optional<T>& x, const optional<T>& y);

/**
    @brief optional と nullopt_t との間の等値演算子です。同様に、逆順のものや operator!= も定義されます。
    @tparam T optional のテンプレート引数です。
    @param x 比較の一つ目の引数です。
    @param nullopt 比較の二つ目の引数です。
    @return 比較の結果を返します。
*/
template <typename T>
bool operator==(const optional<T>& x, nullopt_t nullopt) NN_NOEXCEPT;

/**
    @brief optional と nullopt_t との間の比較演算子です。同様に、逆順のものや operator>, operator<=, operator>= も定義されます。
    @tparam T optional のテンプレート引数です。
    @param x 比較の一つ目の引数です。
    @param nullopt 比較の二つ目の引数です。
    @return 比較の結果を返します。
*/
template <typename T>
bool operator<(const optional<T>& x, nullopt_t nullopt) NN_NOEXCEPT;

/**
    @brief optional と内部オブジェクト型との間の等値演算子です。同様に、逆順のものや operator!= も定義されます。
    @tparam T optional のテンプレート引数です。
    @param x 比較の一つ目の引数です。
    @param y 比較の二つ目の引数です。
    @return 比較の結果を返します。
*/
template <typename T>
bool operator==(const T& x, const optional<T>& y);

/**
    @brief optional と内部オブジェクト型との間の比較演算子です。同様に、逆順のものや operator>, operator<=, operator>= も定義されます。
    @tparam T optional のテンプレート引数です。
    @param x 比較の一つ目の引数です。
    @param y 比較の二つ目の引数です。
    @return 比較の結果を返します。
*/
template <typename T>
bool operator<(const T& x, const optional<T>& y);

#else

template <typename T>
inline bool operator==(const optional<T>& x, const optional<T>& y)
{
    return x
        ? (static_cast<bool>(y) && *x == *y)
        : !y;
}

template <typename T>
inline bool operator!=(const optional<T>& x, const optional<T>& y)
{
    return !(x == y);
}

template <typename T>
inline bool operator<(const optional<T>& x, const optional<T>& y)
{
    return x
        ? (static_cast<bool>(y) && *x < *y)
        : static_cast<bool>(y);
}

template <typename T>
inline bool operator>(const optional<T>& x, const optional<T>& y)
{
    return y < x;
}

template <typename T>
inline bool operator<=(const optional<T>& x, const optional<T>& y)
{
    return !(y < x);
}

template <typename T>
inline bool operator>=(const optional<T>& x, const optional<T>& y)
{
    return !(x < y);
}

#define NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR(op, nulloptOpValue, nulloptOpNullopt, reverseOp) \
    template <typename T> \
    inline bool operator op(const optional<T>& x, const T& v) \
    { \
        return static_cast<bool>(x) ? *x op v : nulloptOpValue; \
    } \
    template <typename T> \
    inline bool operator op(nullopt_t, const optional<T>& x) NN_NOEXCEPT \
    { \
        return static_cast<bool>(x) ? nulloptOpValue : nulloptOpNullopt; \
    } \
    template <typename T> \
    inline bool operator reverseOp(const T& v, const optional<T>& x) \
    { \
        return x op v; \
    } \
    template <typename T> \
    inline bool operator reverseOp(const optional<T>& x, nullopt_t) NN_NOEXCEPT \
    { \
        return nullopt op x; \
    }

    NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR(==, false, true,  ==)
    NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR(!=, true,  false, !=)
    NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR(<,  true,  false, > )
    NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR(>,  false, false, < )
    NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR(<=, true,  true,  >=)
    NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR(>=, false, true,  <=)

#undef NN_UTIL_DETAIL_OPTIONAL_DEFINE_COMPARISON_OPERATOR

#endif // NN_BUILD_FOR_DOCUMENT_GENERATION

//! @}

}}
