﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/util/util_Config.h>

/**
* @file
* @brief    数学関数で利用する型を定義します。
*/

namespace nn { namespace util {

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(push)
#pragma warning(disable:4201)
#endif

/**
* @brief    1 円周を 0x100000000 とするインデックスによって角度を表現します。
*/
typedef uint32_t AngleIndex;

/**
* @brief    8バイト符号なし整数型による 4 要素 1 次元配列を持つ構造体です。
*/
struct Uint8x4
{
    uint8_t v[4];     //!< 要素配列
};

/**
* @brief    正規化された、8バイト符号なし整数型による 4 要素 1 次元配列を持つ構造体です。
*/
struct Unorm8x4
{
    uint8_t v[4];     //!< 要素配列
};

/**
* @brief    単精度浮動小数点型の配列によって 2 次元ベクトルを表現します。
*/
struct Float2
{
    union
    {
        float v[2];     //!< 要素配列
        struct
        {
            float x;
            float y;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 3 次元ベクトルを表現します。
*/
struct Float3
{
    union
    {
        float v[3];     //!< 要素配列
        struct
        {
            float x;
            float y;
            float z;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 4 次元ベクトルを表現します。
*/
struct Float4
{
    union
    {
        float v[4];     //!< 要素配列
        struct
        {
            float x;
            float y;
            float z;
            float w;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 3 行 2 列の行列を行優先で表現します。
*/
struct FloatRowMajor3x2
{
    union
    {
        float m[3][2];  //!< 要素配列
        struct
        {
            float m00;
            float m01;
            float m10;
            float m11;
            float m20;
            float m21;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 4 行 3 列の行列を行優先で表現します。
*/
struct FloatRowMajor4x3
{
    union
    {
        float m[4][3];  //!< 要素配列
        struct
        {
            float m00;
            float m01;
            float m02;
            float m10;
            float m11;
            float m12;
            float m20;
            float m21;
            float m22;
            float m30;
            float m31;
            float m32;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 3 行 2 列の行列を列優先で表現します。
*/
struct FloatColumnMajor3x2
{
    union
    {
        float m[2][3];  //!< 要素配列
        struct
        {
            float m00;
            float m10;
            float m20;
            float m01;
            float m11;
            float m21;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 4 行 3 列の行列を列優先で表現します。
*/
struct FloatColumnMajor4x3
{
    union
    {
        float m[3][4];  //!< 要素配列
        struct
        {
            float m00;
            float m10;
            float m20;
            float m30;
            float m01;
            float m11;
            float m21;
            float m31;
            float m02;
            float m12;
            float m22;
            float m32;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 4 行 4 列の行列を行優先で表現します。
*/
struct FloatRowMajor4x4
{
    union
    {
        float m[4][4];  //!< 要素配列
        struct
        {
            float m00;
            float m01;
            float m02;
            float m03;
            float m10;
            float m11;
            float m12;
            float m13;
            float m20;
            float m21;
            float m22;
            float m23;
            float m30;
            float m31;
            float m32;
            float m33;
        };
    };
};

/**
* @brief    単精度浮動小数点型の配列によって 4 行 4 列の行列を列優先で表現します。
*/
struct FloatColumnMajor4x4
{
    union
    {
        float m[4][4];  //!< 要素配列
        struct
        {
            float m00;
            float m10;
            float m20;
            float m30;
            float m01;
            float m11;
            float m21;
            float m31;
            float m02;
            float m12;
            float m22;
            float m32;
            float m03;
            float m13;
            float m23;
            float m33;
        };
    };
};

#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)

/**
* @brief    単精度浮動小数点型による 2 次元ベクトルを表現します。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：8 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct Vector2fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 3 次元ベクトルを表現します。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct Vector3fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 4 次元ベクトルを表現します。また、クォータニオンを表現する場合にも用いられます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  ベクトル以外にも、nn::util::QuaternionAdd などのクォータニオンを扱う関数において、クォータニオンを表現するためにも用いられます。
*  ベクトルの要素を先頭から順に x, y, z, w、クォータニオンの虚数成分を i, j, k としたとき、クォータニオン q は、q = xi + yj + zk + w と表されます。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @details  ベクトル・行列計算を提供する型です。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct Vector4fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 3 行 2 列の行列を行優先で表現します。
*           nn::util::MatrixColumnMajor3x2fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：8 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct MatrixRowMajor3x2fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 4 行 3 列の行列を行優先で表現します。
*           nn::util::MatrixColumnMajor4x3fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct MatrixRowMajor4x3fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 4 行 4 列の行列を行優先で表現します。
*           nn::util::MatrixColumnMajor4x4fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct MatrixRowMajor4x4fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 3 行 2 列の行列を列優先で表現します。
*           nn::util::MatrixRowMajor3x2fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct MatrixColumnMajor3x2fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 4 行 3 列の行列を列優先で表現します。
*           nn::util::MatrixRowMajor4x3fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct MatrixColumnMajor4x3fType
{
    // メンバは実装によって異なります。
};

/**
* @brief    単精度浮動小数点型による 4 行 4 列の行列を列優先で表現します。
*           nn::util::MatrixRowMajor4x4fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
struct MatrixColumnMajor4x4fType
{
    // メンバは実装によって異なります。
};

#endif

#if defined(NN_BUILD_CONFIG_OS_WIN)
#pragma warning(pop)
#endif

}}

// 環境に応じて実装を切り替えます。
#if NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_GENERAL
    #include <nn/util/detail/util_MathTypes.general.h>

    namespace nn { namespace util {
        using namespace nn::util::general;
    }}

#elif NN_UTIL_CONFIG_VECTOR_MATRIX_TARGET_IS_NEON
    #include <nn/util/detail/util_MathTypes.neon.h>

    namespace nn { namespace util {
        using namespace nn::util::neon;
    }}

#else
    #error   "未サポートのターゲットが指定されています。"
#endif

namespace nn { namespace util {

// 4 要素 1 次元配列を持つ構造体の別名

/**
* @brief  4 要素各 8 ビットの符号なし整数によってカラーを表現します。
*/
typedef Unorm8x4 Color4u8Type;

/**
* @brief  4 要素各 32 ビットの浮動小数点数によってカラーを表現します。
*/
typedef Vector4fType Color4fType;


// ベクトル・行列構造体の別名

/**
* @brief    単精度浮動小数点型の配列によって 3 行 2 列の行列を表現します。
*/
typedef FloatRowMajor3x2 Float3x2;

/**
* @brief    単精度浮動小数点型の配列によって 4 行 3 列の行列を表現します。
*/
typedef FloatRowMajor4x3 Float4x3;

/**
* @brief    単精度浮動小数点型の配列によって 4 行 4 列の行列を表現します。
*/
typedef FloatRowMajor4x4 Float4x4;

/**
* @brief    単精度浮動小数点型による 3 行 2 列の行列を表現します。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：8 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
typedef MatrixRowMajor3x2fType Matrix3x2fType;

/**
* @brief    単精度浮動小数点型による 4 行 3 列の行列を表現します。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
typedef MatrixRowMajor4x3fType Matrix4x3fType;

/**
* @brief    単精度浮動小数点型による 4 行 4 列の行列を表現します。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
typedef MatrixRowMajor4x4fType Matrix4x4fType;

/**
* @brief    単精度浮動小数点型の配列によって 3 行 2 列の行列を表現します。
*           nn::util::Float3x2 に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*/
typedef FloatColumnMajor3x2 FloatT3x2;

/**
* @brief    単精度浮動小数点型の配列によって 4 行 3 列の行列を表現します。
*           nn::util::Float4x3 に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*/
typedef FloatColumnMajor4x3 FloatT4x3;

/**
* @brief    単精度浮動小数点型の配列によって 4 行 3 列の行列を表現します。
*           nn::util::Float4x4 に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*/
typedef FloatColumnMajor4x4 FloatT4x4;

/**
* @brief    単精度浮動小数点型による 3 行 2 列の行列を表現します。
*           nn::util::Matrix3x2fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
typedef MatrixColumnMajor3x2fType MatrixT3x2fType;

/**
* @brief    単精度浮動小数点型による 4 行 3 列の行列を表現します。
*           nn::util::Matrix4x3fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
typedef MatrixColumnMajor4x3fType MatrixT4x3fType;

/**
* @brief    単精度浮動小数点型による 4 行 4 列の行列を表現します。
*           nn::util::Matrix4x4fType に対し、行と列の優先度が逆転されたメモリレイアウトを持ちます。
*
* @details  ベクトル・行列計算を提供する型です。
*
*  この型を用いた計算は、SIMD ユニットを使える環境においては、SIMD ユニットを利用した実装に切り替わることを想定しています。
*  SIMD ユニットを使えない環境では、汎用の実装を提供します。
*
* @attention
*  この型は、計算のために最適化されています。このため、メモリレイアウトもサイズも環境によって
*  異なる場合があります。よって、 @b 内部要素への直接のアクセスは禁止します 。
*  要素へのアクセスは、専用のマクロ、および、関数を利用してください。
*
* @attention
*  また、SIMD ユニットでの計算を最適化するため、この型はメモリアライメントを要求します。
*  @b この型をメモリ上に確保する場合には、要求されるアライメントを満たすよう注意してください。
*  要求するアライメントは内部実装ごとに異なる場合があります。それぞれの実装が要求するアライメントを列挙します。
*  - 汎用実装版：16 byte
*  - NEON 実装版：16 byte
*  .
*  汎用実装版は、他のすべての SIMD 実装版が要求するアライメントを満たすことができるようにアライメントを指定してします。
*/
typedef MatrixColumnMajor4x4fType MatrixT4x4fType;

}}
