﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief Main API for the EDID library.
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/edid/edid_Error.h>

namespace nn { namespace edid {

    struct Edid;

    /**
     * @brief Opens an EDID for use with the library.
     *
     * @param[out] pOutEdid EDID handle.
     * @param[in]  pData    Raw data.
     * @param[in]  size     Size (in bytes) of @a pData.
     *
     * @return                   Whether the EDID was opened successfully.
     * @retval Error_None        The EDID was verified successfully.
     * @retval Error_InvalidSize The EDID size is not standard.
     * @retval Error_Corrupt     EDID checksums are invalid.
     *
     * @pre
     *       - @a pOutEdid != nullptr
     *       - @a pData != nullptr
     * @post @a pOutEdid may contain valid handle information.
     *
     * @details If the EDID is validated successfully, then a shallow handle is
     *          created.  @a pData must remain valid for all operations in the library!
     */
    Error OpenEdid(Edid* pOutEdid, const void* pData, size_t size) NN_NOEXCEPT;

    /**
     * @brief Closes an EDID handle in use by the library.
     *
     * @param[in] pEdid EDID handle.
     *
     * @pre
     *       - @a pEdid != nullptr
     * @post @a pEdid is invalidated.
     */
    void CloseEdid(Edid* pEdid) NN_NOEXCEPT;

    /**
     * @brief Prints the EDID in human-readable format to standard out.
     *
     * @param[in] pEdid EDID handle created from OpenEdid().
     *
     * @pre
     *       - @a pEdid != nullptr
     * @post The EDID is printed to standard out.
     *
     * @details This function is only intended as a debugging aid for other libraries
     *          (ex. nn::vi, nn::audio).
     */
    void PrintEdid(const Edid* pEdid) NN_NOEXCEPT;

}}
