﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief API for querying display modes.
 */

#pragma once

#include <nn/nn_Macro.h>

namespace nn { namespace edid {

    struct Edid;
    struct DisplayModeInfo;
    struct DisplayTimingInfo;
    struct ImageSizeInfo;

    /**
     * @brief Callback function that will execute when a display mode is found.
     *
     * @param[in] pMode Display mode.
     * @param[in] pTimingInfo Optional timing information.
     * @param[in] pSizeInfo   Optional size information.
     * @param[in] pUserData   User defined data passed from VisitDisplayModes().
     *
     * @return Whether to continue searching for display modes.
     *
     * @details The parameters @a pTimingInfo and @a pSizeInfo are optional and may be null when
     *          the function executes if the information is not available.
     *
     *          The same resolution may appear multiple times, each with a different StereoMode.
     */
    typedef bool (*DisplayModeVisitor)(const DisplayModeInfo* pMode, const DisplayTimingInfo* pTimingInfo, const ImageSizeInfo* pSizeInfo, void* pUserData);

    /**
     * @brief Visitor function that will execute a callback for every display mode found.
     *
     * @param[in] pEdid     EDID created from OpenEdid().
     * @param[in] visitor   Callback function to execute.
     * @param[in] pUserData User defined data to pass to the callback function.
     *
     * @pre
     *       - @a pEdid != nullptr
     *       - @a visitor != nullptr
     * @post The function @a visitor is executed for each display mode found.
     */
    void VisitDisplayModes(const Edid* pEdid, DisplayModeVisitor visitor, void* pUserData) NN_NOEXCEPT;

    /**
     * @brief Retrieve the preferred display mode.
     *
     * @param[out] pOutMode   Display mode.
     * @param[out] pOutTiming Timing information.
     * @param[out] pOutSize   Size information.
     * @param[in]  pEdid      EDID created from OpenEdid().
     *
     * @pre
     *       - @a pOutMode != nullptr
     *       - @a pOutTiming != nullptr
     *       - @a pOutSize != nullptr
     *       - @a pEdid != nullptr
     * @post Display mode information is written to @a pOutMode, @a pOutTiming, and @a pOutSize.
     */
    void GetDisplayModePreference(DisplayModeInfo* pOutMode, DisplayTimingInfo* pOutTiming, ImageSizeInfo* pOutSize, const Edid* pEdid) NN_NOEXCEPT;

}}
