﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/crypto/detail/crypto_BigNum.h>
#include <nn/nn_SdkAssert.h>
#include <cstring>

namespace nn { namespace crypto { namespace detail {

template <int bits>
class EccKey
{
public:
    static const size_t BitsCount = bits;
    static const int PrivateKeySize = (bits + 7) / 8;
    static const int PublicKeySize = PrivateKeySize * 2;
    static const int SignatureSize = PublicKeySize;
    static const int SharedSecretSize = PrivateKeySize;

    EccKey() NN_NOEXCEPT
    {
        m_HasPublicKey = false;
        m_HasPrivateKey = false;
    }

    ~EccKey() NN_NOEXCEPT
    {
        if ( m_HasPrivateKey )
        {
            std::memset( &m_PrivateKey, 0, sizeof(m_PrivateKey) );
        }
    }

    void Initialize( const void* publicKey, size_t publicKeySize,
                     const void* privateKey, size_t privateKeySize )  NN_NOEXCEPT
    {
        if ( publicKeySize )
        {
            NN_SDK_ASSERT (publicKeySize == PublicKeySize);

            nn::crypto::detail::DigitsFromOctetString( m_PublicKey.m_X, DigitsCount,
                                                        publicKey, PublicKeySize / 2);
            nn::crypto::detail::DigitsFromOctetString( m_PublicKey.m_Y, DigitsCount,
                                                        static_cast<const char*>(publicKey)
                                                        + PublicKeySize / 2,
                                                        PublicKeySize / 2);
            m_HasPublicKey = true;
        }
        else
        {
            m_HasPublicKey = false;
        }

        if ( privateKeySize )
        {
            NN_SDK_ASSERT (privateKeySize == PrivateKeySize);

            nn::crypto::detail::DigitsFromOctetString( m_PrivateKey, DigitsCount,
                                                        privateKey, PrivateKeySize);
            m_HasPrivateKey = true;
        }
        else
        {
            m_HasPrivateKey = false;
        }
    }

    //protected:
    typedef nn::crypto::detail::BigNum::Digit Digit;
    static const size_t DigitsCount = (bits + sizeof(Digit) * 8 - 1) /
                                      (sizeof(Digit) * 8);
    struct Point
    {
        Digit m_X[DigitsCount];
        Digit m_Y[DigitsCount];
    }
    m_PublicKey;

    Digit m_PrivateKey[DigitsCount];

    bool m_HasPublicKey;
    bool m_HasPrivateKey;
};

}}} // namespace nn::crypto::detail
