﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief RSA-OAEP-SHA-256 のメッセージをデコードする API の宣言
 */

#include <cstring>
#include <nn/nn_SdkAssert.h>
#include <nn/crypto/crypto_Sha256Generator.h>
#include <nn/crypto/detail/crypto_RsaOaepPadding.h>

namespace nn { namespace crypto {

/**
    @brief      鍵サイズが 2048 bit の RSA-OAEP-SHA-256 メッセージから平文を取り出す関数です。

    @param[out] pOutPlain             復元された平文データを取得するバッファへのポインタ。
    @param[in]  outSize               pOutPlain が指すバッファのバイトサイズ。
    @param[in]  pLabelDigest          ラベルデータのハッシュ値へのポインタ。
    @param[in]  labelDigestSize       ラベルデータのハッシュ値のバイトサイズ。
    @param[in]  pEncodedMessage       復元するデータへのポインタ
    @param[in]  encodedMessageSize    復元するデータのバイトサイズ。

    @return     復元された平文データのサイズ。
                不正なメッセージが与えられるなどして復元に失敗した場合は 0 が返ります。

    @pre
    - labelDigestSize == 32
    - encodedMessageSize == 256

    @post
    - pOutPlain に平文が格納される。

    @details
    鍵サイズが 2048bit の RSA-OAEP-SHA-256 でエンコードされたメッセージから平文を取り出します。

    この関数ではメッセージの復号化は行いません。
    復号化が必要な場合は DecryptRsa2048OaepSha256 を利用してください。
 */
inline size_t DecodeRsa2048OaepSha256(void* pOutPlain, size_t outSize,
                                      const void* pLabelDigest, size_t labelDigestSize,
                                      const void* pEncodedMessage, size_t encodedMessageSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(encodedMessageSize == 256);

    detail::RsaOaepPadding<Sha256Generator> rsaOaep;
    uint8_t em[256];

    std::memcpy(em, pEncodedMessage, encodedMessageSize);
    size_t result = rsaOaep.RecoverFromPad(pOutPlain, outSize, pLabelDigest, labelDigestSize, em, sizeof(em));
    // Wipe out secret from stack
    std::memset(em, 0, sizeof(em));
    return result;
}

}} // namespace nn::crypto
