﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief 鍵サイズが 128 bit の AES の XTS モードによる暗号化を行うための API の宣言
 */

#include <nn/nn_SdkAssert.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_XtsEncryptor.h>

namespace nn { namespace crypto {

/**
    @brief      鍵サイズが 128bit の AES の XTS モードによる暗号化を利用するためのクラスです。

    @details
    このクラスは以下の状態を持ちます。
    インスタンスの生成直後は未初期化状態です。
    - 未初期化状態
    - 処理可能状態
    - 完了状態

 */
class Aes128XtsEncryptor
{
    NN_DISALLOW_COPY(Aes128XtsEncryptor);

public:
    static const size_t BlockSize = AesEncryptor128::BlockSize; //!< ブロックサイズを表す定数です。
    static const size_t IvSize    = AesEncryptor128::BlockSize; //!< Tweak のサイズを表す定数です。

public:
    Aes128XtsEncryptor() NN_NOEXCEPT {}

    /**
        @brief      鍵サイズが 128 bit の AES-XTS モードによる暗号化の処理を初期化します。

        @param[in]  pKey1    データ列を暗号化する鍵データへのポインタ。
        @param[in]  pKey2    Tweak を暗号化する鍵データへのポインタ。
        @param[in]  keySize  鍵データのバイトサイズ。
        @param[in]  pIv      Tweak データへのポインタ。
        @param[in]  ivSize   Tweak データのバイトサイズ。

        @pre
        - keySize == 16
        - ivSize == 16

        @post
        - インスタンスは処理可能状態になる。

        @details
        鍵データと Tweak データを入力し、暗号化処理を初期化します。
        鍵データおよび Tweak データは内部にコピーされます。
     */
    void Initialize(const void* pKey1, const void* pKey2, size_t keySize, const void* pIv, size_t ivSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(keySize == 16);
        NN_SDK_REQUIRES(ivSize == IvSize);

        m_Aes1.Initialize(pKey1, keySize);
        m_Aes2.Initialize(pKey2, keySize);
        m_XtsEncryptor.Initialize(&m_Aes1, &m_Aes2, pIv, ivSize);
    }

    /**
        @brief      鍵サイズが 128 bit の AES-XTS モードによる暗号化を行います。

        @copydetails XtsEncryptor::Update
     */
    size_t Update(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
    {
        return m_XtsEncryptor.Update(pDst, dstSize, pSrc, srcSize);
    }

    /**
        @brief      鍵サイズが 128 bit の AES-XTS モードによる暗号化を完了します。

        @copydetails XtsEncryptor::Finalize
     */
    size_t Finalize(void* pDst, size_t dstSize) NN_NOEXCEPT
    {
        return m_XtsEncryptor.Finalize(pDst, dstSize);
    }

private:
    AesEncryptor128 m_Aes1;
    AesEncryptor128 m_Aes2;
    XtsEncryptor<AesEncryptor128> m_XtsEncryptor;
};

/**
    @brief    鍵サイズが 128 bit の AES-XTS モードによる暗号化を行うユーティリティ関数です。

    @param[out] pDst     暗号化されたデータを受け取るバッファへのポインタ。
    @param[in]  dstSize  pDst が指すバッファのバイトサイズ。
    @param[in]  pKey1    データ列を暗号化する鍵データへのポインタ。
    @param[in]  pKey2    Tweak を暗号化する鍵データへのポインタ。
    @param[in]  keySize  鍵データのバイトサイズ。
    @param[in]  pIv      Tweak データへのポインタ。
    @param[in]  ivSize   Tweak データのバイトサイズ。
    @param[in]  pSrc     暗号化するデータが格納されているバッファへのポインタ。
    @param[in]  srcSize  暗号化するデータのバイトサイズ。

    @return     実際に暗号化されたバイト数

    @pre
    - keySize == 16
    - ivSize == 16

    @post
    - pDst に暗号化された結果が書き込まれる。

    @details
    与えられたデータ列を AES-XTS モードで暗号化します。
 */
inline size_t EncryptAes128Xts(void* pDst, size_t dstSize,
                               const void* pKey1, const void* pKey2, size_t keySize,
                               const void* pIv, size_t ivSize,
                               const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    uint8_t* pDst8 = static_cast<uint8_t*>(pDst);
    const uint8_t* pSrc8 = static_cast<const uint8_t*>(pSrc);
    size_t processed = 0;

    Aes128XtsEncryptor aesXts;
    aesXts.Initialize(pKey1, pKey2, keySize, pIv, ivSize);

    processed = aesXts.Update(pDst8, dstSize, pSrc8, srcSize);
    pDst8 += processed;
    dstSize -= processed;

    processed += aesXts.Finalize(pDst8, dstSize);

    return processed;
}

}} // namespace nn::crypto

