﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief 鍵サイズが 128 bit の AES の XTS モードによる復号化を行うための API の宣言
 */

#include <nn/nn_SdkAssert.h>
#include <nn/crypto/crypto_AesEncryptor.h>
#include <nn/crypto/crypto_AesDecryptor.h>
#include <nn/crypto/crypto_XtsDecryptor.h>

namespace nn { namespace crypto {

/**
    @brief      鍵サイズが 128bit の AES の XTS モードによる復号化を利用するためのクラスです。

    @details
    このクラスは以下の状態を持ちます。
    インスタンスの生成直後は未初期化状態です。
    - 未初期化状態
    - 処理可能状態
    - 完了状態

 */
class Aes128XtsDecryptor
{
    NN_DISALLOW_COPY(Aes128XtsDecryptor);

public:
    static const size_t BlockSize = AesEncryptor128::BlockSize; //!< ブロックサイズを表す定数です。
    static const size_t IvSize    = AesEncryptor128::BlockSize; //!< Tweak のサイズを表す定数です。

public:
    Aes128XtsDecryptor() NN_NOEXCEPT {}

    /**
        @brief      鍵サイズが 128 bit の AES-XTS モードによる復号化の処理を初期化します。

        @param[in]  pKey1    データ列を復号化する鍵データへのポインタ。
        @param[in]  pKey2    Tweak を暗号化する鍵データへのポインタ。
        @param[in]  keySize  鍵データのバイトサイズ。
        @param[in]  pIv      Tweak データへのポインタ。
        @param[in]  ivSize   Tweak データのバイトサイズ。

        @pre
        - keySize == 16
        - ivSize == 16

        @post
        - インスタンスは処理可能状態になる。

        @details
        鍵データと Tweak データを入力し、暗号化処理を初期化します。
        鍵データおよび Tweak データは内部にコピーされます。
     */
    void Initialize(const void* pKey1, const void* pKey2, size_t keySize, const void* pIv, size_t ivSize) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES(keySize == 16);
        NN_SDK_REQUIRES(ivSize == IvSize);

        m_Aes1.Initialize(pKey1, keySize);
        m_Aes2.Initialize(pKey2, keySize);
        m_XtsDecryptor.Initialize(&m_Aes1, &m_Aes2, pIv, ivSize);
    }

    /**
        @brief      鍵サイズが 128 bit の AES-XTS モードによる復号化を行います。

        @copydetails XtsDecryptor::Update
     */
    size_t Update(void* pDst, size_t dstSize, const void* pSrc, size_t srcSize) NN_NOEXCEPT
    {
        return m_XtsDecryptor.Update(pDst, dstSize, pSrc, srcSize);
    }

    /**
        @brief      鍵サイズが 128 bit の AES-XTS モードによる復号化を完了します。

        @copydetails XtsDecryptor::Finalize
     */
    size_t Finalize(void* pDst, size_t dstSize) NN_NOEXCEPT
    {
        return m_XtsDecryptor.Finalize(pDst, dstSize);
    }

private:
    AesDecryptor128 m_Aes1;
    AesEncryptor128 m_Aes2;
    XtsDecryptor<AesDecryptor128> m_XtsDecryptor;
};

/**
    @brief    鍵サイズが 128 bit の AES-XTS モードによる復号化を行うユーティリティ関数です。

    @param[out] pDst     復号化されたデータを受け取るバッファへのポインタ。
    @param[in]  dstSize  pDst が指すバッファのバイトサイズ。
    @param[in]  pKey1    データ列を復号化する鍵データへのポインタ。
    @param[in]  pKey2    Tweak を暗号化する鍵データへのポインタ。
    @param[in]  keySize  鍵データのバイトサイズ。
    @param[in]  pIv      カウンタの初期値データへのポインタ。
    @param[in]  ivSize   カウンタの初期値データのバイトサイズ。
    @param[in]  pSrc     復号化するデータが格納されているバッファへのポインタ。
    @param[in]  srcSize  復号化するデータのバイトサイズ。

    @return     実際に復号化されたバイト数

    @pre
    - keySize == 16
    - ivSize == 16

    @post
    - pDst に復号化された結果が書き込まれる。

    @details
    与えられたデータ列を AES-XTS モードで復号化します。
 */
inline size_t DecryptAes128Xts(void* pDst, size_t dstSize,
                               const void* pKey1, const void* pKey2, size_t keySize,
                               const void* pIv, size_t ivSize,
                               const void* pSrc, size_t srcSize) NN_NOEXCEPT
{
    uint8_t* pDst8 = static_cast<uint8_t*>(pDst);
    const uint8_t* pSrc8 = static_cast<const uint8_t*>(pSrc);
    size_t processed = 0;

    Aes128XtsDecryptor aesXts;
    aesXts.Initialize(pKey1, pKey2, keySize, pIv, ivSize);

    processed = aesXts.Update(pDst8, dstSize, pSrc8, srcSize);
    pDst8 += processed;
    dstSize -= processed;

    processed += aesXts.Finalize(pDst8, dstSize);

    return processed;
}

}} // namespace nn::crypto

