﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.IO.Compression;
using System.Linq;
using System.Security.Cryptography;
using System.Text;
using System.Threading.Tasks;
using ZarfCreator.Parser;

namespace ZarfCreator.ZarfDefinitionData
{
    public class RollbackInfo
    {

        public IEnumerable<string> TrackFilePatternList { get; set; }

        internal bool Validate(out string errorMessage)
        {
            var msg = new StringBuilder();
            var result = true;

            // デフォルト値の設定
            if (this.TrackFilePatternList == null)
            {
                this.TrackFilePatternList = new List<string>();
            }

            errorMessage = msg.ToString();
            return result;
        }

        public IEnumerable<FileInventoryInfo> EnumerateFileInventories(IEnumerable<string> zipFiles, string installDirName)
        {
            // ワイルドカードを含むパスのリストを正規表現文字列に変換してつなげる。
            var trackFilePattern = new WildcardPattern(this.TrackFilePatternList);

            return
                (   from zipFile in zipFiles
                    from fileInfo in CreateFileInventories(zipFile, installDirName.Length, trackFilePattern)
                    where fileInfo != null
                    select fileInfo
                );
        }


        private static FileInventoryInfo[] CreateFileInventories(string zipPath, int installDirNameLen, WildcardPattern trackFilePattern)
        {
            // まず、zip のエントリの数だけ結果を保持する配列を用意する。
            int entryNum;
            using (var fs = new FileStream(zipPath, FileMode.Open, FileAccess.Read, FileShare.Read))
            using (var zipArchive = new ZipArchive(fs, ZipArchiveMode.Read, true))
            {
                entryNum = zipArchive.Entries.Count;
            }
            var results = new FileInventoryInfo[entryNum];

            // 各ファイルのSHA1ハッシュ計算について、範囲パーティション分割による並列処理を行う。
            Parallel.ForEach(
                Partitioner.Create(0, entryNum),
                (range, loopState) =>
                    {
                        using (var sha1 = SHA1.Create())
                        using (var fs = new FileStream(zipPath, FileMode.Open, FileAccess.Read, FileShare.Read))
                        using (var zipArchive = new ZipArchive(fs, ZipArchiveMode.Read, true))
                        {
                            for (int i = range.Item1; i < range.Item2; i++)
                            {
                                var entry = zipArchive.Entries[i];
                                if (!entry.FullName.EndsWith("/") && !entry.FullName.EndsWith(@"\"))
                                {
                                    results[i] = GetFileInventoryInfo(entry, sha1, installDirNameLen, trackFilePattern);
                                }
                            }
                        }
                    }
            );

            return results;
        }

        private static FileInventoryInfo GetFileInventoryInfo(ZipArchiveEntry entry, SHA1 sha1, int installDirNameLen, WildcardPattern trackFilePattern)
        {
            var fullName = entry.FullName.Replace('\\', '/');

            var fileTrackType =
                trackFilePattern.IsMatch(fullName) ?
                    FileTrackType.Track :
                    FileTrackType.NotTrack;

            string signature = null;
            if (fileTrackType == FileTrackType.Track)
            {
                byte[] hash;
                using (var stream = entry.Open())
                {
                    hash = sha1.ComputeHash(stream);
                }
                var hashStr = string.Concat(hash.Select(bt => bt.ToString("x2")));
                signature = string.Format("{0}.{1}", hashStr, entry.Length);
            }

            return
                new FileInventoryInfo(
                    @"{release-installdir}" + fullName.Substring(installDirNameLen),
                    signature,
                    fileTrackType);
        }

    }

    public enum FileTrackType
    {
        Track,
        NotTrack
    }

    public class FileInventoryInfo
    {
        public FileInventoryInfo(string path, string signature, FileTrackType trackType)
        {
            this.Path = path;
            this.Signature = signature;
            this.TrackType = trackType;
        }

        public string Path { get; set; }

        public string Signature { get; set; }

        public FileTrackType TrackType { get; set; }
    }
}
