﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading.Tasks;

namespace ZarfCreator.VersionData.SourceType
{
    class Text : VersionBase
    {
        private static readonly Regex VersionRegex = new Regex(string.Format(@"^{0}(\.{0}(\.{0}(?:-({1}))?(?:\+({1}))?)?)?$", NumStr, IdentifiersStr));
        private static readonly Regex DigitRegex = new Regex(@"^\d+$");

        public string rootDirectory { get; private set; }

        public Text(string source, string rootDir) : base(source)
        {
            this.rootDirectory = rootDir;
        }

        public override string Read()
        {
            var versionFilePath = Path.Combine(this.rootDirectory, this.versionInfo);

            string versionFileData;
            try
            {
                using (var sr = new StreamReader(versionFilePath))
                {
                    versionFileData = sr.ReadLine();
                }
            }
            catch (IOException)
            {
                Console.Error.WriteLine("ERROR: Failed to open following file.\n    {0}", versionFilePath);
                throw;
            }

            return ReadSemanticVersion(versionFileData, versionFilePath);
        }

        private static string ReadSemanticVersion(string version, string versionFilePath)
        {
            var match = VersionRegex.Match(version);
            if (!match.Success ||
                (match.Groups[3].Success &&                      // プレリリースがある
                    !ValidatePrerelease(match.Groups[3].Value))
            )
            {
                Console.Error.WriteLine("ERROR: Invalid version format has specified.\n    {0}", versionFilePath);
                throw new ArgumentException("Invalid format has specified.");
            }

            if (!match.Groups[1].Success) // Minor, Micro が省略された場合
            {
                version += ".0.0";
            }
            else if (!match.Groups[2].Success)  // Micro が省略された場合
            {
                version += ".0";
            }

            return version;
        }

        /// <summary>
        /// プレリリース文字列を検証します。
        /// </summary>
        /// <param name="str">プレリリース文字列</param>
        /// <returns>正しい場合は true。</returns>
        private static bool ValidatePrerelease(string str)
        {
            return
                str.Split(new[] { '.' })
                    .All(
                        s =>
                            !DigitRegex.IsMatch(s) ||   // 数値文字だけでない
                            NumRegex.IsMatch(s)         // 0が前についていない数値
                    );
        }
    }
}
