﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.ToolFoundation.IO;
using System.Runtime.CompilerServices;

[assembly: InternalsVisibleTo("ZarfCreatorTest")]

namespace ZarfCreator
{
    internal class Program
    {
        private const string ProgramName = "ZarfCreator";
        private const string ProgramDescription = "MUGEN 用の .zarf ファイルを作成します。";

        internal static void Main(string[] args)
        {
            // 引数の解析
            ZarfCreatorParam param;

            try
            {
                if (!CommandLineParser.Default.ParseArgs(args, out param))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            try
            {
                string checkResult;
                if(!CheckFiles(out checkResult, param.SourceZarfDefinitionPath, param.SourceArchivePaths))
                {
                    Console.Error.WriteLine(checkResult);
                    throw new ArgumentException("Invalid commandline arguments.");
                }
                var zarfCreator = new ZarfCreator(param);
                zarfCreator.Create();
            }
            catch (Exception ex)
            {
                Console.Error.WriteLine("ERROR: Failed to create the package from:");
                Console.Error.WriteLine("'{0}'\n", param.SourceZarfDefinitionPath);
                PrintException(ex);

                Environment.Exit(-1);
            }
        }

        private static void PrintException(Exception ex)
        {
            Console.Error.WriteLine("------------------------------ Details of Exception ------------------------------");
            Console.Error.WriteLine(ex);
            Console.Error.WriteLine("--------------------------------- End of Details --------------------------------\n");
        }

        private static bool CheckFiles(out string errorMessage, string sourceZarfDefinitionPath, string[] sourceArchivePaths)
        {
            var result = true;
            var errorString = new StringBuilder();

            var zarfFileExists = File.Exists(sourceZarfDefinitionPath);
            if (!zarfFileExists)
            {
                errorString.AppendLine("ERROR: Not found the file specified with '--source-file' option.");
                errorString.AppendFormat("'{0}'\n", sourceZarfDefinitionPath);
                result = false;
            }

            var invalidArchivePath = from file in sourceArchivePaths
                                     where !File.Exists(file)
                                     select file;
            if (invalidArchivePath.Count() != 0)
            {
                errorString.AppendLine("ERROR: Not found the file specified with '--archive-path' option.");
                foreach (var path in invalidArchivePath)
                {
                    errorString.AppendFormat("'{0}'\n", path);
                }
                result = false;
            }

            errorMessage = errorString.ToString();

            return result;
        }
    }
}
