﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Management.Automation;
using YamlDotNet.RepresentationModel;

namespace Nintendo.Yaml
{
    /// <summary>
    /// Yaml ファイルの内容を読み込みオブジェクトを生成するコマンドレットクラスです。
    /// </summary>
    [Cmdlet(VerbsData.Import, "Yaml")]
    public class ImportYamlCommand : PSCmdlet
    {
        [Parameter(
            Mandatory = true,
            Position = 0,
            ValueFromPipeline = true,
            ValueFromPipelineByPropertyName = true)]
        [Alias("PSPath")]
        public string Path { get; set; }

        protected override void ProcessRecord()
        {
            var fullPath = SessionState.Path.GetUnresolvedProviderPathFromPSPath(this.Path);
            this.WriteObject(
                from yamlDoc in GetYamlDocument(fullPath)
                select GetYamlDataSub(yamlDoc.RootNode), true);
        }

        /// <summary>
        /// Yamlの実際の型に応じて、PowerShell でアクセスしやすい Array, Hashtable に
        /// 変換します。
        /// </summary>
        /// <param name="node"></param>
        /// <returns></returns>
        private object GetYamlDataSub(YamlNode node)
        {
            YamlSequenceNode sequenceNode = null;
            YamlMappingNode mappingNode = null;
            YamlScalarNode scalarNode = null;
            if (null != (sequenceNode = node as YamlSequenceNode))
            {
                return
                    (from childNode in sequenceNode.Children
                     select GetYamlDataSub(childNode)).ToArray();
            }
            else if (null != (mappingNode = node as YamlMappingNode))
            {
                var dic = new System.Collections.Hashtable();
                foreach (var childNode in mappingNode.Children)
                {
                    dic[(GetYamlDataSub(childNode.Key))] = GetYamlDataSub(childNode.Value);
                }
                return dic;
            }
            else if (null != (scalarNode = node as YamlScalarNode))
            {
                return scalarNode.Value;
            }
            else
            {
                ThrowTerminatingError(
                    new ErrorRecord(
                        new ArgumentException(
                            string.Format("unknown yaml type {0}", node.GetType())),
                        "UnknownYamlType",
                        ErrorCategory.InvalidArgument,
                        this.Path));
                return null;
            }
        }

        /// <summary>
        /// Yaml ドキュメントを取得します。
        /// </summary>
        /// <param name="path"></param>
        /// <returns></returns>
        private IList<YamlDocument> GetYamlDocument(string path)
        {
            using (var sr = new StreamReader(path))
            {
                try
                {
                    var yamlStream = new YamlStream();
                    yamlStream.Load(sr);
                    return yamlStream.Documents;
                }
                catch (YamlDotNet.Core.SyntaxErrorException e)
                {
                    ThrowTerminatingError(
                        new ErrorRecord(
                            e,
                            "YamlSyntaxError",
                            ErrorCategory.InvalidArgument,
                            path));
                    return null;
                }
            }
        }
    }
}
