﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;

namespace VsSolutionBuilderNinjaExecutor
{
    /// <summary>
    /// ビルドシステムに関する機能を提供します。
    /// </summary>
    internal static class BuildSystem
    {
        internal const string EnviromentVariableNintendoSdkRoot = "NINTENDO_SDK_ROOT";

        internal const string NinjaRuleFileName = "build.ninja";
        internal const string NinjaDepsFileName = ".ninja_deps";
        internal const string NinjaLogFileName = ".ninja_log";

        private const string NinjaProgramFileName = "ninja.exe";

        /// <summary>
        /// VsSolutionBuilderのカレントフォルダからsdkのルートパスを探す。
        /// </summary>
        /// <param name="currentPath">VsSolutionBuilderのカレントフォルダ</param>
        /// <returns>sdkのルートパス</returns>
        private static string GetNintendoSdkRootPath(string startPath)
        {
            var ppath = Path.GetDirectoryName(startPath);
            while (ppath != null && ppath != string.Empty)
            {
                if (File.Exists(Path.Combine(ppath, "NintendoSdkRootMark")) == true)
                {
                    return ppath;
                }
                startPath = ppath;
                ppath = Path.GetDirectoryName(startPath);
            }
            return string.Empty;
        }
        /// <summary>
        /// VsSolutionBuilderのカレントフォルダからsdkのルートフォルダを探し、
        ///  CommandLineToolsがあるフルパスを作成する。
        /// </summary>
        /// <param name="currentPath">VsSolutionBuilderのカレントフォルダ</param>
        /// <returns>CommandLineToolsがあるフルパス</returns>
        private static string GetNintendoSdkCommandLineToolsRootPath(string currentPath)
        {
            var ppath = GetNintendoSdkRootPath(currentPath);
            if (ppath != string.Empty)
            {
                ppath = Path.Combine(ppath, "Integrate\\CommandLineTools");
            }
            return ppath;
        }
        /// <summary>
        /// VsSolutionBuilderのカレントフォルダからsdkのルートフォルダを探し、
        ///  ninja.exeのフルパスを作成する。
        /// </summary>
        /// <param name="currentPath">VsSolutionBuilderのカレントフォルダ</param>
        /// <returns>ninja.exeのフルパス</returns>
        internal static string GetNinjaProgramFilePath(string currentPath)
        {
            var ppath = GetNintendoSdkRootPath(currentPath);
            if (ppath != string.Empty)
            {
                ppath = Path.Combine(ppath, @"Externals\Binaries\Ninja");
                return Path.Combine(ppath, NinjaProgramFileName);
            }
            else
            {
                return NinjaProgramFileName;
            }
        }
        /// <summary>
        /// ビルドプログラムに渡す引数を生成します。
        /// </summary>
        internal class Arguments
        {
            internal Arguments()
            {
                this.Options = new Dictionary<string, string>();

                this.UserOptions = new Dictionary<string, string>();
            }

            private Dictionary<string, string> Options { get; set; }

            private Dictionary<string, string> UserOptions { get; set; }

            /// <summary>
            /// このインスタンスの値を String に変換します。
            /// </summary>
            public override string ToString()
            {
                var list = new List<string>(this.Options.Count + this.UserOptions.Count);

                foreach (var pair in this.Options)
                {
                    list.Add(pair.Value == null
                        ? pair.Key : string.Format("{0} {1}", pair.Key, pair.Value));
                }

                foreach (var pair in this.UserOptions)
                {
                    list.Add(string.Format("{0}={1}", pair.Key, pair.Value));
                }

                return string.Join(" ", list);
            }

            /// <summary>
            /// ビルド中にエラーが発生しても停止せず継続する引数を設定する。
            /// </summary>
            /// <returns>設定操作が完了した後のこのインスタンスへの参照です。</returns>
            internal Arguments SetKeepGoingJobsFails()
            {
                this.Options.Add("-k", "0");

                return this;
            }

            /// <summary>
            /// 並列して実行するルールの上限数を設定します。
            /// </summary>
            /// <param name="count">並列して実行するルールの上限数です。</param>
            /// <returns>設定操作が完了した後のこのインスタンスへの参照です。</returns>
            internal Arguments SetExecuteParallel(int count)
            {
                if (count > 0)
                {
                    this.Options.Add("-j", count.ToString());
                }
                else
                {
                    this.Options.Add("-j", Environment.ProcessorCount.ToString());
                }
                return this;
            }
        }
    }
}
