﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text;
using VsSolutionBuilderNinjaExecutor.ServiceMessage;

namespace VsSolutionBuilderNinjaExecutor
{
    /// <summary>
    /// テスト結果ファイルのビルドルールを作成します。
    /// </summary>
    public class BuildRuleMaker
    {
        /// <summary>
        /// MSBuildのツールバージョン
        /// </summary>
        public const string MsbuildToolsetVersion150 = "15.0";
        public const string MsbuildToolsetVersion140 = "14.0";
        public const string MsbuildToolsetVersion2015 = "2015";
        public const string MsbuildToolsetVersion2017 = "2017";

        /// <summary>
        /// MSBuildWrapper.exeの保存先
        /// </summary>
        private const string MsBuildWrapperPath
            = "..\\..\\..\\MSBuildWrapper\\Release\\MSBuildWrapper.exe";
        /// <summary>
        /// build.ninjaファイルの保存先フォルダ名
        /// </summary>
        private const string NinjaRuleOutputFolder = "BuildcheckNinja";
        /// <summary>
        /// VsSolutionBuilderNinjaCompileRuleの作成パターン
        /// ・パターン１、作成しない
        /// ・パターン２、作成するが、Resultsが空白
        /// ・パターン３：作成する。Resultsを記述する。
        /// </summary>
        private enum VsSolutionBuilderNinjaCompileRulePattern
        { NOT_CREATE, CREATE_WITHOUT_RESULTS, CREATE_WITH_RESULTS }
        /// <summary>
        /// VsSolutionBuilderNinjaBuildRuleの作成パターン
        /// ・パターン１、作成しない
        /// ・パターン２、作成するが、Sources・Resultsが空白
        /// ・パターン３：作成する。Sources・Resultsを記述する。
        /// </summary>
        private enum VsSolutionBuilderNinjaBuildRulePattern
        { NOT_CREATE, CREATE_WITHOUT_SOURCES_AND_RESULTS, CREATE_WITH_SOURCES_AND_RESULTS }

        /// <summary>
        /// VsSolutionBuilderのインストールフォルダ
        /// このフォルダを起点にsdkのルートパスを探す。
        /// </summary>
        private string InstallPath { get; set; }
        /// <summary>
        /// --pathで指定したフォルダ
        /// </summary>
        private string PathFolder { get; set; }
        /// <summary>
        /// --fileで指定されたslnファイルの一覧 または、
        /// --pathで指定したフォルダ内のslnファイルの一覧
        /// </summary>
        private IEnumerable<string> FileList { get; set; }
        /// <summary>
        /// --configurationで指定されたコンフィグレーション名
        /// </summary>
        private List<string> Configuration { get; set; }
        /// <summary>
        /// --platformで指定されたプラットフォーム名
        /// </summary>
        private List<string> Platform { get; set; }
        /// <summary>
        /// --CIModeで指定されたTeamCity 用のログを出力するかどうかを指定。
        /// </summary>
        private bool IsCiMode { get; set; }
        /// <summary>
        /// 並列実行数の上限値。
        /// </summary>
        private int JobCount { get; set; }
        /// <summary>
        /// MSBuild実行回数
        /// </summary>
        private int MsbuildRunCount { get; set; }
        /// <summary>
        /// MSBuildWrapper.exeが格納されているパス
        /// </summary>
        private string MSBuildWrapperPath { get; set; }
        /// <summary>
        /// --target引数（をプログラムで使用できるよう整形した一覧）
        /// </summary>
        private List<string> TargetsToBuildList { get; set; }
        /// <summary>
        /// --pathで指定されたslnファイルが所属するフォルダの一覧
        /// </summary>
        private List<string> SlnFolderList { get; set; }
        /// <summary>
        /// 異なるslnファイルが共通のプロジェクトファイルを使用した場合に
        /// 　成果物の重複有無を判断するためのルール化済み
        ///   のプロジェクトファイル情報を保存するリスト
        /// NXの場合のみ、2015/2017のソリューションファイルによる
        /// 　成果物の重複有無を判断も含む
        /// </summary>
        private List<RuledProjectInfo> RuledProjectInfoList { get; set; }
        /// <summary>
        /// PreBuildEventを直列実行するためのProjectファイルのDictionary
        /// </summary>
        private Dictionary<string, List<ProjectFile>>
                ProjfileDictionaryForPreBuildEventSerial { get; set; }
        /// <summary>
        /// PostBuildEventを直列実行するためのProjectファイルのDictionary
        /// </summary>
        private Dictionary<string, List<ProjectFile>>
                ProjfileDictionaryForPostBuildEventSerial { get; set; }
        /// <summary>
        /// VsSolutionBuilderNinjaCompileRule出力時のパターン
        /// --targetの組み合わせごとに値が変化する。
        /// </summary>
        private VsSolutionBuilderNinjaCompileRulePattern
                VsSolutionBuilderNinjaCompileRulePatternImpl { get; set; }
        /// <summary>
        /// VsSolutionBuilderNinjaBuildRule出力時のパターン
        /// --targetの組み合わせごとに値が変化する。
        /// </summary>
        private VsSolutionBuilderNinjaBuildRulePattern
                VsSolutionBuilderNinjaBuildRulePatternImpl { get; set; }

        /// <summary>
        /// PreBuildEvent実施前にビルドが必要なslnファイル名のリスト
        /// </summary>
        private List<PreBuildSolution> VsPrebuildSolutionList { get; set; }

        /// <summary>
        /// PreBuildEvent実施前にビルドが必要なslnファイル名のリスト
        /// </summary>
        private List<string> VsExcludeProjectNameList { get; set; }

        /// <summary>
        /// 標準出力、標準エラーを出力をスレッドセーフにして、
        /// メッセージの整合性を確保するオブジェクト
        /// </summary>
        private static object lockObject = new object();

        /// <summary>
        /// 環境変数TMP
        /// </summary>
        private const string EnviromentVariableTmp = "TMP";
        /// <summary>
        /// 一時フォルダの一覧
        /// </summary>
        private List<string> TmpFolderList { get; set; }
        /// <summary>
        /// 一時フォルダをすべて削除する前の待ち時間
        /// </summary>
        private const int WaitTimeBeforeDeleteFolder = 10 * 1000;
        /// <summary>
        /// IO 処理に失敗した際のリトライ回数
        /// </summary>
        private const int RetryCount = 10;
        /// <summary>
        /// IO 処理に失敗した際のリトライ前の待ち時間
        /// </summary>
        private const int WaitTimeBeforeRetry = 1000;
        /// <summary>
        /// clean実行時の処理シーケンス番号
        /// </summary>
        private int buildCleanSequenceNo = 0;

        /// <summary>
        /// BuildRuleMaker クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="pathFolder">--pathで指定されたフォルダパス
        /// <param name="fileList">--fileで指定されたslnファイルの一覧
        /// または、--pathで指定したフォルダ内のslnファイルの一覧</param>
        /// <param name="configuration">--configurationで指定されたコンフィグレーション名</param>
        /// <param name="platform">--platformで指定されたプラットフォーム名</param>
        /// <param name="targetsToBuildList">--targetで指定されたMSBuild時のターゲット</param>
        /// <param name="isCiMode">--CIModeで指定されたTeamCity 用のログを出力するかどうかを指定
        /// </param>
        /// <param name="jobCount">並列実行数の上限値です。</param>
        public BuildRuleMaker(string pathFolder, IEnumerable<string> fileList,
            List<string> configuration, List<string> platform,
            List<string> targetsToBuildList, bool isCiMode, int jobCount)
        {
            this.PathFolder = pathFolder;
            this.FileList = fileList;
            this.Configuration = configuration;
            this.Platform = platform;
            this.TargetsToBuildList = targetsToBuildList;
            this.IsCiMode = isCiMode;
            this.JobCount = jobCount;
            this.RuledProjectInfoList = new List<RuledProjectInfo>();
            var myAssembly = Assembly.GetEntryAssembly();
            var path = myAssembly.Location;
            this.InstallPath = Path.GetDirectoryName(path);
            var baseUri = new Uri(path);
            this.MSBuildWrapperPath = new Uri(baseUri, MsBuildWrapperPath).LocalPath;
            this.ProjfileDictionaryForPreBuildEventSerial = new Dictionary<string, List<ProjectFile>>();
            this.ProjfileDictionaryForPostBuildEventSerial = new Dictionary<string, List<ProjectFile>>();
            this.VsPrebuildSolutionList = PreBuildSolution.GetPreBuildSolutionList();
            this.VsExcludeProjectNameList = GetVsExcludeProjectNameList();
            this.MsbuildRunCount = 0;
            this.TmpFolderList = new List<string>();
        }

        /// <summary>
        /// Ninjaによる並列ビルドのメイン関数
        /// </summary>
        /// <note>
        /// TestRunner#Runメソッドを参考に作成
        /// </note>
        public void Run()
        {
            //  引数--targetを整形する。
            ConvertToTargetsToBuildList();
            //  ルール作成パターンの判定
            DecisionRulePattern();
            //  引数で指定されたslnファイルが所属するフォルダ一覧を抽出。
            GetSlnFolderList();

            // （ループ①）getSlnFolderListで抽出したフォルダ数分繰り返し
            foreach (string oneFolder in this.SlnFolderList)
            {
                var sbninja = new StringBuilder();

                //  該当フォルダに属するslnファイル一覧の抽出
                var fileListInSolutionFolder = GetSolutionFileListInFolder(oneFolder);

                //  build.ninjaファイル出力先フォルダ作成
                //  slnファイルがあるフォルダ内に「BuildcheckNinja」サブフォルダを作成する。
                var resultNinjaRootPath = Path.Combine(oneFolder, NinjaRuleOutputFolder);
                Directory.CreateDirectory(resultNinjaRootPath);

                //  Ninja共通ルールの作成
                sbninja.AppendLine(this.GetVsSolutionBuilderNinjaRule());

                //  ループ②
                //  ビルドルール作成のForEach処理、slnファイル数分繰り返し
                foreach (var file in fileListInSolutionFolder.ToList())
                {
                    var sln = new SolutionFile(file, this.VsExcludeProjectNameList);

                    //  1つのslnファイルに含まれるビルド対象の組み合わせを取得する。
                    var buildSettingInfos = GetBuildSettingInfosInSolutionFile(sln);

                    try
                    {
                        foreach (var vsPrebuildSolutionItem in VsPrebuildSolutionList)
                        {
                            if (sln.FilePath.Contains(vsPrebuildSolutionItem.FileName) == true &&
                                sln.FilePath.Contains(vsPrebuildSolutionItem.VsVersion) == true)
                            {
                                vsPrebuildSolutionItem.IsMatched = true;
                                ExecutePrepareBuild(sln,
                                    vsPrebuildSolutionItem.Configuration,
                                    vsPrebuildSolutionItem.Platform);
                            }
                        }

                        //  ループ③
                        //  ビルド対象のplatform／configurationの組み合わせ分繰り返し
                        foreach (var buildSettingInfo in buildSettingInfos)
                        {
                            var currentResultsFileList = new List<string>();

                            if (buildSettingInfo.IsErrorOccurred == true)
                            {
                                ErrorOccurredSlnReport(buildSettingInfo);
                                continue;
                            }
                            //  ループ④
                            //  ビルドルール作成のForEach 処理、
                            //  slnファイル内のvcxprojファイル数分繰り返し
                            foreach (var projectFile in buildSettingInfo.ProjectFiles)
                            {
                                ExecuteCsProj(projectFile);
                                ListupPrePostBuildEvent(projectFile);

                                if (IsRuledProjectInfo(projectFile.ProjectFileName,
                                    projectFile.Configuration,
                                    projectFile.Platform) == false)
                                {
                                    //　成果物が重複する場合は次の組み合わせを処理する。
                                    continue;
                                }
                                //  vcxprojファイル単位で、
                                //  /t:Clean、/t:ClCompileを実施するルール(※１)を記述する。
                                //  ※１：VsSolutionBuilderNinjaCompileRule
                                sbninja.Append(this.GetVsSolutionBuilderNinjaCompileRuleInstance(
                                    sln.Spec, buildSettingInfo, projectFile));
                                //  vcxprojファイル単位で、/t:Build
                                //  /t:(その他任意のターゲット)を実施するルール（※２）を記述する。
                                //  ※２：VsSolutionBuilderNinjaBuildRule
                                sbninja.Append(this.GetVsSolutionBuilderNinjaBuildRuleInstance(
                                    sln.Spec, sln.FilePath, buildSettingInfo,
                                    currentResultsFileList, projectFile));
                            }
                        }
                    }
                    catch (Exception e)
                    {
                        BuildRuleMaker.PrintException(e);
                    }
                }

                //  build.ninjaのファイル保存
                var buildninjaFile = Path.Combine(resultNinjaRootPath, BuildSystem.NinjaRuleFileName);
                var ninjadepsFile = Path.Combine(resultNinjaRootPath, BuildSystem.NinjaDepsFileName);
                var ninjalogFile = Path.Combine(resultNinjaRootPath, BuildSystem.NinjaLogFileName);
                File.WriteAllText(buildninjaFile, sbninja.ToString(), Encoding.ASCII);

                UnmatchedSlnReport();

                try
                {
                    //  MSBuildのPreBuildEventを直列実行する。
                    ExecutePreBuildEvent();
                    //  Process()を介して、Ninjaを実行する。
                    ExecuteNinja(resultNinjaRootPath);
                    //  MSBuildのPostBuildEventを直列実行する。
                    ExecutePostBuildEvent();
                }
                finally
                {
                    // 作成したファイルの削除
                    File.Delete(buildninjaFile);
                    File.Delete(ninjadepsFile);
                    File.Delete(ninjalogFile);
                }
            }

            //  一時フォルダを削除する
            DeleteFolder();
        }

        /// <summary>
        /// 発生した例外をコンソールに出力する。
        /// </summary>
        /// <param name="exception">例外インスタンス</param>
        public static void PrintException(Exception exception)
        {
            var detailBuilder = new StringBuilder();
            for (var ex = exception; ex != null; ex = ex.InnerException)
            {
                if (ex.Data.Contains("Detail"))
                {
                    detailBuilder.Append(ex.Data["Detail"]);
                }
            }

            Console.Error.WriteLine("Error: {0}", exception.Message + detailBuilder.ToString());
            Console.Error.WriteLine(string.Empty);
            Console.Error.WriteLine("== Exception ==");
            Console.Error.WriteLine(string.Empty);
            Console.Error.WriteLine(exception.ToString());
        }

        /// <summary>
        /// 引数で指定されたslnファイルが所属するフォルダ一覧を抽出。
        /// </summary>
        /// <note>
        /// 具体例：
        /// --file=
        /// (sdk)\Samples\Sources\Applications\Demo1\Demo1-spec.Generic.autogen.vs2015.sln,
        /// (sdk)\Samples\Sources\Applications\Demo1\Demo1-spec.Generic.autogen.vs2017.sln,
        /// (sdk)\Samples\Sources\Applications\
        ///         HidNpadIntegrate\HidNpadIntegrate-spec.NX.autogen.vs2015.sln,
        /// (sdk)\Samples\Sources\Applications\
        ///         HidNpadIntegrate\HidNpadIntegrate-spec.NX.autogen.vs2017.sln
        /// 上記４ファイルが--file引数の場合は、以下２フォルダが抽出されslnFolderListに保存される。
        /// (sdk)\Samples\Sources\Applications\Demo1\,
        /// (sdk)\Samples\Sources\Applications\HidNpadIntegrate\
        /// Ninja実行は--fileで指定したフォルダ順となる。
        /// </note>
        private void GetSlnFolderList()
        {
            this.SlnFolderList = new List<string>();
            if (this.PathFolder == null)
            {
                // 引数で指定されたフォルダ数を特定する。
                foreach (var file in this.FileList.ToList())
                {
                    if (this.SlnFolderList.Contains(Path.GetDirectoryName(file)) == false)
                    {
                        this.SlnFolderList.Add(Path.GetDirectoryName(file));
                    }
                }
            }
            else
            {
                this.SlnFolderList.Add(this.PathFolder);
            }
        }
        /// <summary>
        /// fileListの要素で、引数で渡されたフォルダに属するslnファイル一覧の抽出
        /// </summary>
        /// <param name="oneFolder">抽出対象のフォルダ</param>
        /// <returns>引数で渡されたフォルダに属するslnファイル一覧</returns>
        private List<string> GetSolutionFileListInFolder(string oneFolder)
        {
            var slnListOneFolder = new List<string>();
            if (this.PathFolder == null)
            {
                // 引数で指定されたフォルダ単位でNinjaを実行する。
                foreach (var file in this.FileList)
                {
                    if (oneFolder == Path.GetDirectoryName(file))
                    {
                        // 同じフォルダ配下のslnファイルのpickup
                        slnListOneFolder.Add(file);
                    }
                }
            }
            else
            {
                slnListOneFolder.AddRange(this.FileList);
            }
            return slnListOneFolder;
        }
        /// <summary>
        /// 引数--targetを整形する。
        /// 重複の削除や、大文字、小文字混在時の置換
        /// </summary>
        /// <note>
        /// 混在例：期待値 ClCompile⇒clcompile、Clcompile、CLCOMPILEなど
        /// </note>
        private void ConvertToTargetsToBuildList()
        {
            var textInfo = CultureInfo.CurrentCulture.TextInfo;

            //  targetsToBuildの変換
            //  Clean、ClCompile、Build、Rebuildにおいて、
            //  大文字・小文字が混在する場合、すべて先頭大文字に統一する。
            for (int i = 0; i < this.TargetsToBuildList.Count; i++)
            {
                if (textInfo.ToTitleCase(this.TargetsToBuildList[i])
                    == textInfo.ToTitleCase(TargetTypeDefinition.CLEAN))
                {
                    this.TargetsToBuildList[i] = TargetTypeDefinition.CLEAN;
                }
                else if (textInfo.ToTitleCase(this.TargetsToBuildList[i])
                    == textInfo.ToTitleCase(TargetTypeDefinition.CLCOMPILE))
                {
                    this.TargetsToBuildList[i] = TargetTypeDefinition.CLCOMPILE;
                }
                else if (textInfo.ToTitleCase(this.TargetsToBuildList[i])
                    == textInfo.ToTitleCase(TargetTypeDefinition.BUILD))
                {
                    this.TargetsToBuildList[i] = TargetTypeDefinition.BUILD;
                }
                else if (textInfo.ToTitleCase(this.TargetsToBuildList[i])
                    == textInfo.ToTitleCase(TargetTypeDefinition.REBUILD))
                {
                    this.TargetsToBuildList[i] = TargetTypeDefinition.REBUILD;
                }
            }
            //  重複の削除
            this.TargetsToBuildList = this.TargetsToBuildList.Distinct().ToList();

            //　Clean＋BuildはRebuildに変換する。
            if (this.TargetsToBuildList.Contains(TargetTypeDefinition.CLEAN) == true)
            {
                if (this.TargetsToBuildList.Contains(TargetTypeDefinition.BUILD) == true)
                {
                    this.TargetsToBuildList.RemoveAll(target =>
                        target == TargetTypeDefinition.CLEAN);
                    this.TargetsToBuildList.RemoveAll(target =>
                        target == TargetTypeDefinition.BUILD);
                    if (this.TargetsToBuildList.Contains(TargetTypeDefinition.REBUILD) == false)
                    {
                        this.TargetsToBuildList.Add(TargetTypeDefinition.REBUILD);
                    }
                }
            }
            //　Clean＋RebuildはCleanを削除する。
            if (this.TargetsToBuildList.Contains(TargetTypeDefinition.CLEAN) == true)
            {
                if (this.TargetsToBuildList.Contains(TargetTypeDefinition.REBUILD) == true)
                {
                    this.TargetsToBuildList.RemoveAll(target =>
                        target == TargetTypeDefinition.CLEAN);
                }
            }
            //　ClCompile＋BuildはClCompileを削除する。
            if (this.TargetsToBuildList.Contains(TargetTypeDefinition.BUILD) == true)
            {
                this.TargetsToBuildList.RemoveAll(target =>
                    target == TargetTypeDefinition.CLCOMPILE);
            }
            //　ClCompile＋RebuildはClCompileを削除する。
            if (this.TargetsToBuildList.Contains(TargetTypeDefinition.REBUILD) == true)
            {
                this.TargetsToBuildList.RemoveAll(target =>
                    target == TargetTypeDefinition.CLCOMPILE);
            }
            //　Build＋RebuildはBuildを削除する。
            if (this.TargetsToBuildList.Contains(TargetTypeDefinition.REBUILD) == true)
            {
                this.TargetsToBuildList.RemoveAll(target =>
                    target == TargetTypeDefinition.BUILD);
            }
        }
        /// <summary>
        /// ビルドルール作成時のパターン判定
        /// １、VsSolutionBuilderNinjaCompileRule
        /// 　--targetの組み合わせに応じて以下３パターンに分類可能
        /// 　パターン１：作成しない。
        /// 　パターン２：作成するが、Resultsが空白
        /// 　パターン３：作成する。Resultsを記述する。
        /// ２、VsSolutionBuilderNinjaBuildRule
        /// 　--targetの組み合わせに応じて以下３パターンに分類可能
        /// 　パターン１：作成しない。
        /// 　パターン２：作成するが、Sources・Resultsが空白
        /// 　パターン３：作成する。Sources・Resultsを記述する。
        /// </summary>
        /// <note>
        /// 具体例）ClCompileが--target引数に指定された場合の判定結果は以下。
        /// 　VsSolutionBuilderNinjaCompileRuleはパターン３
        /// 　VsSolutionBuilderNinjaBuildRuleはパターン１
        /// </note>
        private void DecisionRulePattern()
        {
            var targetItems = new List<string>()
            {
                TargetTypeDefinition.CLEAN,
                TargetTypeDefinition.CLCOMPILE,
                TargetTypeDefinition.BUILD,
                TargetTypeDefinition.REBUILD
            };

            //　VsSolutionBuilderNinjaCompileRulePatternの判定
            if (this.TargetsToBuildList.Contains(TargetTypeDefinition.CLCOMPILE) == true)
            {
                this.VsSolutionBuilderNinjaCompileRulePatternImpl
                    = VsSolutionBuilderNinjaCompileRulePattern.CREATE_WITH_RESULTS;
            }
            else if (this.TargetsToBuildList.Contains(TargetTypeDefinition.CLEAN) == true)
            {
                this.VsSolutionBuilderNinjaCompileRulePatternImpl
                    = VsSolutionBuilderNinjaCompileRulePattern.CREATE_WITHOUT_RESULTS;
            }
            else
            {
                this.VsSolutionBuilderNinjaCompileRulePatternImpl
                    = VsSolutionBuilderNinjaCompileRulePattern.NOT_CREATE;
            }
            //　VsSolutionBuilderNinjaBuildRulePatternの判定
            if (this.TargetsToBuildList.Contains(TargetTypeDefinition.BUILD) == true ||
                 this.TargetsToBuildList.Contains(TargetTypeDefinition.REBUILD) == true)
            {
                this.VsSolutionBuilderNinjaBuildRulePatternImpl
                    = VsSolutionBuilderNinjaBuildRulePattern.CREATE_WITH_SOURCES_AND_RESULTS;
            }
            else if (this.TargetsToBuildList.Except(targetItems).Count() > 0)
            {   //  Clean、ClCompile、Build、Rebuild以外の任意のtargetが指定された。
                this.VsSolutionBuilderNinjaBuildRulePatternImpl
                    = VsSolutionBuilderNinjaBuildRulePattern.CREATE_WITHOUT_SOURCES_AND_RESULTS;
            }
            else
            {
                this.VsSolutionBuilderNinjaBuildRulePatternImpl
                    = VsSolutionBuilderNinjaBuildRulePattern.NOT_CREATE;
            }
        }

        /// <summary>
        /// 1つのslnファイルに含まれるビルド対象の組み合わせを取得する。
        /// 引数(--configuration、--platform)に該当しないものは組み合わせから除外する。
        /// 引数を指摘しない場合は、slnファイルに含まれるすべての組み合わせがビルド対象。
        /// </summary>
        /// <note>
        /// slnファイルには以下のようなビルドの組み合わせがあらかじめ設定されている。
        /// この組み合わせのうち、ビルド対象となるものを本関数で抽出する。
        ///     GlobalSection(SolutionConfigurationPlatforms) = preSolution
        ///         Debug|Win32 = Debug|Win32
        ///         Debug|x64 = Debug|x64
        ///         Develop|Win32 = Develop|Win32
        ///         Develop|x64 = Develop|x64
        ///         Release|Win32 = Release|Win32
        ///         Release|x64 = Release|x64
        /// </note>
        /// <param name="sln">処理中のslnファイルをSolutionFileクラスにインスタンス化したもの
        /// </param>
        /// <returns>1つのslnファイルに含まれるビルド対象の組み合わせ</returns>
        private List<BuildSettingInfo> GetBuildSettingInfosInSolutionFile(SolutionFile sln)
        {
            var settings = from setting in sln.Configuration
                           where this.Configuration != null
                                ? this.Configuration.Contains(setting.Configuration)
                                : true   // configuration の絞り込み
                           where this.Platform != null
                                ? this.Platform.Contains(setting.Platform)
                                : true   // platform の絞り込み
                           select setting;
            return settings.ToList();
        }

        /// <summary>
        /// Ninja共通ルールの作成
        /// </summary>
        /// <returns>Ninja共通ルールの文字列</returns>
        private string GetVsSolutionBuilderNinjaRule()
        {
            var cleanCompileTarget = GetVsSolutionBuilderNinijaCleanCompileRule();
            var buildTarget = GetVsSolutionBuilderNinjaBuildRule();

            var sbninjaRule = new StringBuilder();
            sbninjaRule.AppendLine(@"program = " + this.MSBuildWrapperPath);

            if (cleanCompileTarget != string.Empty)
            {
                sbninjaRule.AppendLine(
@"MsBuildTargetCleanClcompile = "
+ cleanCompileTarget + @"
rule MsbuildRuleCleanClcompile
   command = $program $TargetFilePath $Configuration $Platform " + this.IsCiMode.ToString() + " "
+ @"$TempFolderName $TargetFilePath /m /nologo /v:minimal /t:$MsBuildTargetCleanClcompile "
+ @"/p:Configuration=$Configuration /p:Platform=$Platform /p:PreBuildEventUseInBuild=false "
+ @"/p:PostBuildEventUseInBuild=false $PreferredToolArchitecture $VsVersion
"
);
            }
            if (buildTarget != string.Empty)
            {
                sbninjaRule.AppendLine(
@"MsBuildTargetBuild = "
+ buildTarget + @"
rule MsbuildRuleBuild
   command = $program $TargetFilePath $Configuration $Platform " + this.IsCiMode.ToString() + " "
+ @"$TempFolderName $TargetFilePath /m /nologo /v:minimal /t:$MsBuildTargetBuild "
+ @"/p:Configuration=$Configuration /p:Platform=$Platform /p:PreBuildEventUseInBuild=false "
+ @"/p:PostBuildEventUseInBuild=false $PreferredToolArchitecture $VsVersion
"
);
            }

            return sbninjaRule.ToString();
        }
        /// <summary>
        /// ビルド共通ルールの作成(Clean,ClCompile)
        /// ・VsSolutionBuilderNinjaCompileRule
        /// </summary>
        /// <returns>VsSolutionBuilderNinjaCompileRuleルールの文字列</returns>
        private string GetVsSolutionBuilderNinijaCleanCompileRule()
        {
            var isClean = this.TargetsToBuildList.Contains(TargetTypeDefinition.CLEAN);
            var isClCompile = this.TargetsToBuildList.Contains(TargetTypeDefinition.CLCOMPILE);
            var cleanCompileTarget = string.Empty;

            if (this.VsSolutionBuilderNinjaCompileRulePatternImpl
                == VsSolutionBuilderNinjaCompileRulePattern.NOT_CREATE)
            {
                return string.Empty;
            }

            if (isClean == true && isClCompile == true)
            {
                cleanCompileTarget = TargetTypeDefinition.CLEAN
                                        + "," + TargetTypeDefinition.CLCOMPILE;
            }
            else if (isClean == true && isClCompile == false)
            {
                cleanCompileTarget = TargetTypeDefinition.CLEAN;
            }
            else if (isClean == false && isClCompile == true)
            {
                cleanCompileTarget = TargetTypeDefinition.CLCOMPILE;
            }

            return cleanCompileTarget;
        }
        /// <summary>
        /// ビルド共通ルールの作成(Build)
        /// ・VsSolutionBuilderNinjaBuildRule
        /// </summary>
        /// <returns>VsSolutionBuilderNinjaBuildRuleルールの文字列</returns>
        private string GetVsSolutionBuilderNinjaBuildRule()
        {
            var targetsToBuildListTemp = new List<string>(this.TargetsToBuildList);

            if (this.VsSolutionBuilderNinjaBuildRulePatternImpl
                == VsSolutionBuilderNinjaBuildRulePattern.NOT_CREATE)
            {
                return string.Empty;
            }

            //  CleanとClCompileはslnファイルのターゲット指定からは除外
            targetsToBuildListTemp.Remove(TargetTypeDefinition.CLEAN);
            targetsToBuildListTemp.Remove(TargetTypeDefinition.CLCOMPILE);

            return string.Join(",", targetsToBuildListTemp.Select(
                        target => string.Format("{0}", target)));
        }

        /// <summary>
        /// 1プロジェクトファイル単位で、ビルドルール(clean,clcompile)の文字列を作成する。
        /// </summary>
        /// <param name="spec">Generic/NX</param>
        /// <param name="buildSettingInfo">ビルド設定種別</param>
        /// <param name="projectFile">プロジェクトファイルのインスタンス</param>
        /// <returns>ビルドルール(clean,clcompile)の文字列</returns>
        private string GetVsSolutionBuilderNinjaCompileRuleInstance(
            string spec, BuildSettingInfo buildSettingInfo, ProjectFile projectFile)
        {
            var objectFileList = new List<string>();

            switch (this.VsSolutionBuilderNinjaCompileRulePatternImpl)
            {
                // CleanもClCompileも実施しない場合はルールを記載しない。
                case VsSolutionBuilderNinjaCompileRulePattern.NOT_CREATE:
                    return string.Empty;
                case VsSolutionBuilderNinjaCompileRulePattern.CREATE_WITHOUT_RESULTS:
                    objectFileList.Add(((buildCleanSequenceNo++).ToString()));
                    break;
                case VsSolutionBuilderNinjaCompileRulePattern.CREATE_WITH_RESULTS:
                default:
                    objectFileList = projectFile.GetObjectFileList(
                        spec, buildSettingInfo.Configuration);
                    break;
            }

            var tempFolderName = CreateTempFolder();
            // /p:VisualStudioVersionの設定
            var vsVersion = GetMsbuildVsversionParameter(projectFile.Configuration);
            // /p:PreferredToolArchitectureの設定
            var preferredToolArchitecture
                = GetMsbuildPreferredToolArchitectureParameter(projectFile.Configuration);
            return
@"build " + string.Join(" ",
                objectFileList.Select(
                outputFile => string.Format("{0}", SeparateDriveLetter(outputFile))))
 + @": MsbuildRuleCleanClcompile
    Platform = " + projectFile.Platform + @"
    Configuration = " + projectFile.Configuration + @"
    TargetFilePath = " + projectFile.ProjectFileName + @"
    TempFolderName = " + tempFolderName + @"
    PreferredToolArchitecture = " + preferredToolArchitecture + @"
    VsVersion = " + vsVersion + @"
";
        }
        /// <summary>
        /// 1プロジェクトファイル単位で、ビルドルール(build)の文字列を作成する。
        /// </summary>
        /// <param name="spec">Generic/NX</param>
        /// <param name="filePath">ソリューションファイルのパス</param>
        /// <param name="buildSettingInfo">ビルド設定種別</param>
        /// <returns>ビルドルール(build)の文字列</returns>
        private string GetVsSolutionBuilderNinjaBuildRuleInstance(
            string spec, string filePath, BuildSettingInfo buildSettingInfo,
            List<string> currentResultsFileList,
            ProjectFile projectFile)
        {
            var sourcesFileList = new List<string>();
            var resultsFileList = new List<string>();

            switch (this.VsSolutionBuilderNinjaBuildRulePatternImpl)
            {
                // Buildもその他のtargetも実施しない場合はルールを記載しない。
                case VsSolutionBuilderNinjaBuildRulePattern.NOT_CREATE:
                    return string.Empty;
                case VsSolutionBuilderNinjaBuildRulePattern.CREATE_WITHOUT_SOURCES_AND_RESULTS:
                    resultsFileList.Add((buildCleanSequenceNo++).ToString());
                    break;
                case VsSolutionBuilderNinjaBuildRulePattern.CREATE_WITH_SOURCES_AND_RESULTS:
                default:
                    sourcesFileList.AddRange(projectFile.GetDependenciesOutputFiles(
                        buildSettingInfo.Configuration));
                    //  次回のslnファイルのビルド順番を保証するため、
                    //  今回のslnファイルの成果物をResultsに追加する。
                    resultsFileList = projectFile.GetOutputFiles(
                        buildSettingInfo.Configuration);
                    currentResultsFileList.AddRange(resultsFileList);
                    break;
            }

            var tempFolderName = CreateTempFolder();
            // /p:VisualStudioVersionの設定
            var vsVersion = GetMsbuildVsversionParameter(projectFile.Configuration);
            // /p:PreferredToolArchitectureの設定
            var preferredToolArchitecture
                = GetMsbuildPreferredToolArchitectureParameter(projectFile.Configuration);

            return
@"build " + string.Join(" ",
                resultsFileList.Select(
                outputFile => string.Format("{0}", SeparateDriveLetter(outputFile))))
 + @": MsbuildRuleBuild " + string.Join(" ",
                sourcesFileList.Select(
                objectFile => string.Format("{0}", SeparateDriveLetter(objectFile)))) + @"
    Platform = " + projectFile.Platform + @"
    Configuration = " + projectFile.Configuration + @"
    TargetFilePath = " + projectFile.ProjectFileName + @"
    TempFolderName = " + tempFolderName + @"
    PreferredToolArchitecture = " + preferredToolArchitecture + @"
    VsVersion = " + vsVersion + @"
";
        }

        /// <summary>
        /// フルパスのファイル名からドライブレターを除去したファイル名を返却する。
        /// </summary>
        /// <param name="filepath">フルパスのファイル名</param>
        private string SeparateDriveLetter(string filepath)
        {
            char[] separater = { ':' };
            var separatedFilename = filepath.Split(separater);
            if (separatedFilename.Length <= 1)
            {
                return separatedFilename[0];
            }
            return separatedFilename[1];
        }

        /// <summary>
        /// Process()を介して、Ninjaを実行する。
        /// </summary>
        /// <param name="ninjaRunPath">ninjaを実行するフォルダ</param>
        private void ExecuteNinja(string ninjaRunPath)
        {
            var exitCode = ExecuteProgram(
                BuildSystem.GetNinjaProgramFilePath(this.InstallPath),
                                new BuildSystem.Arguments()
                        .SetExecuteParallel(this.JobCount)
                        .SetKeepGoingJobsFails().ToString(),
                ninjaRunPath, true);
        }
        /// <summary>
        /// Process()を介して、PreBuildEvent実行前にビルドが必要なサンプルアプリをビルドする。
        /// </summary>
        /// <param name="solutionFile">ソリューションファイル</param>
        /// <param name="configuration">コンフィグレーション</param>
        /// <param name="platform">プラットフォーム</param>
        private void ExecutePrepareBuild(SolutionFile solutionfile,
            string configuration, string platform)
        {
            var tempFolderName = CreateTempFolder();
            string[] args = { solutionfile.FilePath,
                configuration,
                platform,
                this.IsCiMode.ToString(),
                tempFolderName,
                solutionfile.FilePath,
                "/m /nologo /v:minimal /t:Build",
                "/p:Configuration=" + configuration,
                "/p:Platform=" + platform };
            var exitCode = ExecuteProgram(
                this.MSBuildWrapperPath,
                string.Join(" ", args),
                Path.GetDirectoryName(solutionfile.FilePath), false);
        }
        /// <summary>
        /// Process()を介して、MSBuildでC#のプロジェクトファイルをビルドする。
        /// </summary>
        /// <param name="projectFile">C#のプロジェクトファイル</param>
        private void ExecuteCsProj(ProjectFile projectFile)
        {
            //  依存するcsprojの事前ビルド
            foreach (var csproj in projectFile.DependenciesCsproj)
            {
                var tempFolderName = CreateTempFolder();
                string[] args = { csproj,
                    projectFile.Configuration,
                    projectFile.Platform,
                    this.IsCiMode.ToString(),
                    tempFolderName,
                    csproj,
                    "/m /nologo /v:minimal /t:Build",
                    "/p:Configuration=" + projectFile.Configuration,
                    "/p:Platform=" + projectFile.Platform };
                var exitCode = ExecuteProgram(
                    this.MSBuildWrapperPath,
                    string.Join(" ", args),
                    Path.GetDirectoryName(projectFile.ProjectFileName), false);
            }
        }
        /// <summary>
        /// プロジェクトファイル単位でPreBuildEvent/PostBuildEventが必要な、
        /// Configuration、Platformをリストアップする。
        /// </summary>
        /// <param name="projectFile">PreBuildEvent/PostBuildEventを実行するプロジェクトファイル</param>
        private void ListupPrePostBuildEvent(ProjectFile projectFile)
        {
            // プロジェクトを解析したインスタンスを再度使用
            var project = projectFile.ProjectInstanceImpl;

            // 直列実行する PreBuildEvent
            if (project.Items.Where(x => x.ItemType == "PreBuildEvent").Any(x => x.Metadata.Any(y => y.Name == "Command")))
            {
                var ProjectFileListPreSerial = new List<ProjectFile>();

                if (this.ProjfileDictionaryForPreBuildEventSerial.ContainsKey(projectFile.ProjectFileName)
                    == true)
                {
                    ProjectFileListPreSerial
                        = this.ProjfileDictionaryForPreBuildEventSerial[projectFile.ProjectFileName];
                }

                var contains = false;
                foreach (var projItem in ProjectFileListPreSerial)
                {
                    if ((projItem.Configuration == projectFile.Configuration) &&
                         (projItem.Platform == projectFile.Platform))
                    {
                        contains = true;
                        break;
                    }
                }

                if (contains == false)
                {
                    ProjectFileListPreSerial.Add(projectFile);
                    this.ProjfileDictionaryForPreBuildEventSerial[projectFile.ProjectFileName]
                        = ProjectFileListPreSerial;
                }
            }

            // 直列実行する PostBuildEvent
            if (project.Items.Where(x => x.ItemType == "PostBuildEvent").Any(x => x.Metadata.Any(y => y.Name == "Command")))
            {
                var ProjectFileListPostSerial = new List<ProjectFile>();

                if (this.ProjfileDictionaryForPostBuildEventSerial.ContainsKey(projectFile.ProjectFileName)
                    == true)
                {
                    ProjectFileListPostSerial
                        = this.ProjfileDictionaryForPostBuildEventSerial[projectFile.ProjectFileName];
                }

                var contains = false;
                foreach (var projItem in ProjectFileListPostSerial)
                {
                    if ((projItem.Configuration == projectFile.Configuration) &&
                         (projItem.Platform == projectFile.Platform))
                    {
                        contains = true;
                        break;
                    }
                }

                if (contains == false)
                {
                    ProjectFileListPostSerial.Add(projectFile);
                    this.ProjfileDictionaryForPostBuildEventSerial[projectFile.ProjectFileName]
                        = ProjectFileListPostSerial;
                }
            }
        }
        /// <summary>
        /// Process()を介して、MSBuildのPreBuildEventをプロジェクトファイル単位で直列実行する。
        /// </summary>
        private void ExecutePreBuildEvent()
        {
            foreach (var oneDictionary in ProjfileDictionaryForPreBuildEventSerial)
            {
                foreach (var projectFile in oneDictionary.Value)
                {
                    var tempFolderName = CreateTempFolder();
                    string[] args = { projectFile.ProjectFileName,
                        projectFile.Configuration,
                        projectFile.Platform,
                        this.IsCiMode.ToString(),
                        tempFolderName,
                        projectFile.ProjectFileName,
                        "/m /nologo /v:minimal",
                        "/t:PrepareForBuild,PreBuildEvent",
                        "/p:Configuration=" + projectFile.Configuration,
                        "/p:Platform=" + projectFile.Platform,
                        GetMsbuildPreferredToolArchitectureParameter(projectFile.Configuration),
                        GetMsbuildVsversionParameter(projectFile.Configuration) };
                    var exitCode = ExecuteProgram(
                        this.MSBuildWrapperPath,
                        string.Join(" ", args),
                        Path.GetDirectoryName(projectFile.ProjectFileName), false);
                }
            }
        }
        /// <summary>
        /// Process()を介して、MSBuildのPostBuildEventをプロジェクトファイル単位で直列実行する。
        /// </summary>
        private void ExecutePostBuildEvent()
        {
            foreach (var oneDictionary in ProjfileDictionaryForPostBuildEventSerial)
            {
                foreach (var projectFile in oneDictionary.Value)
                {
                    var tempFolderName = CreateTempFolder();
                    string[] args = { projectFile.ProjectFileName,
                        projectFile.Configuration,
                        projectFile.Platform,
                        this.IsCiMode.ToString(),
                        tempFolderName,
                        projectFile.ProjectFileName,
                        "/m /nologo /v:minimal",
                        "/t:PostBuildEvent",
                        "/p:Configuration=" + projectFile.Configuration,
                        "/p:Platform=" + projectFile.Platform,
                        GetMsbuildPreferredToolArchitectureParameter(projectFile.Configuration),
                        GetMsbuildVsversionParameter(projectFile.Configuration) };
                    var exitCode = ExecuteProgram(
                        this.MSBuildWrapperPath,
                        string.Join(" ", args),
                        Path.GetDirectoryName(projectFile.ProjectFileName), false);
                }
            }
        }
        /// <summary>
        /// Process()を介して、指定したプログラムを実行する。
        /// </summary>
        /// <param name="filename">実行ファイル</param>
        /// <param name="argument">実行時引数</param>
        /// <param name="workDirectory">作業フォルダ</param>
        /// <param name="isNinjaRun">Ninja実行かどうかのフラグ</param>
        private int ExecuteProgram(string filename, string argument,
                string workDirectory, bool isNinjaRun)
        {
            var consoleList = new List<string>();

            using (var proc = new Process())
            {
                proc.StartInfo = new ProcessStartInfo()
                {
                    FileName = filename,
                    Arguments = argument,
                    WorkingDirectory = workDirectory,
                    UseShellExecute = false,
                    CreateNoWindow = true,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true,
                };

                var handler = new DataReceivedEventHandler(
                    (object obj, DataReceivedEventArgs args) =>
                    {
                        if (args.Data == null)
                        {
                            return;
                        }

                        if (isNinjaRun == false)
                        {
                            //  標準出力、標準エラーをスレッドセーフにするため、
                            //  リストの情報を保存する。
                            lock (consoleList)
                            {
                                consoleList.Add(args.Data);
                            }
                        }
                        else
                        {
                            Console.WriteLine(args.Data);
                        }
                    });

                proc.OutputDataReceived += handler;
                proc.ErrorDataReceived += handler;
                proc.Start();
                proc.BeginOutputReadLine();
                proc.BeginErrorReadLine();
                proc.WaitForExit();

                if (isNinjaRun == false)
                {
                    //  標準出力、標準エラーをスレッドセーフにして、
                    //  メッセージの整合性を確保する。
                    lock (lockObject)
                    {
                        consoleList.ForEach(x => Console.WriteLine(x));
                    }
                }
                return proc.ExitCode;
            }
        }
        /// <summary>
        /// 異なるslnファイルが共通のプロジェクトファイルを使用した場合に
        /// 成果物の重複有無を判断し、重複する場合はルール作成対象外とする。
        /// NXの場合のみ、成果物の重複有無を判断し、重複する場合はルール作成対象外とする。
        /// </summary>
        /// <note>
        /// 以下重複例
        /// FontDemo-spec.Generic.autogen.vs2017.sln
        /// Ui2dDemo-spec.Generic.autogen.vs2017.sln
        /// 　上記２slnファイルが共通で使用するプロジェクトファイルは
        /// 　　libnns_gfx-spec.Generic.autogen.vcxproj
        /// 　　libnns_hid-spec.Generic.autogen.vcxproj
        /// 以下重複例
        /// Demo1-spec.NX.autogen.vs2017.sln platform=NX32 configuration=Debug
        /// Demo1-spec.NX.autogen.vs2015.sln platform=NX32 configuration=Debug
        /// </note>
        /// <param name="file">ソリューションファイルパス</param>
        /// <param name="configuration">ビルドの種類</param>
        /// <param name="platform">プラットフォームの種類</param>
        /// <returns>成果物の重複判断結果</returns>
        private bool IsRuledProjectInfo(string file, string configuration, string platform)
        {
            var ruledProjectInfo = new RuledProjectInfo(
                file, configuration, platform);
            if (this.RuledProjectInfoList.Contains(ruledProjectInfo) == true)
            {
                return false;
            }
            else
            {
                this.RuledProjectInfoList.Add(ruledProjectInfo);
                return true;
            }
        }
        /// <summary>
        /// ConfigurationからMSBuildの/p:VisualStudioVersionを取得する。
        /// </summary>
        /// <param name="configuration">ビルドの種類</param>
        /// <returns>MSBuildの/p:VisualStudioVersion文字列</returns>
        private string GetMsbuildVsversionParameter(string configuration)
        {
            // /p:VisualStudioVersionの設定
            var vsVersion = MsbuildToolsetVersion140;
            if (configuration.Contains(MsbuildToolsetVersion2015) == true)
            {
                vsVersion = MsbuildToolsetVersion140;
            }
            else if (configuration.Contains(MsbuildToolsetVersion2017) == true)
            {
                vsVersion = MsbuildToolsetVersion150;
            }
            else
            {
                vsVersion = MsbuildToolsetVersion150;
            }
            return "/p:VisualStudioVersion=" + vsVersion;
        }
        /// <summary>
        /// ConfigurationからMSBuildの/p:PreferredToolArchitectureを取得する。
        /// </summary>
        /// <param name="configuration">ビルドの種類</param>
        /// <returns>MSBuildの/p:PreferredToolArchitecture文字列</returns>
        private string GetMsbuildPreferredToolArchitectureParameter(string configuration)
        {
            // /p:PreferredToolArchitectureの設定
            if (configuration.Contains(MsbuildToolsetVersion2017) == true)
            {
                return "/p:PreferredToolArchitecture=x64";
            }
            else
            {
                return string.Empty;
            }
        }
        /// <summary>
        /// エラー発生したvcxprojファイルをTeamCity形式のログでエラー報告する。
        /// </summary>
        /// <param name="buildSettingInfo">エラー発生したビルド対象のconfigurationの組み合わせ
        /// </param>
        private void ErrorOccurredSlnReport(BuildSettingInfo buildSettingInfo)
        {
            if (IsCiMode == false)
            {
                return;
            }

            var testName =
            $"BuildVerificationTest({buildSettingInfo.Configuration}|{buildSettingInfo.Platform})";

            foreach (var projectFile in buildSettingInfo.ProjectFiles)
            {
                if (projectFile.IsErrorOccurred == true)
                {
                    Console.WriteLine(TeamCityMessage.EnterMessage(
                        projectFile.ProjectFileName, testName));
                    Console.WriteLine(TeamCityMessage.ExitMessage(
                        projectFile.ProjectFileName, testName, projectFile.ErrorOccurredMessage));
                }
            }
        }
        /// <summary>
        /// PreBuildEvent前のビルドで指定されたにも関わらずビルド対象とならなかったslnファイルを
        /// TeamCity形式のログでエラー報告する。
        /// </summary>
        private void UnmatchedSlnReport()
        {
            if (IsCiMode == false)
            {
                return;
            }

            foreach (var preBuildSolution in VsPrebuildSolutionList)
            {
                if (preBuildSolution.IsMatched == false)
                {
                    var testName = $"BuildVerificationTest({preBuildSolution.Configuration}|{preBuildSolution.Platform})";
                    Console.WriteLine(TeamCityMessage.EnterMessage(
                        preBuildSolution.FileName + "."
                            + preBuildSolution.VsVersion, testName));
                    Console.WriteLine(TeamCityMessage.ExitMessage(
                        preBuildSolution.FileName + "."
                            + preBuildSolution.VsVersion, testName, "not exist."));
                }
            }
        }
        /// <summary>
        /// 環境変数のTMPフォルダを取得し、フォルダを作成する。
        /// </summary>
        private string CreateTempFolder()
        {
            // MSBuild 実行回数
            this.MsbuildRunCount++;

            //  環境変数のTMPフォルダを取得する。
            var tempFolderName = (Environment.GetEnvironmentVariable(
                EnviromentVariableTmp) ?? ".");
            tempFolderName = Path.Combine(tempFolderName,
                DateTime.Now.ToString("yyyyMMddHHmmss") + "-" + this.MsbuildRunCount.ToString());

            // 10回 (10s) までのリトライを許容する
            for (var retryCount = 0; retryCount < RetryCount; retryCount++)
            {
                try
                {
                    Directory.CreateDirectory(tempFolderName);
                    break;
                }
                catch (IOException)
                {
                    // 別のプロセスが使用されている可能性がある
                    // 1s 待ってから再試行
                    System.Threading.Thread.Sleep(WaitTimeBeforeRetry);
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.ToString());
                    break;
                }
            }

            // 一時フォルダ名をリストにストックする。
            this.TmpFolderList.Add(tempFolderName);

            return tempFolderName;
        }
        /// <summary>
        /// 一時フォルダリスト内のフォルダをすべて削除する。
        /// </summary>
        private void DeleteFolder()
        {
            // フォルダを削除する前に 10s 待つ
            System.Threading.Thread.Sleep(WaitTimeBeforeDeleteFolder);

            foreach (var folderName in this.TmpFolderList)
            {
                DeleteOneFolder(folderName);
            }
        }
        /// <summary>
        /// 指定したフォルダを削除する。
        /// </summary>
        /// <param name="folderName">削除するフォルダ名</param>
        private void DeleteOneFolder(string folderName)
        {
            var directoryInfo = new DirectoryInfo(folderName);

            // 読み取り専用属性を解除する。
            UnsetReadOnly(directoryInfo);

            // 10回 (10s) までのリトライを許容する
            for (var retryCount = 0; retryCount < RetryCount; retryCount++)
            {
                try
                {
                    // フォルダを削除する。
                    directoryInfo.Delete(true);
                    break;
                }
                catch (IOException)
                {
                    // 別のプロセスが使用されている可能性がある
                    // 1s 待ってから再試行
                    System.Threading.Thread.Sleep(WaitTimeBeforeRetry);
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.ToString());
                    break;
                }
            }
        }
        /// <summary>
        /// 読み取り専用属性を解除する。
        /// </summary>
        private void UnsetReadOnly(DirectoryInfo directoryInfo)
        {
            FileInfo[] fileInfoList = null;

            // 10回 (10s) までのリトライを許容する
            for (var retryCount = 0; retryCount < RetryCount; retryCount++)
            {
                try
                {
                    fileInfoList = directoryInfo.GetFiles();
                    break;
                }
                catch (DirectoryNotFoundException e)
                {
                    Console.WriteLine(e.ToString());

                    // 指定したフォルダが存在しない場合ループを抜ける
                    break;
                }
                catch (Exception e)
                {
                    Console.WriteLine(e.ToString());

                    // 1s 待ってから再試行
                    System.Threading.Thread.Sleep(WaitTimeBeforeRetry);
                }
            }

            // フォルダ内のファイルの読み取り専用属性を解除する。
            foreach (var fileInfo in fileInfoList)
            {
                if ((fileInfo.Attributes & FileAttributes.ReadOnly) == FileAttributes.ReadOnly)
                {
                    fileInfo.Attributes = FileAttributes.Normal;
                }
            }

            // サブフォルダの読み取り専用属性を解除する。
            foreach (var subFolder in directoryInfo.GetDirectories())
            {
                UnsetReadOnly(subFolder);
            }

            // フォルダの読み取り専用属性を解除する。
            if ((directoryInfo.Attributes & FileAttributes.ReadOnly) == FileAttributes.ReadOnly)
            {
                directoryInfo.Attributes = FileAttributes.Directory;
            }
        }
        /// <summary>
        /// PreBuildEvent実施前にビルドが必要なslnファイル名などの情報を
        /// 　環境変数から取得しリスト化して返却する。
        /// </summary>
        private static List<string> GetVsExcludeProjectNameList()
        {
            // PreBuildEvent実施前にビルドが必要なファイル名の取得
            var returnList = PreBuildSolution.GetPreBuildArgumentList(
                PreBuildSolution.EnviromentVariableVsPrebuildNames, string.Empty);

            return returnList;
        }
        private class PreBuildSolution
        {
            /// <summary>
            /// PreBuildEvent実施前にビルドが必要なファイル名と
            /// Platform、Configurationを指定する環境変数名
            /// </summary>
            internal const string EnviromentVariableVsPrebuildNames = "VS_PREBUILD_NAMES";
            private const string EnviromentVariableVsPrebuildPlatforms = "VS_PREBUILD_PLATFORMS";
            private const string EnviromentVariableVsPrebuildConfigrations =
                                                                    "VS_PREBUILD_CONFIGURATIONS";
            private const string EnviromentVariableVsVersions = "VS_PREBUILD_VSVERSION";
            private const string VsPrebuildConfigrationDefault = "Develop";
            internal string FileName { get; set; }
            internal string Platform { get; set; }
            internal string Configuration { get; set; }
            internal string VsVersion { get; set; }
            internal bool IsMatched { get; set; }

            /// <summary>
            /// コンストラクタ
            /// </summary>
            /// <param name="filename">ビルドするslnファイル名</param>
            /// <param name="platform">ビルドするplatform</param>
            /// <param name="configuration">ビルドするconfigration</param>
            /// <param name="vsVerion">ビルドするVSのバージョン</param>
            private PreBuildSolution(string filename, string platform,
                string configuration, string vsVersion)
            {
                this.FileName = filename;
                this.Platform = platform;
                this.Configuration = configuration;
                this.VsVersion = vsVersion;
                this.IsMatched = false;
            }

            /// <summary>
            /// PreBuildEvent実施前にビルドが必要なslnファイル名などの情報を
            /// 　環境変数から取得しリスト化して返却する。
            /// </summary>
            internal static List<PreBuildSolution> GetPreBuildSolutionList()
            {
                var returnList = new List<PreBuildSolution>();

                // PreBuildEvent実施前にビルドが必要なファイル名の取得
                var vsPrebuildNames = GetPreBuildArgumentList(
                    EnviromentVariableVsPrebuildNames, string.Empty);
                if (vsPrebuildNames.Count == 0)
                {
                    return returnList;
                }

                // PreBuildEvent実施前にビルドが必要なPlatformの取得
                var vsPrebuildPlatforms = GetPreBuildArgumentList(
                    EnviromentVariableVsPrebuildPlatforms, PlatformDefinition.WIN32);

                // PreBuildEvent実施前にビルドが必要なConfigurationの取得
                var vsPrebuildConfigrations = GetPreBuildArgumentList(
                    EnviromentVariableVsPrebuildConfigrations, VsPrebuildConfigrationDefault);

                // PreBuildEvent実施前にビルドが必要なVsVersionの取得
                var vsPrebuildVsVersions = GetPreBuildArgumentList(
                    EnviromentVariableVsVersions, BuildRuleMaker.MsbuildToolsetVersion2015);

                // env.VS_PREBUILD_NAMES × env.VS_PREBUILD_CONFIGURATIONS
                //  × env.VS_PREBUILD_PLATFORMS × env.VS_PREBUILD_VSVERSION
                // の組み合わせをビルド対象とする。
                foreach (var VsPrebuildName in vsPrebuildNames)
                {
                    foreach (var VsPrebuildPlatform in vsPrebuildPlatforms)
                    {
                        foreach (var VsPrebuildConfigration in vsPrebuildConfigrations)
                        {
                            foreach (var VsPrebuildVsVersion in vsPrebuildVsVersions)
                            {
                                returnList.Add(new PreBuildSolution(
                                VsPrebuildName, VsPrebuildPlatform,
                                    VsPrebuildConfigration, VsPrebuildVsVersion));
                            }
                        }
                    }
                }

                return returnList;
            }
            /// <summary>
            /// 環境変数から取得しリスト化して返却する。
            /// </summary>
            internal static List<string> GetPreBuildArgumentList(
                string enviromentName, string defautlValue)
            {
                var envValue = (Environment.GetEnvironmentVariable(
                    enviromentName) ?? defautlValue).Trim(',');
                var returnList = (envValue != string.Empty ? envValue : defautlValue)
                            .Split(',').ToList().Distinct().ToList();
                returnList.RemoveAll(value => value == string.Empty);
                return returnList;
            }
        }
    }
}
