﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using YamlDotNet.Core;
    using YamlDotNet.RepresentationModel;

    /// <summary>
    /// YAML パーサーを定義します。
    /// </summary>
    internal static class YamlParser
    {
        /// <summary>
        /// YAML ファイルをパースします。
        /// </summary>
        /// <param name="fileName">YAML ファイルの絶対パス文字列です。</param>
        /// <returns>YAML ファイルから取得したテストステートメントの配列です。</returns>
        internal static object[] Read(string fileName)
        {
            return (new Parser()).ParseFile(fileName);
        }

        private sealed class Parser
        {
            private static readonly IReadOnlyDictionary<string, object>
                SymbolMap = new Dictionary<string, object>()
            {
                { "null", null },
                { "Null", null },
                { "NULL", null },
                { "~",    null },
                { "true", true },
                { "True", true },
                { "TRUE", true },
                { "false", false },
                { "False", false },
                { "FALSE", false }
            };

            internal Parser()
            {
                this.CurrentYamlPath = string.Empty;

                this.IncludePaths = new HashSet<string>();

                this.AnchorMap = new Dictionary<string, object>();

                this.VariableManager = new VariableManager();
            }

            private string CurrentYamlPath { get; set; }

            private HashSet<string> IncludePaths { get; set; }

            private Dictionary<string, object> AnchorMap { get; set; }

            private VariableManager VariableManager { get; set; }

            internal object[] ParseFile(string fileName)
            {
                this.CurrentYamlPath = fileName;

                this.IncludePaths.Add(fileName);

                var docs = new List<object>();

                using (var sr = new StreamReader(fileName, Encoding.UTF8))
                {
                    var ys = new YamlStream();

                    try
                    {
                        ys.Load(sr);
                    }
                    catch (YamlException ex)
                    {
                        var message = string.Format(
                            "failed to load '{0}': {1}", fileName, ex.Message);

                        throw new ArgumentException(message, ex);
                    }

                    foreach (var doc in ys.Documents)
                    {
                        docs.Add(this.ParseNode(doc.RootNode));
                    }
                }

                return docs.ToArray();
            }

            private static string GetErrorMessage(
                string message, string fileName, int line, int offset)
            {
                return string.Format(
                    "failed to load '{0}': {1} (Line:{2}, Offset:{3})",
                    fileName, message, line, offset);
            }

            private static string GetUnexpectedArgumentLengthErrorMessage(
                string fileName, Mark mark, string macroName, int expected,
                int given)
            {
                var message = string.Format(
                    "'{0}' expects {1} argument(s), given {2}",
                    macroName, expected, given);

                return GetErrorMessage(
                    message, fileName, mark.Line, mark.Column);
            }

            private static string GetUnexpectedNodeTypeErrorMessage(
                string fileName, Mark mark)
            {
                return GetErrorMessage(
                    "unexpected node type", fileName, mark.Line, mark.Column);
            }

            private Dictionary<string, object> ParseMappingNode(
                YamlMappingNode node)
            {
                var dict = new Dictionary<string, object>();

                foreach (var entry in node.Children)
                {
                    string key = ((YamlScalarNode)entry.Key).Value;

                    dict[key] = this.ParseNode(entry.Value);
                }

                return dict;
            }

            private object ParseSequenceNode(YamlSequenceNode nodes)
            {
                if (nodes.Count() > 0)
                {
                    switch (nodes.First().Tag)
                    {
                        case Macro.Include:
                            return this.IncludeFile(nodes);

                        case Macro.Load:
                            return this.LoadAnchoredNode(nodes);

                        case Macro.Add:
                        case "! ": // YamlDotNet 4.1 で add マクロが正しくパースされない問題のワークアラウンド
                            return this.AddNodes(nodes);
                    }
                }

                return nodes.Select(node => this.ParseNode(node))
                            .ToList<object>();
            }

            private object ParseScalarNode(YamlScalarNode node)
            {
                var value = node.Value;

                return SymbolMap.ContainsKey(value) ? SymbolMap[value] : value;
            }

            private object ParseNode(YamlNode node)
            {
                object parsedNode = null;

                if (node is YamlMappingNode)
                {
                    var mappingNode = (YamlMappingNode)node;

                    parsedNode = this.ParseMappingNode(mappingNode);
                }
                else if (node is YamlSequenceNode)
                {
                    var sequenceNode = (YamlSequenceNode)node;

                    parsedNode = this.ParseSequenceNode(sequenceNode);
                }
                else
                {
                    var scalarNode = (YamlScalarNode)node;

                    parsedNode = this.ParseScalarNode(scalarNode);
                }

                if (!string.IsNullOrEmpty(node.Anchor))
                {
                    this.AnchorMap[node.Anchor] = parsedNode;
                }

                return parsedNode;
            }

            private object IncludeFile(YamlSequenceNode nodes)
            {
                string yamlPath = this.CurrentYamlPath;

                int nodeCount = nodes.Count();

                if (nodeCount != 1)
                {
                    string message = GetUnexpectedArgumentLengthErrorMessage(
                        yamlPath, nodes.Start, Macro.Include, 1, nodeCount);

                    throw new ArgumentException(message);
                }

                var argsNode = nodes.First();

                var fileName = this.ParseNode(argsNode);

                if (!(fileName is string))
                {
                    string message = GetUnexpectedNodeTypeErrorMessage(
                        yamlPath, argsNode.Start);

                    throw new ArgumentException(message);
                }

                int line = argsNode.Start.Line, offset = argsNode.Start.Column;

                var expandedName = this.VariableManager.ExpandVariables(
                    (string)fileName);

                if (string.IsNullOrEmpty(expandedName))
                {
                    string message = GetErrorMessage(
                        "empty value was specified", yamlPath, line, offset);

                    throw new ArgumentException(message);
                }

                var fullName = Path.Combine(
                    Path.GetDirectoryName(yamlPath), expandedName);

                if (!File.Exists(fullName))
                {
                    var message = string.Format(
                        "'{0}' does not exist", fullName);

                    message = GetErrorMessage(message, yamlPath, line, offset);

                    throw new FileNotFoundException(message);
                }

                if (this.IncludePaths.Contains(fullName))
                {
                    var message = string.Format(
                        "'{0}' is circularly referenced", fullName);

                    message = GetErrorMessage(message, yamlPath, line, offset);

                    throw new ArgumentException(message);
                }

                try
                {
                    return this.ParseFile(fullName).FirstOrDefault();
                }
                finally
                {
                    this.CurrentYamlPath = yamlPath;

                    this.IncludePaths.Remove(fullName);
                }
            }

            private object LoadAnchoredNode(YamlSequenceNode nodes)
            {
                string yamlPath = this.CurrentYamlPath;

                int nodeCount = nodes.Count();

                if (nodeCount != 1)
                {
                    string message = GetUnexpectedArgumentLengthErrorMessage(
                        yamlPath, nodes.Start, Macro.Load, 1, nodeCount);

                    throw new ArgumentOutOfRangeException(message);
                }

                var argNode = nodes.First();

                var anchor = this.ParseNode(argNode);

                if (!(anchor is string))
                {
                    string message = GetUnexpectedNodeTypeErrorMessage(
                        yamlPath, argNode.Start);

                    throw new ArgumentException(message);
                }

                var key = (string)anchor;

                if (this.AnchorMap.ContainsKey(key))
                {
                    return this.AnchorMap[key];
                }
                else
                {
                    int line = argNode.Start.Line;

                    int offset = argNode.Start.Column;

                    var message = "anchor '{0}' not defined";

                    message = string.Format(message, key);

                    message = GetErrorMessage(message, yamlPath, line, offset);

                    throw new ArgumentException(message);
                }
            }

            private object AddNodes(YamlSequenceNode nodes)
            {
                string yamlPath = this.CurrentYamlPath;

                int nodeCount = nodes.Count();

                var lhs = this.ParseNode(nodes.First());

                if (lhs is string)
                {
                    var sb = new StringBuilder((string)lhs);

                    for (var i = 1; i < nodeCount; ++i)
                    {
                        var rhs = this.ParseNode(nodes.ElementAt(i));

                        if (rhs is string)
                        {
                            sb.Append((string)rhs);
                        }
                        else
                        {
                            string message = GetUnexpectedNodeTypeErrorMessage(
                                yamlPath, nodes.ElementAt(i).Start);

                            throw new ArgumentException(message);
                        }
                    }

                    return sb.ToString();
                }
                else if (lhs is List<object>)
                {
                    var list = new List<object>((List<object>)lhs);

                    for (var i = 1; i < nodeCount; ++i)
                    {
                        var rhs = this.ParseNode(nodes.ElementAt(i));

                        if (rhs is List<object>)
                        {
                            list.AddRange((List<object>)rhs);
                        }
                        else
                        {
                            list.Add(rhs);
                        }
                    }

                    return list;
                }
                else if (lhs is Dictionary<string, object>)
                {
                    var dict = (Dictionary<string, object>)lhs;

                    dict = new Dictionary<string, object>(dict);

                    for (var i = 1; i < nodeCount; ++i)
                    {
                        var rhs = this.ParseNode(nodes.ElementAt(i));

                        if (rhs is Dictionary<string, object>)
                        {
                            var xs = (Dictionary<string, object>)rhs;

                            foreach (KeyValuePair<string, object> x in xs)
                            {
                                dict[x.Key] = x.Value;
                            }
                        }
                        else
                        {
                            string message = GetUnexpectedNodeTypeErrorMessage(
                                yamlPath, nodes.ElementAt(i).Start);

                            throw new ArgumentException(message);
                        }
                    }

                    return dict;
                }
                else
                {
                    string message = GetUnexpectedNodeTypeErrorMessage(
                        yamlPath, nodes.First().Start);

                    throw new ArgumentException(message);
                }
            }

            private static class Macro
            {
                internal const string Include = "!include";

                internal const string Load = "!*";

                internal const string Add = "!+";
            }
        }
    }
}
