﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text.RegularExpressions;

    /// <summary>
    /// テストリスト検証用の関数を定義します。
    /// </summary>
    internal static class TestListValidation
    {
        /// <summary>
        /// マッピングノードが有効なマッピングノードをキーと対応付けて持つかどうかを示す値を返します。要素の型も確認します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <typeparam name="T">マッピングノードの要素の型です。</typeparam>
        /// <returns>マッピングノードが有効なマッピングノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidMappingNodeOf<T>(
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsMappingNodeOf<T>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが不正なマッピングノードをキーと対応付けて持つかどうかを示す値を返します。要素の型も確認します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <typeparam name="T">マッピングノードの要素の型です。</typeparam>
        /// <returns>マッピングノードが不正なマッピングノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongMappingNodeOf<T>(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsMappingNodeOf<T>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが有効なシーケンスノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが有効なシーケンスノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidSequenceNode(
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsExceptedTypeNode<
                List<object>>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが不正なシーケンスノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが不正なシーケンスノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongSequenceNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsExceptedTypeNode<
                List<object>>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが有効なシーケンスノードをキーと対応付けて持つかどうかを示す値を返します。要素の型も確認します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <typeparam name="T">シーケンスノードの要素の型です。</typeparam>
        /// <returns>マッピングノードが有効なシーケンスノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidSequenceNodeOf<T>(
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsSequenceNodeOf<T>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが不正なシーケンスノードをキーと対応付けて持つかどうかを示す値を返します。要素の型も確認します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <typeparam name="T">シーケンスノードの要素の型です。</typeparam>
        /// <returns>マッピングノードが不正なシーケンスノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongSequenceNodeOf<T>(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsSequenceNodeOf<T>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが有効なシーケンスノードをキーと対応付けて持つかどうかを示す値を返します。
        /// 要素がパス文字列であるかも確認します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが有効なシーケンスノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidSequencePathStringNode(
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsSequencePathStringNode(
                out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが不正なシーケンスノードをキーと対応付けて持つかどうかを示す値を返します。
        /// 要素がパス文字列であるかも確認します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが不正なシーケンスノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongSequencePathStringNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsSequencePathStringNode(
                out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが有効な型のノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <typeparam name="T">ノードの型です。</typeparam>
        /// <returns>マッピングノードが有効な型のノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidTypedNode<T>(
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsExceptedTypeNode<T>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが不正な型のノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <typeparam name="T">ノードの型です。</typeparam>
        /// <returns>マッピングノードが不正な型のノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongTypedNode<T>(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsExceptedTypeNode<T>(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが有効なパス文字列ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが有効なパス文字列ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidPathStringNode(
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsPathStringNode(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが不正なパス文字列ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが不正なパス文字列ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongPathStringNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsPathStringNode(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが有効な正規表現文字列ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが有効な正規表現文字列ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidRegexStringNode(
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsRegexStringNode(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが不正な正規表現文字列ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが不正な正規表現文字列ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongRegexStringNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsRegexStringNode(out message, mappingNode[key], key);
        }

        /// <summary>
        /// マッピングノードが有効なパターンノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <param name="regex">有効なパターンを表す正規表現です。</param>
        /// <returns>マッピングノードが有効なパターンノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidPatternStringNode(
            IReadOnlyDictionary<string, object> mappingNode, string key,
            Regex regex)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsPatternStringNode(
                out message, mappingNode[key], key, regex);
        }

        /// <summary>
        /// マッピングノードが不正なパターンノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <param name="regex">有効なパターンを表す正規表現です。</param>
        /// <returns>マッピングノードが不正なパターンノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongPatternStringNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key,
            Regex regex)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsPatternStringNode(
                out message, mappingNode[key], key, regex);
        }

        /// <summary>
        /// マッピングノードが不正なテスト名文字列ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <returns>マッピングノードが不正なテスト名文字列ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongTestNameStringNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            if (HasWrongTypedNode<string>(out message, mappingNode, key))
            {
                return true;
            }

            if (HasNode(out message, mappingNode, key))
            {
                var testName = (string)mappingNode[key];

                if (testName == string.Empty)
                {
                    message = $"Empty string for '{key}'";

                    return true;
                }

                if (Path.GetInvalidPathChars().Concat(new[] { '/', '\\' })
                        .Any(c => testName.Contains(c)))
                {
                    message =
                        $"Invalid test name string '{testName}' for '{key}'";

                    return true;
                }
            }

            return false;
        }

        /// <summary>
        /// マッピングノードが有効な整数ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <param name="minValue">整数の下限です。</param>
        /// <param name="maxValue">整数の上限です。</param>
        /// <returns>マッピングノードが有効な整数ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidIntegerNode(
            IReadOnlyDictionary<string, object> mappingNode, string key,
            int minValue = int.MinValue, int maxValue = int.MaxValue)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsIntegerNode(
                out message, mappingNode[key], key, minValue, maxValue);
        }

        /// <summary>
        /// マッピングノードが不正な整数ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <param name="minValue">整数の下限です。</param>
        /// <param name="maxValue">整数の上限です。</param>
        /// <returns>マッピングノードが不正な整数ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongIntegerNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key,
            int minValue = int.MinValue, int maxValue = int.MaxValue)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsIntegerNode(
                out message, mappingNode[key], key, minValue, maxValue);
        }

        /// <summary>
        /// マッピングノードが有効な選択肢ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <param name="choices">有効な選択肢です。</param>
        /// <returns>マッピングノードが有効な選択肢ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasValidChoiceNode(
            IReadOnlyDictionary<string, object> mappingNode, string key,
            IReadOnlyList<string> choices)
        {
            var message = string.Empty;

            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return IsChoiceNode(out message, mappingNode[key], key, choices);
        }

        /// <summary>
        /// マッピングノードが不正な選択肢ノードをキーと対応付けて持つかどうかを示す値を返します。
        /// </summary>
        /// <param name="message">不正なノードだと判断された理由を取得します。</param>
        /// <param name="mappingNode">マッピングノードです。</param>
        /// <param name="key">キーです。</param>
        /// <param name="choices">有効な選択肢です。</param>
        /// <returns>マッピングノードが不正な選択肢ノードをキーと対応付けて持つかどうかを示す値です。</returns>
        internal static bool HasWrongChoiceNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key,
            IReadOnlyList<string> choices)
        {
            if (!HasNode(out message, mappingNode, key))
            {
                return false;
            }

            return !IsChoiceNode(out message, mappingNode[key], key, choices);
        }

        private static string GetInvalidNodeTypeMessage(string key)
        {
            return $"Invalid node type for '{key}'";
        }

        private static string GetInvalidNodeValueMessage(
            string key, string value)
        {
            return $"Invalid value '{value}' for '{key}'";
        }

        private static string GetInvalidEntryTypeMessage(string key)
        {
            return $"Invalid node type for an entry of '{key}'";
        }

        private static bool HasNode(
            out string message,
            IReadOnlyDictionary<string, object> mappingNode, string key)
        {
            message = string.Empty;

            return mappingNode.ContainsKey(key);
        }

        private static bool IsExceptedTypeNode<T>(
            out string message, object node, string key)
        {
            var result = node is T;

            message = result ? string.Empty : GetInvalidNodeTypeMessage(key);

            return result;
        }

        private static bool IsMappingNodeOf<T>(
            out string message, object node, string key)
        {
            if (!IsExceptedTypeNode<
                    Dictionary<string, object>>(out message, node, key))
            {
                return false;
            }

            var result = ((Dictionary<string, object>)node)
                .All(x => x.Value is T);

            message = result ? string.Empty : GetInvalidEntryTypeMessage(key);

            return result;
        }

        private static bool IsSequenceNodeOf<T>(
            out string message, object node, string key)
        {
            if (!IsExceptedTypeNode<List<object>>(out message, node, key))
            {
                return false;
            }

            var result = ((List<object>)node).All(x => x is T);

            message = result ? string.Empty : GetInvalidEntryTypeMessage(key);

            return result;
        }

        private static bool IsSequencePathStringNode(
            out string message, object node, string key)
        {
            if (!IsExceptedTypeNode<List<object>>(out message, node, key))
            {
                return false;
            }

            foreach (object scalarNode in (List<object>)node)
            {
                if (!(scalarNode is string))
                {
                    message = GetInvalidEntryTypeMessage(key);

                    return false;
                }

                var path = (string)scalarNode;

                if (path == string.Empty)
                {
                    message = $"an entry of '{key}'";

                    message = $"Empty string for ${message}";

                    return false;
                }

                if (Path.GetInvalidPathChars().Any(c => path.Contains(c)))
                {
                    message = $"an entry of '{key}'";

                    message = $"Invalid path string '{path}' for {message}";

                    return false;
                }
            }

            return true;
        }

        private static bool IsPathStringNode(
            out string message, object node, string key)
        {
            if (!IsExceptedTypeNode<string>(out message, node, key))
            {
                return false;
            }

            var path = (string)node;

            if (path == string.Empty)
            {
                message = $"Empty string for '{key}'";

                return false;
            }

            if (Path.GetInvalidPathChars().Any(c => path.Contains(c)))
            {
                message = $"Invalid path string '{path}' for '{key}'";

                return false;
            }

            return true;
        }

        private static bool IsRegexStringNode(
            out string message, object node, string key)
        {
            if (!IsExceptedTypeNode<string>(out message, node, key))
            {
                return false;
            }

            var regex = (string)node;

            try
            {
                new Regex(regex);
            }
            catch
            {
                message = $"Invalid regex string '{regex}' for '{key}'";

                return false;
            }

            return true;
        }

        private static bool IsPatternStringNode(
            out string message, object node, string key, Regex regex)
        {
            if (!IsExceptedTypeNode<string>(out message, node, key))
            {
                return false;
            }

            var scalarNode = (string)node;

            var result = regex.IsMatch(scalarNode);

            message = result ?
                string.Empty : GetInvalidNodeValueMessage(key, scalarNode);

            return result;
        }

        private static bool IsIntegerNode(
            out string message,
            object node, string key, int minValue, int maxValue)
        {
            if (!IsExceptedTypeNode<string>(out message, node, key))
            {
                return false;
            }

            var scalarNode = (string)node;

            var value = 0;

            var result =
                int.TryParse(scalarNode, out value) &&
                minValue <= value && value <= maxValue;

            message = result ?
                string.Empty : GetInvalidNodeValueMessage(key, scalarNode);

            return result;
        }

        private static bool IsChoiceNode(
            out string message,
            object node, string key, IReadOnlyList<string> choices)
        {
            if (!IsExceptedTypeNode<string>(out message, node, key))
            {
                return false;
            }

            var scalarNode = (string)node;

            var result = choices.Contains(scalarNode);

            message = result ?
                string.Empty : GetInvalidNodeValueMessage(key, scalarNode);

            return result;
        }
    }
}
