﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.Collections.ObjectModel;
    using System.Linq;
    using System.Text.RegularExpressions;
    using Executer;
    using static TargetManager;

    /// <summary>
    /// 開発機をテストに割り当てます。
    /// </summary>
    internal sealed class TargetAssigner
    {
        /// <summary>
        /// TargetAssigner クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal TargetAssigner()
        {
            this.TargetManager = new TargetManager();

            this.Tests = new List<IEnumerable<TestContext>>();

            this.Observers = Enumerable.Empty<TestContext>();

            this.Table = new Dictionary<ulong, TargetEntry>();
        }

        /// <summary>
        /// 開発機の割り当て表を取得します。
        /// </summary>
        internal IReadOnlyDictionary<ulong, TargetEntry> AllocationTable
        {
            get
            {
                return new ReadOnlyDictionary<ulong, TargetEntry>(this.Table);
            }
        }

        private IList<IEnumerable<TestContext>> Tests { get; }

        private IEnumerable<TestContext> Observers { get; set; }

        private IDictionary<ulong, TargetEntry> Table { get; }

        private TargetManager TargetManager { get; }

        /// <summary>
        /// 開発機の割り当ての対象となるテストのコンテキストを追加します。
        /// </summary>
        /// <param name="testContexts">テストコンテキストの配列です。</param>
        internal void AddTests(IEnumerable<TestContext> testContexts)
        {
            this.Tests.Add(testContexts);
        }

        /// <summary>
        /// 開発機の割り当ての対象となるオブザーバのコンテキストを設定します。
        /// </summary>
        /// <param name="testContexts">テストコンテキストの配列です。</param>
        internal void SetObservers(IEnumerable<TestContext> testContexts)
        {
            this.Observers = testContexts;
        }

        /// <summary>
        /// 開発機の割り当てを実施します。
        /// </summary>
        /// <param name="entries">開発機のエントリです。</param>
        internal void Flush(IReadOnlyList<TargetEntry> entries)
        {
            IList<TestContext> testContexts = this.GetTargetTestContexts();

            KeyValuePair<ulong, IList<Func<TargetEntry, bool>>>[]
                conditions = this.GetConditions(testContexts).ToArray();

            if (entries.Count < conditions.Length)
            {
                TestContext testContext = null;

                if (entries.Count == 0)
                {
                    testContext =
                        GetTestContext(conditions.First().Key, testContexts);

                    testContext.ErrorMessage = "No target found";
                }
                else
                {
                    testContext =
                        GetTestContext(conditions.Last().Key, testContexts);

                    testContext.ErrorMessage = string.Format(
                        "Over the number of targets '{0}'", entries.Count);
                }

                testContext.ResultCode = ResultCode.NO_TARGET;

                throw new TestContextException(
                    testContext.ErrorMessage, testContext);
            }

            int[][] candidates =
                this.GetCandidates(testContexts, entries, conditions);

            int[] indices =
                this.ResolveCandidates(candidates, testContexts, conditions);

            this.Table.Clear();

            for (int i = 0; i < conditions.Length; ++i)
            {
                this.Table[conditions[i].Key] = entries[indices[i]];
            }

            foreach (TestContext testContext in testContexts)
            {
                var key = TestUnitInfo.EncodeUnitHash(
                    testContext.RoleType, testContext.UnitId);

                var value = this.Table[key];

                testContext.TargetName = value.Name;

                testContext.TargetInterface = value.Interface;

                testContext.TargetAddress = value.Address;
            }

            this.Tests.Clear();

            this.Observers = Enumerable.Empty<TestContext>();
        }

        private static TestContext GetTestContext(
            ulong hash, IList<TestContext> testContexts)
        {
            return testContexts.FirstOrDefault(x =>
                hash == TestUnitInfo.EncodeUnitHash(x.RoleType, x.UnitId));
        }

        private IList<TestContext> GetTargetTestContexts()
        {
            var testContexts = new List<TestContext>();

            foreach (IEnumerable<TestContext> xs in this.Tests)
            {
                testContexts.AddRange(
                    xs.Where(x => this.TargetManager.Supports(x)));
            }

            testContexts.AddRange(
                this.Observers.Where(x => this.TargetManager.Supports(x)));

            return testContexts;
        }

        private IDictionary<ulong, IList<Func<TargetEntry, bool>>>
            GetConditions(IList<TestContext> testContexts)
        {
            var conditions =
                new SortedDictionary<ulong, IList<Func<TargetEntry, bool>>>();

            var namePatterns = new Dictionary<ulong, ISet<string>>();

            var interfacePatterns = new Dictionary<ulong, ISet<string>>();

            foreach (TestContext testContext in testContexts)
            {
                var key = TestUnitInfo.EncodeUnitHash(
                    testContext.RoleType, testContext.UnitId);

                if (!conditions.ContainsKey(key))
                {
                    conditions[key] = new List<Func<TargetEntry, bool>>();
                }

                if (!string.IsNullOrEmpty(testContext.TargetNamePattern))
                {
                    ISet<string> names = null;

                    if (!namePatterns.TryGetValue(key, out names))
                    {
                        names = new HashSet<string>();

                        namePatterns[key] = names;
                    }

                    names.Add(testContext.TargetNamePattern);
                }

                if (!string.IsNullOrEmpty(testContext.TargetInterfacePattern))
                {
                    ISet<string> interfaces = null;

                    if (!interfacePatterns.TryGetValue(key, out interfaces))
                    {
                        interfaces = new HashSet<string>();

                        interfacePatterns[key] = interfaces;
                    }

                    interfaces.Add(testContext.TargetInterfacePattern);
                }
            }

            foreach (var pair in namePatterns)
            {
                IList<Func<TargetEntry, bool>> funcs = conditions[pair.Key];

                foreach (var pattern in pair.Value)
                {
                    var regex = new Regex(pattern, RegexOptions.Compiled);

                    funcs.Add(new Func<TargetEntry, bool>(
                        x => regex.IsMatch(x.Name)));
                }
            }

            foreach (var pair in interfacePatterns)
            {
                IList<Func<TargetEntry, bool>> funcs = conditions[pair.Key];

                foreach (var pattern in pair.Value)
                {
                    var regex = new Regex(pattern, RegexOptions.Compiled);

                    funcs.Add(new Func<TargetEntry, bool>(
                        x => regex.IsMatch(x.Interface)));
                }
            }

            return conditions;
        }

        private int[][] GetCandidates(
            IList<TestContext> testContexts,
            IReadOnlyList<TargetEntry> entries,
            KeyValuePair<ulong, IList<Func<TargetEntry, bool>>>[] conditions)
        {
            var candidates = new int[conditions.Length][];

            for (int i = 0; i < conditions.Length; ++i)
            {
                int[] indices = Enumerable.Range(0, entries.Count)
                    .Where(x => conditions[i].Value.All(f => f(entries[x])))
                    .ToArray();

                if (indices.Length == 0)
                {
                    TestContext testContext =
                        GetTestContext(conditions[i].Key, testContexts);

                    testContext.ErrorMessage =
                        "No target meets the conditions";

                    testContext.ResultCode = ResultCode.NO_TARGET;

                    throw new TestContextException(
                        testContext.ErrorMessage, testContext);
                }

                candidates[i] = indices;
            }

            return candidates;
        }

        private int[] ResolveCandidates(
            int[][] candidates,
            IList<TestContext> testContexts,
            KeyValuePair<ulong, IList<Func<TargetEntry, bool>>>[] conditions)
        {
            var results = new int[candidates.Length];

            if (0 < candidates.Length)
            {
                var checks = new bool[candidates.Max(xs => xs.Max()) + 1];

                var frames = new StackFrame[candidates.Length]
                    .Select(x => new StackFrame()).ToArray();

                var depth = 0;

                var maxDepth = 0;

                while (depth < candidates.Length)
                {
                    maxDepth = Math.Max(depth, maxDepth);

                    StackFrame frame = frames[depth];

                    int[] indices = candidates[depth];

                    if (frame.Steps)
                    {
                        frame.Steps = false;

                        frame.Offset += 1;

                        checks[results[depth]] = false;
                    }
                    else
                    {
                        int index = indices[frame.Offset];

                        if (checks[index])
                        {
                            frame.Offset += 1;
                        }
                        else
                        {
                            frame.Steps = true;

                            checks[index] = true;

                            results[depth] = index;

                            depth += 1;
                        }
                    }

                    if (frame.Offset == indices.Length)
                    {
                        if (0 < depth)
                        {
                            frame.Offset = 0;

                            depth -= 1;
                        }
                        else
                        {
                            TestContext testContext = GetTestContext(
                                conditions[maxDepth].Key, testContexts);

                            testContext.ErrorMessage = "Targets are exhausted";

                            testContext.ResultCode = ResultCode.NO_TARGET;

                            throw new TestContextException(
                                testContext.ErrorMessage, testContext);
                        }
                    }
                }
            }

            return results;
        }

        private class StackFrame
        {
            internal bool Steps { get; set; }

            internal int Offset { get; set; }
        }
    }
}
