﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Xml;
    using Executer;
    using Properties;

    /// <summary>
    /// テスト結果の要約を出力します。
    /// </summary>
    internal sealed class ResultSummarizer
    {
        /// <summary>
        /// ResultSummerizer クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal ResultSummarizer()
        {
            this.EnablesInstantSummary = true;

            this.EnablesFileSummary = false;

            this.Date = string.Empty;

            this.BranchName = string.Empty;

            this.CommitHash = string.Empty;

            this.ResultRootPath = string.Empty;

            this.TestRootPath = string.Empty;

            this.Platforms = new List<string>();

            this.BuildTypes = new List<string>();

            this.Modules = new List<string>();

            this.Categories = new List<string>();
        }

        /// <summary>
        /// 要約の即時出力を有効にするかどうかを示す値を取得または設定します。
        /// </summary>
        internal bool EnablesInstantSummary { get; set; }

        /// <summary>
        /// テストリスト中で言及されたファイルの要約を有効にするかどうかを示す値を取得します。
        /// </summary>
        internal bool EnablesFileSummary { get; set; }

        /// <summary>
        /// テストランナー実行開始時の日付を取得または設定します。
        /// </summary>
        internal string Date { get; set; }

        /// <summary>
        /// ブランチ名を取得または設定します。
        /// </summary>
        internal string BranchName { get; set; }

        /// <summary>
        /// コミットハッシュを取得または設定します。
        /// </summary>
        internal string CommitHash { get; set; }

        /// <summary>
        /// テスト結果ファイルの出力先のルートディレクトリの絶対パスを取得または設定します。
        /// </summary>
        internal string ResultRootPath { get; set; }

        /// <summary>
        /// テストリストの探索を行うディレクトリの絶対パスを取得または設定します。
        /// </summary>
        internal string TestRootPath { get; set; }

        /// <summary>
        /// 対象となるプラットフォーム名のリストを取得または設定します。
        /// </summary>
        internal IReadOnlyList<string> Platforms { get; set; }

        /// <summary>
        /// 対象となるビルドタイプ名のリストを取得または設定します。
        /// </summary>
        internal IReadOnlyList<string> BuildTypes { get; set; }

        /// <summary>
        /// 対象となるモジュール名のリストを取得または設定します。
        /// </summary>
        internal IReadOnlyList<string> Modules { get; set; }

        /// <summary>
        /// 対象となるテストカテゴリ名のリストを取得または設定します。
        /// </summary>
        internal IReadOnlyList<string> Categories { get; set; }

        private string SummaryPath
        {
            get
            {
                return this.ResultRootPath + @"\Summary.txt";
            }
        }

        private string XmlReportPath
        {
            get
            {
                return this.ResultRootPath + @"\Report.xml";
            }
        }

        private string HtmlReportPath
        {
            get
            {
                return this.ResultRootPath + @"\Report.html";
            }
        }

        /// <summary>
        /// 要約のヘッダーを出力します。
        /// </summary>
        internal void WriteHeader()
        {
            var sb = new StringBuilder();

            sb.Append('=', 28);
            sb.Append(" NNTest ");
            sb.Append('=', 28);
            sb.AppendLine();

            sb.AppendLine(string.Format(
                "{0,-15}: {1}", "Test date", this.Date));

            sb.AppendLine(string.Format(
                "{0,-15}: {1}", "Test root", this.TestRootPath));

            var options = new[]
            {
                new Option("Platform", "Platforms", this.Platforms),
                new Option("BuildType", "BuildTypes", this.BuildTypes),
                new Option("Module", "Modules", this.Modules),
                new Option("Category", "Categories", this.Categories),
            };

            foreach (var option in options)
            {
                var count = option.Elements.Count;

                if (count > 0)
                {
                    sb.AppendLine(string.Format(
                        "{0,-15}: {1}",
                        count == 1 ? option.Singular : option.Plural,
                        string.Join(", ", option.Elements)));
                }
            }

            if (!string.IsNullOrEmpty(this.BranchName))
            {
                sb.AppendLine(string.Format(
                    "{0,-15}: {1}", "Branch name", this.BranchName));
            }

            if (!string.IsNullOrEmpty(this.CommitHash))
            {
                sb.AppendLine(string.Format(
                    "{0,-15}: {1}", "Commit hash", this.CommitHash));
            }

            this.WriteSummary(sb.ToString());
        }

        /// <summary>
        /// テストリスト処理の結果の要約を出力します。
        /// </summary>
        /// <param name="path">テスト結果ファイルの絶対パスです。</param>
        internal void WriteListResult(string path)
        {
            if (this.EnablesInstantSummary)
            {
                var text = File.ReadAllText(path, Encoding.UTF8);

                var listContext =
                    ObjectSerializer.Deserialize<ListContext>(text);

                this.WriteSummary(this.GetListResultSummary(listContext));
            }
        }

        /// <summary>
        /// テスト実行の結果の要約を出力します。
        /// </summary>
        /// <param name="path">テスト結果ファイルの絶対パスです。</param>
        internal void WriteTestResult(string path)
        {
            if (this.EnablesInstantSummary)
            {
                var text = File.ReadAllText(path, Encoding.UTF8);

                var testContext =
                    ObjectSerializer.Deserialize<TestContext>(text);

                this.WriteSummary(this.GetTestResultSummary(testContext));
            }
        }

        /// <summary>
        /// 要約のフッターを出力します。
        /// </summary>
        /// <returns>テスト結果から算出した終了コードです。</returns>
        internal int WriteFooter()
        {
            var codeCounter = new ResultCodeCounter();

            var durationAggregator = new DurationAggregator();

            var fileAggregator = new FileAggregator();

            var xmlWriter = new XmlReportWriter();

            var htmlWriter = new HtmlReportWriter();

            var aggregators = new List<IAggregator>
            {
                codeCounter,
                durationAggregator,
                xmlWriter,
                htmlWriter,
            };

            if (!this.EnablesInstantSummary)
            {
                var summaryWriter = new SummaryWriter();

                summaryWriter.Summarizer = this;

                aggregators.Add(summaryWriter);
            }

            if (this.EnablesFileSummary)
            {
                aggregators.Add(fileAggregator);
            }

            htmlWriter.Summarizer = this;

            this.AggregateResultSummary(aggregators);

            SortedDictionary<ResultCode, uint> counts = codeCounter.Counts;

            long totalDuration = durationAggregator.TotalDuration;

            xmlWriter.Write(this.XmlReportPath);

            htmlWriter.Write(this.HtmlReportPath);

            int exitCode = counts.Keys.Any(
                x => x.MeansSerious() || x == ResultCode.FAIL) ? 1 : 0;

            var sb = new StringBuilder();

            sb.AppendLine(new string('=', 64));

            var executedTestCount = 0u;

            foreach (var pair in counts)
            {
                sb.AppendLine(string.Format(
                    "- {0,-10}   : {1,3}", pair.Key, pair.Value));

                if (pair.Key.MeansSkip())
                {
                    continue;
                }

                executedTestCount += pair.Value;
            }

            sb.AppendLine();

            sb.AppendLine(FormatTestCount(executedTestCount, totalDuration));

            this.WriteSummary(sb.ToString());

            if (exitCode == 0 && this.EnablesFileSummary)
            {
                foreach (string path in fileAggregator.Paths)
                {
                    Console.WriteLine(path);
                }
            }
            else if (!this.EnablesInstantSummary)
            {
                Console.Write(
                    File.ReadAllText(this.SummaryPath, Encoding.UTF8));
            }

            return exitCode;
        }

        private static string FormatTestCount(uint count, long duration)
        {
            switch (count)
            {
                case 0u:
                    return "No tests were done.";

                case 1u:
                    return "1 test was done.";

                default:
                    return $"{count} tests were done. ({duration} ms total)";

            }
        }

        private void WriteSummary(string str)
        {
            File.AppendAllText(this.SummaryPath, str, Encoding.UTF8);

            if (this.EnablesInstantSummary)
            {
                Console.Write(str);
            }
        }

        private string GetListResultSummary(ListContext context)
        {
            var sb = new StringBuilder();

            sb.Append('=', 64);
            sb.AppendLine();

            sb.AppendFormat("List: {0}", context.ListName);

            if (context.StatementCount > 1)
            {
                sb.AppendFormat(" [{0:D3}]", context.StatementId);
            }

            if (context.ResultCode == ResultCode.PASS)
            {
                var conditions = new List<string>();

                if (context.PlatformCount > 1)
                {
                    conditions.Add(context.PlatformName);
                }

                if (context.BuildTypeCount > 1)
                {
                    conditions.Add(context.BuildTypeName);
                }

                if (conditions.Count > 0)
                {
                    sb.AppendFormat(" ({0})", string.Join(", ", conditions));
                }
            }

            sb.AppendLine();

            sb.Append('=', 64);
            sb.AppendLine();

            sb.AppendLine(string.Format(
                "{0} {1,-3}  {2,-10}  {3,-12}      {4}",
                " ", "No.", "Result", "Duration[ms]", "FileName"));

            if (context.ResultCode != ResultCode.PASS)
            {
                var fileName = Path.GetFileName(context.ListName);

                sb.AppendLine(string.Format(
                    "{0} {1:D3}  {2,-10}  {3,12}      {4}",
                    "*", 0, context.ResultCode, "-", fileName));
            }

            return sb.ToString();
        }

        private string GetTestResultSummary(TestContext context)
        {
            var sb = new StringBuilder();

            var marker = (context.ResultCode == ResultCode.PASS)
                ? " "
                : "*";

            var duration = context.ResultCode.MeansSkip()
                ? "-"
                : context.Duration.ToString();

            var testId = string.Empty;

            switch (context.RoleType)
            {
                case RoleType.Test:
                    testId = string.Format("{0:D3}", context.TestId);
                    break;

                case RoleType.Observer:
                    testId = "OBS";
                    break;
            }

            sb.AppendLine(string.Format(
                "{0} {1}  {2,-10}  {3,12}      {4}",
                marker,
                testId,
                context.ResultCode,
                duration,
                context.TestName));

            return sb.ToString();
        }

        private void AggregateResultSummary(IList<IAggregator> aggregators)
        {
            var dirPaths = Directory.EnumerateDirectories(this.ResultRootPath);

            foreach (var dirPath in dirPaths)
            {
                ListContext listContext = null;

                var paths = Directory
                    .EnumerateFiles(dirPath, "*" + ExtensionDefinition.Result);

                foreach (var path in paths)
                {
                    if (path.EndsWith(ExtensionDefinition.ListResult))
                    {
                        var text = File.ReadAllText(path, Encoding.UTF8);

                        listContext =
                            ObjectSerializer.Deserialize<ListContext>(text);

                        foreach (IAggregator aggregator in aggregators)
                        {
                            aggregator.AcceptListContext(listContext);
                        }
                    }
                    else
                    {
                        if (listContext == null)
                        {
                            throw new FileNotFoundException(
                                $"List result file does not exist in '{dirPath}'");
                        }

                        var text = File.ReadAllText(path, Encoding.UTF8);

                        var testContext =
                            ObjectSerializer.Deserialize<TestContext>(text);

                        foreach (IAggregator aggregator in aggregators)
                        {
                            aggregator.AcceptTestContext(testContext);
                        }
                    }
                }
            }
        }

        private struct Option
        {
            /// <summary>
            /// Option クラスの新しいインスタンスを初期化します。
            /// </summary>
            /// <param name="singular">オプション名の単数形です。</param>
            /// <param name="plural">オプション名の複数形です。</param>
            /// <param name="elements">オプションの要素です。</param>
            internal Option(
                string singular, string plural, IReadOnlyList<string> elements)
            {
                this.Singular = singular;

                this.Plural = plural;

                this.Elements = elements;
            }

            /// <summary>
            /// オプション名の単数形です。
            /// </summary>
            internal string Singular { get; private set; }

            /// <summary>
            /// オプション名の複数形です。
            /// </summary>
            internal string Plural { get; private set; }

            /// <summary>
            /// オプションの要素です。
            /// </summary>
            internal IReadOnlyList<string> Elements { get; private set; }
        }

        private interface IAggregator
        {
            /// <summary>
            /// テストリストコンテキストを受理します。
            /// </summary>
            /// <param name="listContext">テストリストコンテキストです。</param>
            void AcceptListContext(ListContext listContext);

            /// <summary>
            /// テストコンテキストを受理します。
            /// </summary>
            /// <param name="testContext">テストコンテキストです。</param>
            void AcceptTestContext(TestContext testContext);
        }

        private sealed class SummaryWriter : IAggregator
        {
            /// <summary>
            /// ResultCodeCounter クラスの新しいインスタンスを初期化します。
            /// </summary>
            internal SummaryWriter()
            {
                this.Summarizer = null;
            }

            /// <summary>
            /// サマライザーを取得または設定します。
            /// </summary>
            internal ResultSummarizer Summarizer { get; set; }

            /// <summary>
            /// テストリストコンテキストを受理します。
            /// </summary>
            /// <param name="listContext">テストリストコンテキストです。</param>
            public void AcceptListContext(ListContext listContext)
            {
                var summarizer = this.Summarizer;

                string summary = summarizer.GetListResultSummary(listContext);

                summarizer.WriteSummary(summary);
            }

            /// <summary>
            /// テストコンテキストを受理します。
            /// </summary>
            /// <param name="testContext">テストコンテキストです。</param>
            public void AcceptTestContext(TestContext testContext)
            {
                var summarizer = this.Summarizer;

                string summary = summarizer.GetTestResultSummary(testContext);

                summarizer.WriteSummary(summary);
            }
        }

        private sealed class ResultCodeCounter : IAggregator
        {
            /// <summary>
            /// ResultCodeCounter クラスの新しいインスタンスを初期化します。
            /// </summary>
            internal ResultCodeCounter()
            {
                this.Counts = new SortedDictionary<ResultCode, uint>();
            }

            /// <summary>
            /// 結果識別コードの度数を取得します。
            /// </summary>
            internal SortedDictionary<ResultCode, uint> Counts
            {
                get; private set;
            }

            /// <summary>
            /// テストリストコンテキストを受理します。
            /// </summary>
            /// <param name="listContext">テストリストコンテキストです。</param>
            public void AcceptListContext(ListContext listContext)
            {
                if (listContext.ResultCode != ResultCode.PASS)
                {
                    this.Increment(listContext.ResultCode);
                }
            }

            /// <summary>
            /// テストコンテキストを受理します。
            /// </summary>
            /// <param name="testContext">テストコンテキストです。</param>
            public void AcceptTestContext(TestContext testContext)
            {
                this.Increment(testContext.ResultCode);
            }

            private void Increment(ResultCode resultCode)
            {
                var counts = this.Counts;

                if (!counts.ContainsKey(resultCode))
                {
                    counts[resultCode] = 0;
                }

                ++counts[resultCode];
            }
        }

        private sealed class DurationAggregator : IAggregator
        {
            /// <summary>
            /// DurationAggregator クラスの新しいインスタンスを初期化します。
            /// </summary>
            internal DurationAggregator()
            {
                this.TotalDuration = 0L;
            }

            /// <summary>
            /// テスト実行にかかった時間（ミリ秒）の合計値を取得します。
            /// </summary>
            internal long TotalDuration { get; private set; }

            /// <summary>
            /// テストリストコンテキストを受理します。
            /// </summary>
            /// <param name="listContext">テストリストコンテキストです。</param>
            public void AcceptListContext(ListContext listContext)
            {
                // 何もしません
            }

            /// <summary>
            /// テストコンテキストを受理します。
            /// </summary>
            /// <param name="testContext">テストコンテキストです。</param>
            public void AcceptTestContext(TestContext testContext)
            {
                if (!testContext.ResultCode.MeansSkip())
                {
                    this.TotalDuration += testContext.Duration;
                }
            }
        }

        private sealed class FileAggregator : IAggregator
        {
            private HashSet<string> paths;

            private EpiManager epiManager;

            /// <summary>
            /// FileAggregator クラスの新しいインスタンスを初期化します。
            /// </summary>
            internal FileAggregator()
            {
                this.paths = new HashSet<string>();

                this.epiManager = new EpiManager();
            }

            /// <summary>
            /// ファイルパスを取得します。
            /// </summary>
            internal IReadOnlyCollection<string> Paths
            {
                get
                {
                    return this.paths;
                }
            }

            /// <summary>
            /// テストリストコンテキストを受理します。
            /// </summary>
            /// <param name="listContext">テストリストコンテキストです。</param>
            public void AcceptListContext(ListContext listContext)
            {
                // 何もしません
            }

            /// <summary>
            /// テストコンテキストを受理します。
            /// </summary>
            /// <param name="testContext">テストコンテキストです。</param>
            public void AcceptTestContext(TestContext testContext)
            {
                string targetPath = testContext.TargetPath;

                if (IsAcceptableTargetPath(targetPath))
                {
                    switch (Path.GetExtension(targetPath))
                    {
                        case ".dll":
                        case ".exe":
                            AcceptExePath(this.paths, targetPath);
                            break;

                        case ".nspd_root":
                            AcceptNspdRootPath(this.paths, targetPath);
                            break;

                        default:
                            this.paths.Add(targetPath);
                            break;
                    }
                }

                foreach (string resourcePath in testContext.Resources)
                {
                    this.paths.Add(resourcePath);
                }

                var epiPath = testContext.TargetEpiPath;

                if (!string.IsNullOrEmpty(epiPath))
                {
                    this.paths.Add(epiPath);

                    var epi = this.epiManager.GetEpi(epiPath);

                    foreach (string resourcePath in epi.Resources)
                    {
                        this.paths.Add(resourcePath);
                    }
                }
            }

            private static bool IsAcceptableTargetPath(string path)
            {
                if (!path.StartsWith(BuildSystem.RootPath))
                {
                    return false;
                }

                if (!path.Contains(@"\Outputs\"))
                {
                    return false;
                }

                return true;
            }

            private static void AcceptExePath(
                HashSet<string> paths, string path)
            {
                var extensions = new[] { ".dll", ".exe", ".exe.config" };

                IEnumerable<string> filePaths = Directory.EnumerateFiles(
                    Path.GetDirectoryName(path),
                    "*.*",
                    SearchOption.AllDirectories);

                foreach (string filePath in filePaths)
                {
                    if (extensions.Any(x => filePath.EndsWith(x)))
                    {
                        paths.Add(filePath);
                    }
                }
            }

            private static void AcceptNspdRootPath(
                HashSet<string> paths, string path)
            {
                paths.Add(Utility.RemoveExtension(path, "_root") + @"\**");
            }
        }

        private sealed class XmlReportWriter : IAggregator
        {
            /// <summary>
            /// XmlReportWriter クラスの新しいインスタンスを初期化します。
            /// </summary>
            internal XmlReportWriter()
            {
                this.Xml = new XmlDocument();

                this.Xml.AppendChild(
                    this.Xml.CreateXmlDeclaration("1.0", "UTF-8", null));

                this.TestSuites = this.Xml.CreateElement("testsuites");

                this.Xml.AppendChild(this.TestSuites);
            }

            private XmlDocument Xml { get; }

            private XmlElement TestSuites { get; }

            private XmlElement TestSuite { get; set; }

            /// <summary>
            /// テストリストコンテキストを受理します。
            /// </summary>
            /// <param name="listContext">テストリストコンテキストです。</param>
            public void AcceptListContext(ListContext listContext)
            {
                XmlDocument xml = this.Xml;

                this.TestSuite = xml.CreateElement("testsuite");

                string testSuiteName = GetTestSuiteName(listContext);

                this.TestSuite.SetAttribute("name", testSuiteName);

                this.TestSuites.AppendChild(this.TestSuite);

                XmlElement testCase = xml.CreateElement("testcase");

                string testCaseName = GetTestCaseName(listContext);

                testCase.SetAttribute("name", testCaseName);

                testCase.SetAttribute("classname", testSuiteName);

                this.TestSuite.AppendChild(testCase);

                if (!listContext.ResultCode.MeansSerious())
                {
                    return;
                }

                XmlElement failure = xml.CreateElement("failure");

                var builder = new StringBuilder();

                builder.Append(listContext.ResultCode);

                if (!string.IsNullOrEmpty(listContext.ErrorMessage))
                {
                    builder.AppendFormat(": {0}", listContext.ErrorMessage);
                }

                failure.InnerText = builder.ToString();

                testCase.AppendChild(failure);
            }

            /// <summary>
            /// テストコンテキストを受理します。
            /// </summary>
            /// <param name="testContext">テストコンテキストです。</param>
            public void AcceptTestContext(TestContext testContext)
            {
                XmlDocument xml = this.Xml;

                XmlElement testSuite = this.TestSuite;

                XmlElement testCase = xml.CreateElement("testcase");

                string testCaseName = GetTestCaseName(testContext);

                testCase.SetAttribute("name", testCaseName);

                testCase.SetAttribute(
                    "classname", testSuite.GetAttribute("name"));

                this.TestSuite.AppendChild(testCase);

                if (!testContext.ResultCode.MeansSerious())
                {
                    return;
                }

                XmlElement failure = xml.CreateElement("failure");

                var builder = new StringBuilder();

                builder.Append(testContext.ResultCode);

                if (!string.IsNullOrEmpty(testContext.ErrorMessage))
                {
                    builder.AppendFormat(": {0}", testContext.ErrorMessage);
                }
                else if (testContext.ExitCode != 0)
                {
                    builder.AppendFormat(
                        ": ExitCode {0}", testContext.ExitCode);
                }

                failure.InnerText = builder.ToString();

                testCase.AppendChild(failure);
            }

            /// <summary>
            /// 指定されたパスに XML レポートを書き出します。
            /// <param name="path">パスです。</param>
            /// </summary>
            internal void Write(string path)
            {
                XmlDocument xml = this.Xml;

                XmlElement testSuites = this.TestSuites;

                int totalTestCount = 0;

                int totalFailureCount = 0;

                foreach (XmlElement testSuite in testSuites)
                {
                    int testCount = 0;

                    int failureCount = 0;

                    foreach (XmlElement testCase in testSuite.ChildNodes)
                    {
                        testCount += 1;

                        failureCount += testCase.ChildNodes.Count;
                    }

                    totalTestCount += testCount;

                    totalFailureCount += failureCount;

                    testSuite.SetAttribute(
                        "tests", testCount.ToString());

                    testSuite.SetAttribute(
                        "failures", failureCount.ToString());
                }

                testSuites.SetAttribute(
                    "tests", totalTestCount.ToString());

                testSuites.SetAttribute(
                    "failures", totalFailureCount.ToString());

                testSuites.SetAttribute(
                    "name", "AllTestRunnerTests");

                xml.Save(path);
            }

            private static string GetTestSuiteName(ListContext listContext)
            {
                var sb = new StringBuilder();

                sb.Append(listContext.ListName.Replace('.', '_'));

                if (listContext.StatementCount > 1)
                {
                    sb.AppendFormat("[{0:D3}]", listContext.StatementId);
                }

                if (listContext.ResultCode == ResultCode.PASS)
                {
                    var args = new List<string>();

                    if (listContext.PlatformCount > 1)
                    {
                        args.Add(listContext.PlatformName);
                    }

                    if (listContext.BuildTypeCount > 1)
                    {
                        args.Add(listContext.BuildTypeName);
                    }

                    if (args.Count > 0)
                    {
                        sb.AppendFormat("({0})", string.Join(",", args));
                    }
                }

                return sb.ToString();
            }

            private static string GetTestCaseName(ListContext listContext)
            {
                return string.Format("YML[00]{0}",
                    Path.GetFileName(listContext.ListName).Replace('.', '_'));
            }

            private static string GetTestCaseName(TestContext testContext)
            {
                var builder = new StringBuilder();

                switch (testContext.RoleType)
                {
                    case RoleType.Test:
                        builder.AppendFormat("{0:D3}", testContext.TestId);
                        break;

                    case RoleType.Observer:
                        builder.Append("OBS");
                        break;
                }

                builder.AppendFormat("[{0:D2}]", testContext.UnitId);

                builder.Append(
                    Path.GetFileName(testContext.TestName).Replace('.', '_'));

                return builder.ToString();
            }
        }

        private sealed class HtmlReportWriter : IAggregator
        {
            /// <summary>
            /// サマライザーを取得または設定します。
            /// </summary>
            internal ResultSummarizer Summarizer { get; set; }

            private List<string> Results { get; } = new List<string>();

            private List<string> Tests { get; } = new List<string>();

            ListContext ListContext { get; set; }

            /// <summary>
            /// テストリストコンテキストを受理します。
            /// </summary>
            /// <param name="listContext">テストリストコンテキストです。</param>
            public void AcceptListContext(ListContext listContext)
            {
                if (this.Tests.Count > 0)
                {
                    var result = FormatResult(this.ListContext, this.Tests);

                    this.Results.Add(result);

                    this.Tests.Clear();
                }

                this.ListContext = listContext;

                if (listContext.ResultCode != ResultCode.PASS)
                {
                    var sb = new StringBuilder();

                    sb.AppendLine("{");

                    sb.AppendLine(string.Format(
                        "type: \"{0}\",", listContext.ResultCode.MeansSerious()
                            ? ResultCode.ERROR : listContext.ResultCode));

                    sb.AppendLine(
                        "number: \"000\",");

                    sb.AppendLine(string.Format(
                        "result: \"{0}\",", listContext.ResultCode));

                    var uri = new Uri(this.Summarizer.ResultRootPath + @"\");

                    sb.AppendLine(
                        "log: \"\",");

                    sb.AppendLine(string.Format(
                        "xml: \"{0}\",", uri.MakeRelativeUri(
                            new Uri(listContext.Path)).ToString()));

                    sb.AppendLine(
                        "duration: \"0\",");

                    sb.AppendLine(string.Format(
                        "name: \"{0}\"",
                        Path.GetFileName(listContext.ListName)));

                    sb.AppendLine("}");

                    this.Tests.Add(sb.ToString());
                }
            }

            /// <summary>
            /// テストコンテキストを受理します。
            /// </summary>
            /// <param name="testContext">テストコンテキストです。</param>
            public void AcceptTestContext(TestContext testContext)
            {
                var sb = new StringBuilder();

                sb.AppendLine("{");

                ResultCode resultCode = testContext.ResultCode;

                switch (resultCode)
                {
                    case ResultCode.PASS:
                    case ResultCode.FAIL:
                        sb.AppendLine(string.Format(
                            "type: \"{0}\",", resultCode));
                        break;

                    default:
                        sb.AppendLine(string.Format(
                            "type: \"{0}\",", resultCode.MeansSerious()
                                ? ResultCode.ERROR : resultCode));
                        break;
                }

                sb.AppendLine(string.Format(
                    "number: \"{0}\",",
                    testContext.RoleType == RoleType.Observer
                        ? "OBS"
                        : testContext.TestId.ToString("D3")));

                sb.AppendLine(string.Format(
                    "result: \"{0}\",", resultCode));

                var uri = new Uri(this.Summarizer.ResultRootPath + @"\");

                sb.AppendLine(string.Format(
                    "log: \"{0}\",", resultCode.MeansSkip()
                        ? string.Empty : uri.MakeRelativeUri(
                            new Uri(testContext.LogPath)).ToString()));

                sb.AppendLine(string.Format(
                    "xml: \"{0}\",", uri.MakeRelativeUri(
                        new Uri(testContext.Path)).ToString()));

                sb.AppendLine(string.Format(
                    "duration: \"{0}\",", testContext.Duration));

                sb.AppendLine(string.Format(
                    "name: \"{0}\"", testContext.TestName));

                sb.AppendLine("}");

                this.Tests.Add(sb.ToString());
            }

            /// <summary>
            /// 指定されたパスに HTML レポートを書き出します。
            /// <param name="path">パスです。</param>
            /// </summary>
            internal void Write(string path)
            {
                if (this.Tests.Count > 0)
                {
                    var result = FormatResult(this.ListContext, this.Tests);

                    this.Results.Add(result);
                }

                string text = Resources.Report_template_html;

                var sb = new StringBuilder();

                sb.AppendLine("{");

                sb.AppendLine(string.Format(
                    "date: \"{0}\",", this.Summarizer.Date));

                sb.AppendLine(string.Format(
                    "path: \"{0}\",", Escape(this.Summarizer.TestRootPath)));

                sb.AppendLine(string.Format(
                    "platforms: [ {0} ],",
                    string.Join(", ",
                        this.Summarizer.Platforms.Select(x => Format(x)))));

                sb.AppendLine(string.Format(
                    "builds: [ {0} ],",
                    string.Join(", ",
                        this.Summarizer.BuildTypes.Select(x => Format(x)))));

                sb.AppendLine(string.Format(
                    "modules: [ {0} ],",
                    string.Join(", ",
                        this.Summarizer.Modules.Select(x => Format(x)))));

                sb.AppendLine(string.Format(
                    "categories: [ {0} ],",
                    string.Join(", ",
                        this.Summarizer.Categories.Select(x => Format(x)))));

                sb.AppendLine(string.Format(
                    "branch: \"{0}\",",
                    this.Summarizer.BranchName ?? string.Empty));

                sb.AppendLine(string.Format(
                    "commit: \"{0}\",",
                    this.Summarizer.CommitHash ?? string.Empty));

                sb.AppendLine(string.Format(
                    "results: [ {0} ]", string.Join(", ", this.Results)));

                sb.AppendLine("}");

                text = text.Replace("/*DUMMY_REPORTS*/", sb.ToString());

                File.WriteAllText(path, text);
            }

            private static string Escape(string text)
            {
                return text.Replace("\\", "\\\\");
            }

            private static string Format(string text)
            {
                return string.Format("\"{0}\"", text);
            }

            private static string FormatListName(ListContext listContext)
            {
                var sb = new StringBuilder();

                sb.Append(Escape(listContext.ListName));

                if (listContext.StatementCount > 1)
                {
                    sb.AppendFormat(" [{0:D3}]", listContext.StatementId);
                }

                if (listContext.ResultCode == ResultCode.PASS)
                {
                    var args = new List<string>();

                    if (listContext.PlatformCount > 1)
                    {
                        args.Add(listContext.PlatformName);
                    }

                    if (listContext.BuildTypeCount > 1)
                    {
                        args.Add(listContext.BuildTypeName);
                    }

                    if (args.Count > 0)
                    {
                        sb.AppendFormat(" ({0})", string.Join(", ", args));
                    }
                }

                return sb.ToString();
            }

            private static string FormatResult(
                ListContext listContext, List<string> tests)
            {
                var sb = new StringBuilder();

                sb.AppendLine("{");

                sb.AppendLine(string.Format(
                    "path: \"{0}\",", FormatListName(listContext)));

                sb.AppendLine("tests: [");

                sb.AppendLine(string.Join(", ", tests));

                sb.AppendLine("]");

                sb.AppendLine("}");

                return sb.ToString();
            }
        }
    }
}
