﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.ComponentModel.Composition.Hosting;
    using System.IO;
    using System.Linq;
    using Executer;

    /// <summary>
    /// テストコンテキストにインストラクション生成ルールを適用します。
    /// </summary>
    internal sealed class InstructionGenerator
    {
        /// <summary>
        /// InstructionGenerator クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal InstructionGenerator()
        {
            using (var catalog = new AssemblyCatalog(typeof(IRule).Assembly))
            using (var container = new CompositionContainer(catalog))
            {
                container.SatisfyImportsOnce(this);
            }
        }

        /// <summary>
        /// インストラクション生成ルールを定義します。
        /// </summary>
        [InheritedExport]
        internal interface IRule
        {
            /// <summary>
            /// サポート対象のテストコンテキストかどうかを示す値を取得します。
            /// </summary>
            /// <param name="context">テストコンテキストです。</param>
            /// <returns>サポート対象のテストコンテキストかどうかを示す値です。</returns>
            bool Supports(TestContext context);

            /// <summary>
            /// ルール適用時にターゲットファイルを必要とするかどうかを示す値を取得します。
            /// </summary>
            /// <returns>ルール適用時にターゲットファイルを必要とするかどうかを示す値です。</returns>
            bool NeedsTargetFile();

            /// <summary>
            /// テストコンテキストにインストラクション生成ルールを適用します。
            /// </summary>
            /// <param name="context">テストコンテキストです。</param>
            /// <returns>生成されたテストインストラクションです。</returns>
            Instruction Generate(TestContext context);
        }

        [ImportMany]
        private List<IRule> Rules { get; set; }

        /// <summary>
        /// WRONG_TYPE に該当するテストコンテキストかどうかを示す値を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>WRONG_TYPE に該当するテストコンテキストかどうかを示す値です。</returns>
        internal bool IsWrongType(TestContext context)
        {
            return !this.Rules.Any(rule => rule.Supports(context));
        }

        /// <summary>
        /// NO_FILE に該当するテストコンテキストかどうかを示す値を取得します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>NO_FILE に該当するテストコンテキストかどうかを示す値です。</returns>
        internal bool IsNoFile(TestContext context)
        {
            return !this.Rules.Any(rule =>
            {
                if (!rule.Supports(context))
                {
                    return false;
                }
                else
                {
                    if (!rule.NeedsTargetFile())
                    {
                        return true;
                    }
                    else if (!string.IsNullOrEmpty(context.TargetEpiPath))
                    {
                        return true;
                    }
                    else
                    {
                        return File.Exists(context.TargetPath);
                    }
                }
            });
        }

        /// <summary>
        /// テストコンテキストにインストラクション生成ルールを適用します。
        /// </summary>
        /// <param name="context">テストコンテキストです。</param>
        /// <returns>生成されたテストインストラクションです。</returns>
        internal Instruction Generate(TestContext context)
        {
            var foundRule = this.Rules
                .FirstOrDefault(rule => rule.Supports(context));

            if (foundRule != null)
            {
                return foundRule.Generate(context);
            }

            context.ResultCode = ResultCode.WRONG_TYPE;

            context.ErrorMessage = "Unexpected file type";

            throw new TestContextException(context.ErrorMessage, context);
        }

        /// <summary>
        /// テストインストラクションです。
        /// </summary>
        internal sealed class Instruction
        {
            /// <summary>
            /// Instruction クラスの新しいインスタンスを初期化します。
            /// </summary>
            internal Instruction()
            {
                this.Command = string.Empty;

                this.Option = string.Empty;

                this.FailurePatterns = null;

                this.RunSettings = string.Empty;

                this.Platform = string.Empty;

                this.BuildType = string.Empty;

                this.TargetPath = string.Empty;

                this.TargetProjectPath = string.Empty;

                this.ReportPath = string.Empty;

                this.CsxPath = string.Empty;

                this.RunSettingsPath = string.Empty;

                this.DumpFileType = DumpFileTypeDefinition.None;

                this.DumpFilePath = string.Empty;
            }

            /// <summary>
            /// 実行するプログラムの絶対パスを取得または設定します。
            /// </summary>
            internal string Command { get; set; }

            /// <summary>
            /// 実行するプログラムに渡す引数を取得または設定します。
            /// </summary>
            internal string Option { get; set; }

            /// <summary>
            /// 実行するプログラムの失敗条件とするログ出力のパターンを取得または設定します。
            /// </summary>
            internal string[] FailurePatterns { get; set; }

            /// <summary>
            /// 実行するプログラムに渡す .runsettings を取得または設定します。
            /// </summary>
            internal string RunSettings { get; set; }

            /// <summary>
            /// 実行するプログラムのプラットフォームを取得または設定します。
            /// </summary>
            internal string Platform { get; set; }

            /// <summary>
            /// 実行するプログラムのビルドタイプを取得または設定します。
            /// </summary>
            internal string BuildType { get; set; }

            /// <summary>
            /// 実行対象となるファイルの絶対パスを取得または設定します。
            /// </summary>
            internal string TargetPath { get; set; }

            /// <summary>
            /// 実行対象となるファイルのプロジェクトファイルの絶対パスを取得または設定します。
            /// </summary>
            internal string TargetProjectPath { get; set; }

            /// <summary>
            /// 実行するプログラムが生成するレポートファイルの絶対パスを取得または設定します。
            /// </summary>
            internal string ReportPath { get; set; }

            /// <summary>
            /// 実行するプログラムに渡す .csx の絶対パスを取得または設定します。
            /// </summary>
            internal string CsxPath { get; set; }

            /// <summary>
            /// 実行するプログラムに渡す .runsettings の絶対パスを取得または設定します。
            /// </summary>
            internal string RunSettingsPath { get; set; }

            /// <summary>
            /// 実行するプログラムのダンプファイルの形式を取得または設定します。
            /// </summary>
            internal DumpFileTypeDefinition DumpFileType { get; set; }

            /// <summary>
            /// 実行するプログラムのダンプファイルの絶対パスを取得または設定します。
            /// </summary>
            internal string DumpFilePath { get; set; }
        }
    }
}
