﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Reflection;
    using System.Text;
    using System.Text.RegularExpressions;

    /// <summary>
    /// コマンドラインオプションを解析して、テストランナーから利用できるようにします。
    /// </summary>
    internal sealed class CommandLineOption
    {
        private static readonly Parameter[] Parameters = new[]
        {
            new Parameter()
            {
                LongName     = "--help",
                ShortName    = "-h",
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = "print this message",
                Action       = (opt, x) => { opt.EnablesHelpMode = true; }
            },
            new Parameter()
            {
                LongName     = "--output",
                ShortName    = "-o",
                MetaVariable = "path",
                IsMandatory  = true,
                HelpText     = "specifies the output directory",
                Action = (opt, x) =>
                {
                    if (!Directory.Exists(x))
                    {
                        throw new DirectoryNotFoundException(
                            $"The specified directory '{x}' does not exist");
                    }

                    opt.OutputPath = Path.GetFullPath(x);
                }
            },
            new Parameter()
            {
                LongName     = "--branch-name",
                ShortName    = null,
                MetaVariable = "string",
                IsMandatory  = false,
                HelpText     = "specifies the branch name",
                Action = (opt, x) =>
                {
                    opt.BranchName = x;
                }
            },
            new Parameter()
            {
                LongName     = "--commit-hash",
                ShortName    = null,
                MetaVariable = "string",
                IsMandatory  = false,
                HelpText     = "specifies the commit hash",
                Action = (opt, x) =>
                {
                    opt.CommitHash = x;
                }
            },
            new Parameter()
            {
                LongName     = "--verbose",
                ShortName    = "-v",
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = "be extra verbose",
                Action       = (opt, x) => { opt.EnablesVerboseMode = true; }
            },
            new Parameter()
            {
                LongName     = "--quiet-exit",
                ShortName    = null,
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = "return exit code 0 except for internal error",
                Action       = (opt, x) => { opt.EnablesQuietExit = true; }
            },
            new Parameter()
            {
                LongName     = "--file-listing",
                ShortName    = null,
                MetaVariable = null,
                IsMandatory  = false,
                HelpText     = "list files that are mentioned in the testlists",
                Action = (opt, x) =>
                {
                    opt.EnablesFileListingMode = true;
                }
            },
            new Parameter()
            {
                LongName     = "--jobs",
                ShortName    = "-j",
                MetaVariable = "N",
                IsMandatory  = false,
                HelpText     = "allow N jobs at once",
                Action = (opt, x) =>
                {
                    opt.EnablesParallelMode = true;
                    try
                    {
                        opt.JobCount = uint.Parse(x);
                    }
                    catch (Exception ex)
                    {
                        throw new ArgumentException(
                            $"The number of jobs '{x}' is invalid", ex);
                    }
                }
            },
            new Parameter()
            {
                LongName     = "--path",
                ShortName    = "-p",
                MetaVariable = "path",
                IsMandatory  = false,
                HelpText     = "specifies a directory of testlists",
                Action = (opt, x) =>
                {
                    if (!Directory.Exists(x))
                    {
                        throw new DirectoryNotFoundException(
                            $"The specified directory '{x}' does not exist");
                    }

                    opt.MutableDirectoryPaths.Add(Path.GetFullPath(x));
                }
            },
            new Parameter()
            {
                LongName     = "--file",
                ShortName    = "-f",
                MetaVariable = "path",
                IsMandatory  = false,
                HelpText     = "specifies a testlist",
                Action = (opt, x) =>
                {
                    if (!File.Exists(x))
                    {
                        throw new FileNotFoundException(
                            $"The specified testlist '{x}' does not exist");
                    }

                    if (!x.EndsWith(ExtensionDefinition.TestList))
                    {
                        throw new ArgumentException(
                            $"The specified file '{x}' is not a testlist");
                    }

                    opt.MutableFilePaths.Add(Path.GetFullPath(x));
                }
            },
            new Parameter()
            {
                LongName     = "--platform",
                ShortName    = null,
                MetaVariable = "string",
                IsMandatory  = false,
                HelpText     = "specifies a platform",
                Action = (opt, x) =>
                {
                    opt.MutablePlatforms.Add(x);
                }
            },
            new Parameter()
            {
                LongName     = "--build-type",
                ShortName    = null,
                MetaVariable = "string",
                IsMandatory  = false,
                HelpText     = "specifies a build type",
                Action = (opt, x) =>
                {
                    opt.MutableBuildTypes.Add(x);
                }
            },
            new Parameter()
            {
                LongName     = "--module",
                ShortName    = null,
                MetaVariable = "string",
                IsMandatory  = false,
                HelpText     = "specifies a target module",
                Action = (opt, x) =>
                {
                    opt.MutableModules.Add(x);
                }
            },
            new Parameter()
            {
                LongName     = "--category",
                ShortName    = null,
                MetaVariable = "string",
                IsMandatory  = false,
                HelpText     = "specifies a test category",
                Action = (opt, x) =>
                {
                    opt.MutableCategories.Add(x);
                }
            },
            new Parameter()
            {
                LongName     = "--executable-product-info",
                ShortName    = null,
                MetaVariable = "path",
                IsMandatory  = false,
                HelpText     = "specifies the directory of executable product infos",
                Action = (opt, x) =>
                {
                    if (!Directory.Exists(x))
                    {
                        throw new DirectoryNotFoundException(
                            $"The specified directory '{x}' does not exist");
                    }

                    opt.EpiDirectory = Path.GetFullPath(x);
                }
            },
            new Parameter()
            {
                LongName     = "--target-name",
                ShortName    = null,
                MetaVariable = "regex",
                IsMandatory  = false,
                HelpText     = "specifies a target name pattern",
                Action = (opt, x) =>
                {
                    try
                    {
                        opt.TargetNamePattern =
                            new Regex(x, RegexOptions.Compiled);
                    }
                    catch (Exception ex)
                    {
                        throw new ArgumentException(
                            $"The specified pattern '{x}' is invalid", ex);
                    }
                }
            },
            new Parameter()
            {
                LongName     = "--target-interface",
                ShortName    = null,
                MetaVariable = "regex",
                IsMandatory  = false,
                HelpText     = "specifies a target interface pattern",
                Action = (opt, x) =>
                {
                    try
                    {
                        opt.TargetInterfacePattern =
                            new Regex(x, RegexOptions.Compiled);
                    }
                    catch (Exception ex)
                    {
                        throw new ArgumentException(
                            $"The specified pattern '{x}' is invalid", ex);
                    }
                }
            },
            new Parameter()
            {
                LongName     = "--target-address",
                ShortName    = null,
                MetaVariable = "regex",
                IsMandatory  = false,
                HelpText     = "specifies a target address pattern",
                Action = (opt, x) =>
                {
                    try
                    {
                        opt.TargetAddressPattern =
                            new Regex(x, RegexOptions.Compiled);
                    }
                    catch (Exception ex)
                    {
                        throw new ArgumentException(
                            $"The specified pattern '{x}' is invalid", ex);
                    }
                }
            }
        };

        /// <summary>
        /// CommandLineOption クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal CommandLineOption()
        {
            this.EnablesHelpMode = false;

            this.OutputPath = string.Empty;

            this.BranchName = string.Empty;

            this.CommitHash = string.Empty;

            this.EnablesVerboseMode = false;

            this.EnablesQuietExit = false;

            this.EnablesFileListingMode = false;

            this.EnablesParallelMode = false;

            this.JobCount = 0;

            this.DirectoryPaths = new List<string>().AsReadOnly();

            this.FilePaths = new List<string>().AsReadOnly();

            this.Platforms = new List<string>().AsReadOnly();

            this.BuildTypes = new List<string>().AsReadOnly();

            this.Modules = new List<string>().AsReadOnly();

            this.Categories = new List<string>().AsReadOnly();

            this.EpiDirectory = string.Empty;

            this.TargetNamePattern = new Regex("", RegexOptions.Compiled);

            this.TargetInterfacePattern = new Regex("", RegexOptions.Compiled);

            this.TargetAddressPattern = new Regex("", RegexOptions.Compiled);
        }

        /// <summary>
        /// ヘルプメッセージを出力するかどうかを示す値を取得します。
        /// </summary>
        internal bool EnablesHelpMode { get; private set; }

        /// <summary>
        /// テスト結果の出力先となるディレクトリの絶対パスを取得します。
        /// </summary>
        internal string OutputPath { get; private set; }

        /// <summary>
        /// ブランチ名を取得します。
        /// </summary>
        internal string BranchName { get; private set; }

        /// <summary>
        /// コミットハッシュを取得します。
        /// </summary>
        internal string CommitHash { get; private set; }

        /// <summary>
        /// ログ出力をコンソールに出力するかどうかを示す値を取得します。
        /// </summary>
        internal bool EnablesVerboseMode { get; private set; }

        /// <summary>
        /// 終了コードによる自動テストの失敗通知を抑制するかどうかを示す値を取得します。
        /// </summary>
        internal bool EnablesQuietExit { get; private set; }

        /// <summary>
        /// テストリスト中で言及されたファイルを列挙するかどうかを示す値を取得します。
        /// </summary>
        internal bool EnablesFileListingMode { get; private set; }

        /// <summary>
        /// 並列実行を行うかどうかを示す値を取得します。
        /// </summary>
        internal bool EnablesParallelMode { get; private set; }

        /// <summary>
        /// 並列実行数の上限値を取得します。
        /// </summary>
        internal uint JobCount { get; private set; }

        /// <summary>
        /// テストリストの探索を行うディレクトリの絶対パスのリストを取得します。
        /// </summary>
        internal IReadOnlyList<string> DirectoryPaths { get; private set; }

        /// <summary>
        /// 対象となるテストリストの絶対パスのリストを取得します。
        /// </summary>
        internal IReadOnlyList<string> FilePaths { get; private set; }

        /// <summary>
        /// 対象となるプラットフォーム名のリストを取得します。
        /// </summary>
        internal IReadOnlyList<string> Platforms { get; private set; }

        /// <summary>
        /// 対象となるビルドタイプ名のリストを取得します。
        /// </summary>
        internal IReadOnlyList<string> BuildTypes { get; private set; }

        /// <summary>
        /// 対象となるモジュール名のリストを取得します。
        /// </summary>
        internal IReadOnlyList<string> Modules { get; private set; }

        /// <summary>
        /// 対象となるテストカテゴリ名のリストを取得します。
        /// </summary>
        internal IReadOnlyList<string> Categories { get; private set; }

        /// <summary>
        /// 実行情報ファイルの探索を行うディレクトリの絶対パスを取得します。
        /// </summary>
        internal string EpiDirectory { get; private set; }

        /// <summary>
        /// 開発機の名前の正規表現を取得します。
        /// </summary>
        internal Regex TargetNamePattern { get; private set; }

        /// <summary>
        /// 開発機のインターフェイス名の正規表現を取得します。
        /// </summary>
        internal Regex TargetInterfacePattern { get; private set; }

        /// <summary>
        /// 開発機のアドレスの正規表現を取得します。
        /// </summary>
        internal Regex TargetAddressPattern { get; private set; }

        private List<string> MutableDirectoryPaths { get; set; }

        private List<string> MutableFilePaths { get; set; }

        private List<string> MutablePlatforms { get; set; }

        private List<string> MutableBuildTypes { get; set; }

        private List<string> MutableModules { get; set; }

        private List<string> MutableCategories { get; set; }

        /// <summary>
        /// ヘルプメッセージを取得します。
        /// </summary>
        /// <returns>ヘルプメッセージです。</returns>
        internal static string GetHelpMessage()
        {
            var location = Assembly.GetExecutingAssembly().Location;

            var name = Path.GetFileName(location);

            var sb = new StringBuilder();

            sb.AppendFormat("Usage: {0}", name);

            foreach (var param in Parameters)
            {
                if (!param.IsMandatory)
                {
                    continue;
                }

                sb.AppendFormat(" {0}", param.LongName);

                if (param.MetaVariable != null)
                {
                    sb.AppendFormat(" <{0}>", param.MetaVariable);
                }
            }

            sb.AppendLine(" [options]");

            sb.AppendLine();

            sb.AppendLine("options:");

            foreach (var param in Parameters)
            {
                var sb_opt = new StringBuilder();

                sb_opt.AppendFormat("  {0}", param.LongName);

                if (param.MetaVariable != null)
                {
                    sb_opt.AppendFormat(" <{0}>", param.MetaVariable);
                }

                if (param.ShortName != null)
                {
                    sb_opt.AppendFormat(", {0}", param.ShortName);

                    if (param.MetaVariable != null)
                    {
                        sb_opt.AppendFormat(" <{0}>", param.MetaVariable);
                    }
                }

                sb.AppendFormat(
                    "{0,-35} {1}", sb_opt.ToString(), param.HelpText);

                sb.AppendLine();
            }

            return sb.ToString();
        }

        /// <summary>
        /// コマンドラインオプションをパースします。
        /// </summary>
        /// <param name="args">コマンドラインオプションです。</param>
        internal void Parse(string[] args)
        {
            this.MutableDirectoryPaths = new List<string>();

            this.MutableFilePaths = new List<string>();

            this.MutablePlatforms = new List<string>();

            this.MutableBuildTypes = new List<string>();

            this.MutableModules = new List<string>();

            this.MutableCategories = new List<string>();

            var mandates = new List<string>();

            foreach (var param in Parameters)
            {
                if (param.IsMandatory)
                {
                    mandates.Add(param.LongName);
                }
            }

            for (int i = 0; i < args.Length; ++i)
            {
                for (int j = 0; j < Parameters.Length; ++j)
                {
                    var key = args[i];

                    var param = Parameters[j];

                    if ((string.Compare(param.LongName,  key) == 0) ||
                        (string.Compare(param.ShortName, key) == 0))
                    {
                        if (param.MetaVariable == null)
                        {
                            param.Action(this, null);
                        }
                        else
                        {
                            bool isLastArg = (i >= (args.Length - 1));

                            bool isNextKey = args[i + 1].StartsWith("-");

                            if (isLastArg || isNextKey)
                            {
                                throw new ArgumentException(
                                    $"Option '{key}' requires an argument");
                            }

                            param.Action(this, args[++i]);
                        }

                        mandates.Remove(param.LongName);

                        break;
                    }

                    if (j + 1 < Parameters.Length)
                    {
                        continue;
                    }

                    throw new ArgumentException(
                        $"Invalid argument '{args[i]}'");
                }
            }

            if (!this.EnablesHelpMode)
            {
                foreach (var mandate in mandates)
                {
                    throw new ArgumentException(
                        $"Option '{mandate}' was not specified");
                }
            }

            this.DirectoryPaths = this.MutableDirectoryPaths.AsReadOnly();

            this.FilePaths = this.MutableFilePaths.AsReadOnly();

            this.Platforms = this.MutablePlatforms.AsReadOnly();

            this.BuildTypes = this.MutableBuildTypes.AsReadOnly();

            this.Modules = this.MutableModules.AsReadOnly();

            this.Categories = this.MutableCategories.AsReadOnly();
        }

        private class Parameter
        {
            public string LongName { get; set; }

            public string ShortName { get; set; }

            public string MetaVariable { get; set; }

            public bool IsMandatory { get; set; }

            public string HelpText { get; set; }

            public Action<CommandLineOption, string> Action { get; set; }
        }
    }
}
