﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Reflection;
    using System.Text;
    using System.Threading;
    using Executer;
    using Properties;
    using static TargetManager;

    /// <summary>
    /// テスト結果ファイルのビルドルールを作成します。
    /// </summary>
    internal sealed class BuildRuleMaker
    {
        private const string ScriptFileName = "TestExecuter.csx";

        /// <summary>
        /// BuildRuleMaker クラスの新しいインスタンスを初期化します。
        /// </summary>
        /// <param name="jobCount">並列実行数の上限値です。</param>
        internal BuildRuleMaker(uint jobCount = 0)
        {
            this.ResultRootPath = string.Empty;

            this.TargetManager = new TargetManager();

            this.EnablesVerboseMode = false;

            this.JobCount = jobCount;

            this.ObserverEvents = new List<EventWaitHandle>();

            this.ParallelTestRuleCosts = new Dictionary<uint, uint>();

            for (uint i = 0; i < this.JobCount; ++i)
            {
                this.ParallelTestRuleCosts[i] = 0;
            }

            this.ParallelTestRules = new List<TestRule>();

            this.ParallelSources = new string[jobCount];

            this.SerialTestRules = new List<TestRule>();

            this.SerialSources = new List<string>();

            this.Tests = new List<IReadOnlyList<TestContext>>();

            this.Observers = new List<TestContext>();

            this.TargetAssigner = new TargetAssigner();
        }

        /// <summary>
        /// テスト結果ファイルの出力先のルートディレクトリの絶対パスを取得または設定します。
        /// </summary>
        internal string ResultRootPath { get; set; }

        /// <summary>
        /// 開発機の管理クラスを取得または設定します。
        /// </summary>
        internal TargetManager TargetManager { get; set; }

        /// <summary>
        /// ログをコンソールにも出力するか否かを表す値を取得または設定します。
        /// </summary>
        internal bool EnablesVerboseMode { get; set; }

        private uint JobCount { get; set; }

        private List<EventWaitHandle> ObserverEvents { get; set; }

        private Dictionary<uint, uint> ParallelTestRuleCosts { get; set; }

        private List<TestRule> ParallelTestRules { get; set; }

        private string[] ParallelSources { get; set; }

        private List<TestRule> SerialTestRules { get; set; }

        private List<string> SerialSources { get; set; }

        private List<IReadOnlyList<TestContext>> Tests { get; set; }

        private List<TestContext> Observers { get; set; }

        private TargetAssigner TargetAssigner { get; set; }

        /// <summary>
        /// ビルドルールの生成元となるテストのコンテキストをバッファに追加します。
        /// </summary>
        /// <param name="testContexts">テストコンテキストの配列です。</param>
        internal void AddTests(IReadOnlyList<TestContext> testContexts)
        {
            this.Tests.Add(testContexts);

            this.TargetAssigner.AddTests(testContexts);
        }

        /// <summary>
        /// ビルドルールの生成元となるオブザーバのコンテキストをバッファに追加します。
        /// </summary>
        /// <param name="testContexts">テストコンテキストの配列です。</param>
        internal void SetObservers(IReadOnlyList<TestContext> testContexts)
        {
            this.Observers = new List<TestContext>(testContexts);

            this.TargetAssigner.SetObservers(testContexts);
        }

        /// <summary>
        /// バッファを空にします。
        /// </summary>
        /// <param name="appending">バッファ中のテストコンテキストをビルドルールに追加するかどうかを示す値です。</param>
        internal Dictionary<ulong, TargetEntry> Flush(bool appending)
        {
            Dictionary<ulong, TargetEntry> table = null;

            try
            {
                if (!appending)
                {
                    table = new Dictionary<ulong, TargetEntry>();
                }
                else
                {
                    if (this.IsParallelizable())
                    {
                        table = this.AppendParallelTestRule();
                    }
                    else
                    {
                        table = this.AppendSerialTestRule();
                    }
                }
            }
            finally
            {
                this.Tests.Clear();

                this.Observers = new List<TestContext>();

                this.TargetAssigner = new TargetAssigner();
            }

            return table;
        }

        /// <summary>
        /// 作成したビルドルールを実行します。
        /// </summary>
        internal void Run()
        {
            File.WriteAllText(
                Path.Combine(this.ResultRootPath, ScriptFileName),
                Resources.TestExecuter_csx, Encoding.UTF8);

            var sb = new StringBuilder();

            sb.AppendLine(this.GetTestRunnerRuleDefinition());

            var sources = new HashSet<string>(
                this.ParallelSources
                    .Where(source => source != null)).ToArray();

            if (sources.Length > 0)
            {
                foreach (var rule in this.SerialTestRules)
                {
                    if (rule.Sources.Count == 0)
                    {
                        rule.Sources = sources;
                    }
                }
            }

            var rules = this.ParallelTestRules.Concat(this.SerialTestRules);

            foreach (var rule in rules)
            {
                string fileName = this.GetBuildContextFileName(rule);

                File.WriteAllText(
                    fileName, this.GetBuildContextText(rule), Encoding.UTF8);

                sb.Append(this.GetTestRunnerRuleInstance(
                    rule.Sources, rule.TestContexts.First().Path, fileName));
            }

            File.WriteAllText(
                Path.Combine(this.ResultRootPath, BuildSystem.MakeFileName),
                sb.ToString());

            using (var proc = new Process())
            {
                proc.StartInfo = new ProcessStartInfo()
                {
                    FileName = BuildSystem.ProgramFileName,
                    Arguments = new BuildSystem.Arguments()
                        .SetExecuteParallel(this.JobCount)
                        .ToString(),
                    WorkingDirectory = this.ResultRootPath,
                    UseShellExecute = false,
                    CreateNoWindow = true,
                    RedirectStandardOutput = true,
                    RedirectStandardError = true,
                    StandardOutputEncoding = Encoding.UTF8,
                    StandardErrorEncoding = Encoding.UTF8,
                };

                var handler = new DataReceivedEventHandler(
                    (object obj, DataReceivedEventArgs args) =>
                    {
                        if (args.Data == null)
                        {
                            return;
                        }

                        Console.WriteLine(args.Data);
                    });

                proc.OutputDataReceived += handler;

                proc.ErrorDataReceived += handler;

                proc.Start();

                proc.BeginOutputReadLine();

                proc.BeginErrorReadLine();

                proc.WaitForExit();

                if (proc.ExitCode != 0)
                {
                    throw new TestRunnerException(
                        string.Format(
                            "Parallel execution failed with exit code {0}",
                            proc.ExitCode));
                }
            }
        }

        private bool IsParallelizable()
        {
            return this.Tests.Concat(new[] { this.Observers }).All(
                testContexts => testContexts.All(
                    testContext => testContext.Parallelizable));
        }

        private Dictionary<ulong, uint> GetParallelTestRuleCosts()
        {
            var costs = new Dictionary<ulong, uint>();

            uint cost = 0;

            foreach (IReadOnlyList<TestContext> testContexts in this.Tests)
            {
                ++cost;

                foreach (TestContext testContext in testContexts)
                {
                    ulong unitHash = TestUnitInfo.EncodeUnitHash(
                        testContext.RoleType, testContext.UnitId);

                    costs[unitHash] = cost;
                }
            }

            foreach (TestContext testContext in this.Observers)
            {
                ulong unitHash = TestUnitInfo.EncodeUnitHash(
                    testContext.RoleType, testContext.UnitId);

                costs[unitHash] = cost;
            }

            return costs;
        }

        private Dictionary<ulong, uint> GetJobs(bool weights)
        {
            var unitHashes = new SortedSet<ulong>();

            var tests = this.Tests.Concat(new[] { this.Observers });

            foreach (IReadOnlyList<TestContext> testContexts in tests)
            {
                foreach (TestContext testContext in testContexts)
                {
                    ulong unitHash = TestUnitInfo.EncodeUnitHash(
                        testContext.RoleType, testContext.UnitId);

                    unitHashes.Add(unitHash);
                }
            }

            int targetCount = this.TargetManager.TargetEntries.Count;

            var freeJobs = Enumerable.Range(0, (int)this.JobCount)
                                     .Select(x => (uint)x).ToList();

            var targetJobs = new Queue<uint>(
                weights ? freeJobs.Take(targetCount)
                                  .OrderBy(x => this.ParallelTestRuleCosts[x])
                        : freeJobs.Take(targetCount));

            var genericJobs = new Queue<uint>(
                weights ? freeJobs.Skip(targetCount)
                                  .OrderBy(x => this.ParallelTestRuleCosts[x])
                        : freeJobs.Skip(targetCount));

            this.TargetAssigner.Flush(targetJobs
                .Select(x => this.TargetManager.TargetEntries[(int)x])
                .ToList().AsReadOnly());

            var assignedJobs = new Dictionary<ulong, uint>();

            foreach (var pair in this.TargetAssigner.AllocationTable)
            {
                assignedJobs[pair.Key] = (uint)this.TargetManager.TargetEntries
                    .Select((x, i) => new { Entry = x, Index = i })
                    .First(x => x.Entry.Name == pair.Value.Name)
                    .Index;
            }

            targetJobs = new Queue<uint>(
                targetJobs.Where(x => !assignedJobs.Values.Contains(x)));

            var jobs = new Dictionary<ulong, uint>();

            foreach (ulong unitHash in unitHashes)
            {
                uint job = 0;

                if (assignedJobs.TryGetValue(unitHash, out job))
                {
                    jobs[unitHash] = job;
                }
                else
                {
                    if (genericJobs.Count > 0)
                    {
                        jobs[unitHash] = genericJobs.Dequeue();
                    }
                    else
                    {
                        jobs[unitHash] = targetJobs.Dequeue();
                    }
                }
            }

            return jobs;
        }

        private Dictionary<ulong, TargetEntry> AppendParallelTestRule()
        {
            Dictionary<ulong, uint> jobs = this.GetJobs(true);

            var table = new Dictionary<ulong, TargetEntry>();

            foreach (var pair in this.TargetAssigner.AllocationTable)
            {
                table[pair.Key] = pair.Value;
            }

            var sourcesHead = new HashSet<string>(
                jobs.Values.Select(x => this.ParallelSources[x])
                           .Where (x => x != null))
                           .ToArray();

            string[] sources = null;

            foreach (IReadOnlyList<TestContext> testContexts in this.Tests)
            {
                uint[] testIndices = testContexts
                    .Select(testContext => jobs[
                        TestUnitInfo.EncodeUnitHash(
                            testContext.RoleType, testContext.UnitId)])
                    .ToArray();

                sources = (sources == null)
                    ? sourcesHead
                    : testIndices.Select(x => this.ParallelSources[x])
                                 .Where (x => x != null)
                                 .ToArray();

                this.ParallelTestRules.Add(
                    new TestRule(sources, testContexts));

                foreach (uint index in testIndices)
                {
                    this.ParallelSources[index] = testContexts.First().Path;
                }
            }

            if (this.Observers.Count > 0)
            {
                var observerPath = this.Observers.First().Path;

                var observerName = observerPath.Replace(
                    Path.DirectorySeparatorChar, '/');

                var observerRule = new TestRule(sourcesHead, this.Observers);

                this.ObserverEvents.Add(
                    new EventWaitHandle(
                        false, EventResetMode.ManualReset, observerName));

                observerRule.ListenerName = observerName;

                this.ParallelTestRules.Last().NotifierName = observerName;

                var observerIndices = jobs
                    .Where (x => TestUnitInfo.DecodeRoleType(x.Key) ==
                                 RoleType.Observer)
                    .Select(x => x.Value).ToArray();

                foreach (uint index in observerIndices)
                {
                    this.ParallelSources[index] = observerPath;
                }

                this.ParallelTestRules.Add(observerRule);
            }

            Dictionary<ulong, uint> costs = this.GetParallelTestRuleCosts();

            foreach (var unitHash in costs.Keys)
            {
                this.ParallelTestRuleCosts[jobs[unitHash]] += costs[unitHash];
            }

            return table;
        }

        private Dictionary<ulong, TargetEntry> AppendSerialTestRule()
        {
            Dictionary<ulong, uint> jobs = this.GetJobs(false);

            var table = new Dictionary<ulong, TargetEntry>();

            foreach (var pair in this.TargetAssigner.AllocationTable)
            {
                table[pair.Key] = pair.Value;
            }

            var sourcesHead = this.SerialSources.ToArray();

            string[] sources = sourcesHead;

            foreach (IReadOnlyList<TestContext> testContexts in this.Tests)
            {
                this.SerialTestRules.Add(new TestRule(sources, testContexts));

                this.SerialSources = new List<string>();

                this.SerialSources.Add(testContexts.First().Path);

                sources = this.SerialSources.ToArray();
            }

            if (this.Observers.Count > 0)
            {
                var observerPath = this.Observers.First().Path;

                var observerName = observerPath.Replace(
                    Path.DirectorySeparatorChar, '/');

                var observerRule = new TestRule(sourcesHead, this.Observers);

                this.ObserverEvents.Add(
                    new EventWaitHandle(
                        false, EventResetMode.ManualReset, observerName));

                observerRule.ListenerName = observerName;

                this.SerialTestRules.Last().NotifierName = observerName;

                this.SerialSources.Add(observerPath);

                this.SerialTestRules.Add(observerRule);
            }

            return table;
        }

        private string GetBuildContextFileName(TestRule rule)
        {
            return Utility.RemoveExtension(
                rule.TestContexts.First().Path, ExtensionDefinition.Result) +
                ExtensionDefinition.BuildContext;
        }

        private string GetBuildContextText(TestRule rule)
        {
            return ObjectSerializer.Serialize(new BuildContext()
            {
                EnablesVerboseMode = this.EnablesVerboseMode,
                TestContexts = rule.TestContexts.ToArray(),
                ListenerName = rule.ListenerName,
                NotifierName = rule.NotifierName,
            });
        }

        private string GetTestRunnerRuleDefinition()
        {
            var builder = new StringBuilder();

            builder.AppendLine("rule run");

            builder.AppendFormat(
                "  command = \"{0}\" $", VisualStudio.CsiPath);

            builder.AppendLine();

            var locPath = Assembly.GetExecutingAssembly().Location;

            var dirPath = Path.GetDirectoryName(locPath);

            builder.AppendFormat(
                "    /r:\"{0}\\TestRunner.Executer.dll\" $", dirPath);

            builder.AppendLine();

            builder.AppendFormat(
                "    {0} -- $config", ScriptFileName);

            return builder.ToString();
        }

        private string GetTestRunnerRuleInstance(
            IReadOnlyList<string> sources, string result, string fileName)
        {
            var builder = new StringBuilder();

            builder.AppendLine();

            var rootUri = new Uri(this.ResultRootPath + @"\");

            builder.AppendFormat(
                "build {0}: run", rootUri.MakeRelativeUri(new Uri(result)));

            foreach (string source in sources)
            {
                builder.AppendLine(" $");

                builder.AppendFormat(
                    "    {0}", rootUri.MakeRelativeUri(new Uri(source)));
            }

            builder.AppendLine();

            Uri fileNameUri = rootUri.MakeRelativeUri(new Uri(fileName));

            builder.AppendFormat("  config = {0}", fileNameUri);

            builder.AppendLine();

            builder.AppendFormat("  description = {0}", fileNameUri);

            builder.AppendLine();

            return builder.ToString();
        }

        private sealed class TestRule
        {
            internal TestRule(
                IReadOnlyList<string> sources,
                IReadOnlyList<TestContext> testContexts)
            {
                this.Sources = sources;

                this.TestContexts = testContexts;

                this.ListenerName = string.Empty;

                this.NotifierName = string.Empty;
            }

            internal IReadOnlyList<string> Sources { get; set; }

            internal IReadOnlyList<TestContext> TestContexts { get; set; }

            internal string ListenerName { get; set; }

            internal string NotifierName { get; set; }
        }
    }
}
