﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunner.Executer
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.ComponentModel.Composition.Hosting;
    using System.IO;
    using System.Linq;
    using System.Xml;

    /// <summary>
    /// テスト詳細ファイルにテスト対象名を埋め込みます。
    /// </summary>
    internal sealed class TargetNameInserter
    {
        /// <summary>
        /// TargetNameInserter クラスの新しいインスタンスを初期化します。
        /// </summary>
        internal TargetNameInserter()
        {
            using (var catalog = new AssemblyCatalog(typeof(TargetNameInserter.IRule).Assembly))
            using (var container = new CompositionContainer(catalog))
            {
                container.SatisfyImportsOnce(this);
            }
        }

        /// <summary>
        /// テスト対象名埋め込みルールを定義します。
        /// </summary>
        [InheritedExport]
        internal interface IRule
        {
            /// <summary>
            /// サポート対象となるテスト詳細ファイルの拡張子を取得します。
            /// </summary>
            /// <returns>サポート対象となるテスト詳細ファイルの拡張子です。</returns>
            string GetExtention();

            /// <summary>
            /// サポート対象となるテスト詳細ファイルかどうかを示す値を取得します。
            /// </summary>
            /// <param name="path">テスト詳細ファイルのパスです。</param>
            /// <returns>サポート対象となるテスト詳細ファイルかどうかを示す値です。</returns>
            bool Supports(string path);

            /// <summary>
            /// テスト詳細ファイルにテスト対象名を埋め込みます。
            /// </summary>
            /// <param name="dstPath">結果の出力先パスです。</param>
            /// <param name="srcPath">テスト詳細ファイルのパスです。</param>
            /// <param name="targetName">テスト対象名です。</param>
            void Insert(string dstPath, string srcPath, string targetName);
        }

        [ImportMany]
        private List<IRule> Rules { get; set; }

        /// <summary>
        /// XML ノードにテスト対象名埋め込みルールを適用します。
        /// </summary>
        /// <param name="node">XML ノードです。</param>
        /// <param name="targetName">テスト対象名です。</param>
        /// <param name="getValidAttribute">任意の XML ノードからテスト対象名の埋め込み先となるノードを取得するデリゲートです。ノードが見つからない場合は null を返します。</param>
        internal static void InsertIntoXml(XmlNode node, string targetName, Func<XmlNode, XmlNode> getTargetNode)
        {
            while (node != null)
            {
                var targetNode = getTargetNode(node);

                if (targetNode != null)
                {
                    targetNode.Value = targetName + "." + targetNode.Value;
                }

                if (node.HasChildNodes)
                {
                    InsertIntoXml(node.FirstChild, targetName, getTargetNode);
                }

                node = node.NextSibling;
            }
        }

        /// <summary>
        /// テスト詳細ファイルにテスト対象名埋め込みルールを適用します。
        /// </summary>
        /// <param name="path">テスト詳細ファイルのパスです。</param>
        /// <param name="targetPath">テスト対象のパスです。</param>
        internal void Insert(string path, string targetPath)
        {
            var foundRule = this.GetMatchingRule(path);

            if (foundRule != null && path.EndsWith(foundRule.GetExtention()))
            {
                var targetName = Path.GetFileName(targetPath).Replace('.', '_');

                var dstPath = path.Substring(0, path.Length - foundRule.GetExtention().Length) + ".ci" + foundRule.GetExtention();

                foundRule.Insert(dstPath, path, targetName);

                return;
            }

            throw new ArgumentException(string.Format("Unexpected detail path '{0}'", path));
        }

        private IRule GetMatchingRule(string path)
        {
            return this.Rules.FirstOrDefault(rule => rule.Supports(path));
        }
    }
}
