﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace SystemUpdaterNx.Initialize
{
    using System;
    using Properties;
    using TargetShell.CommandPluginBase;
    using TargetShell.Library;
    using TargetShell.PluginInterface;

    /// <summary>
    /// ファームウェアタイプ
    /// </summary>
    public enum FirmwareType
    {
        /// <summary>
        /// 開発用ファーム
        /// </summary>
        Develop,

        /// <summary>
        /// 製品用ファーム
        /// </summary>
        Prod,

        /// <summary>
        /// 製品用ログ付きファーム
        /// </summary>
        ProdWithLog,
    }

    /// <summary>
    /// 初期化のサブコマンド定義
    /// </summary>
    public class Constants
    {
        /// <summary>
        /// サブコマンドの文字列
        /// <summary>
        public const string SubCommand = "initialize";
        /// <summary>
        /// オプション fwtype の文字列
        /// <summary>
        public const string OptionPlatformName = "fwtype";
    }

    /// <summary>
    /// サブコマンドのオプション引数
    /// </summary>
    public class SubCommandArgument : BaseCommand<CommandParameter>, ISubCommandArgument
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public SubCommandArgument()
        {
            this.SubCommandName = Constants.SubCommand;
        }

        /// <summary>
        /// コマンドラインパーサにコマンドを登録します。
        /// </summary>
        /// <param name="parser">コマンドラインパーサです。</param>
        public void RegisterWith(CommandLineParser commandLineParser)
        {
            // サブコマンド
            var subcommand = commandLineParser.ActionCommand
                .AddCommand(Constants.SubCommand);
            subcommand
                .GetBuilder()
                .SetDescription(InitializeResources.Description);

            // 共通オプション
            this.OptionArgs.CommonArgs.RegisterWith(subcommand);

            // サブコマンド固有のオプション
            subcommand
                .AddValueOption(Constants.OptionPlatformName, new Action<string>(AcceptFwtypeOption))
                .GetBuilder()
                .SetDescription(InitializeResources.DescriptionPlatformName);
        }

        /// <summary>
        /// Fwtypeを解析して取得します
        /// </summary>
        /// <param name="connectionPath">解析されて取得した文字列</param>
        public void AcceptFwtypeOption(string targetFirmware)
        {
            // ConnectionPathに変換して取得する
            if (Enum.IsDefined(typeof(FirmwareType), targetFirmware))
            {
                this.DevicePluginArgs.TargetFirmware = (FirmwareType)Enum.Parse(
                        typeof(FirmwareType), targetFirmware);
            }
            else
            {
                this.DevicePluginArgs.TargetFirmware = FirmwareType.Develop;
            }
        }
    }
}
