﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace PublicNx.Settings.Command
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.IO;
    using System.Text;
    using System.Threading;
    using System.Threading.Tasks;
    using PublicNx.SettingCommon;
    using TargetShell.Library;
    using TargetShell.PluginInterface;

    /// <summary>
    /// サブコマンドプラグイン
    /// </summary>
    [Export(typeof(ITargetShellPlugin))]
    public class ImportSetting : SettingsCommon, ITargetShellPlugin
    {
        /// <summary>
        /// サブコマンドを実行する
        /// </summary>
        /// <param name="subCommandArgument">サブコマンド引数</param>
        /// <param name="deviceList">デバイスのリスト</param>
        /// <returns>true：成功  false：失敗</returns>
        public bool InvokePlugin(ISubCommandArgument subCommandArgument,
                List<TargetEntry> deviceList)
        {
            Console.WriteLine("ImportSetting start.");
            var successDeviceCount = 0;
            var totalDevices = deviceList.Count;
            try
            {
                var parameter = PrepareSettingPlugin(subCommandArgument);
                successDeviceCount = Settings(deviceList, parameter);
                Console.WriteLine(
                        "Command Result success {0}/{1}", successDeviceCount, totalDevices);
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
                return false;
            }
            Console.WriteLine("ImportSetting end.");
            return ((successDeviceCount == totalDevices) ? true : false);
        }

        /// <summary>
        /// サブコマンド引数取得
        /// <returns>サブコマンド引数</returns>
        /// </summary>
        public ISubCommandArgument GetSubCommandArgument()
        {
            var argument = new SubCommandArgument(SettingCommonConstants.ImportSubCommand);
            return argument;
        }

        /// <summary>
        /// 設定のインポートを行う
        /// </summary>
        /// <param name="deviceList">デバイスリストのパラメータ</param>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <param name="logOutputDir">シリアルフォルダパス</param>
        /// <returns>成功したデバイスの数</returns>
        public int Settings(List<TargetEntry> deviceList, CommandParameter parameter)
        {
            // 成功したデバイスの数
            var successDeviceCount = 0;

            // SdevとEdevを取得する
            var deviceNx = deviceList.FindAll(
                    device => device.HardwareType == "Sdev" || device.HardwareType == "Edev");

            // インポート元が未設定の場合は、SDK直下の特定フォルダを対象をインポートする
            if (string.IsNullOrEmpty(parameter.FilePath) &&
                    string.IsNullOrEmpty(parameter.DirectoryPath))
            {
                parameter.DirectoryPath = Path.Combine(parameter.NintendoSdkRoot,
                        SettingCommonConstants.ImportDirectory);
            }
            // インポートディレクトリ、ファイルのチェック
            if (!CheckImportDirectory(deviceNx, parameter))
            {
                return successDeviceCount;
            }
            // コマンドを並列処理で実行する
            Parallel.For(0, deviceNx.Count, id =>
            {
                var localParameter = new CommandParameter
                {
                    TargetEntry = deviceNx[id],
                    Verbose = parameter.Verbose,
                    NintendoSdkRoot = parameter.NintendoSdkRoot,
                    FilePath = parameter.FilePath,
                    DirectoryPath = parameter.DirectoryPath,
                };
                // ログ出力先を設定している場合のみログフォルダを作成する
                localParameter.SerialDirectroy =
                    TargetShellLibrary.CreateSerialNumberLogDirectory(parameter.LogOutputDir,
                    localParameter.TargetEntry.SerialNumber);
                SpecificProcessDevicePlugin(localParameter, ref successDeviceCount);
            });
            return successDeviceCount;
        }

        /// <summary>
        /// デバイスプラグイン実行メソッド
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <param name="successDeviceCount">成功デバイスの数</param>
        private void SpecificProcessDevicePlugin(
                CommandParameter parameter, ref int successDeviceCount)
        {
            try
            {
                // コマンドの構築
                var settingsManagerCommand = CreateSettingsManagerCommand(parameter);
                var returnCode = TargetShellLibrary.InvokeExe(
                        Path.Combine(parameter.NintendoSdkRoot, SettingCommonConstants.SettingsManager),
                        settingsManagerCommand.ToString(), parameter.SerialDirectroy,
                        parameter.TargetEntry.SerialNumber,
                        CommonConstants.OutputLogName);
                if (returnCode)
                {
                    Interlocked.Increment(ref successDeviceCount);
                }
            }
            catch (TargetShellLibraryException exception)
            {
                TargetShellLibrary.PrintException(exception);
            }
        }

        /// <summary>
        /// コマンドを構築
        /// </summary>
        /// <param name="parameter">コマンドのパラメータ</param>
        /// <returns>コマンド文字列</returns>
        private string CreateSettingsManagerCommand(CommandParameter parameter)
        {
            var settingsManagerCommandBuilder = new StringBuilder();
            settingsManagerCommandBuilder.AppendFormat(" Import -t {0}",
                    parameter.TargetEntry.SerialNumber);
            // 指定されたパスがフォルダかファイルか判定
            if (!string.IsNullOrEmpty(parameter.DirectoryPath))
            {
                // フォルダの場合は直下のインポートファイルを設定
                settingsManagerCommandBuilder.AppendFormat(" {0}",
                        Path.Combine(parameter.DirectoryPath,
                        parameter.TargetEntry.SerialNumber + ".json"));
            }
            else
            {
                // ファイル
                settingsManagerCommandBuilder.AppendFormat(" {0}", parameter.FilePath);
            }
            return settingsManagerCommandBuilder.ToString();
        }

        /// <summary>
        /// インポートディレクトリチェック
        /// </summary>
        /// <param name="deviceNx">デバイスリストのパラメータ</param>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <returns>True：チェックOK  False：チェックNG</returns>
        private bool CheckImportDirectory(List<TargetEntry> deviceNx, CommandParameter parameter)
        {
            // インポートファイルのチェック
            if (!string.IsNullOrEmpty(parameter.DirectoryPath))
            {
                if (!TargetShellLibrary.IsExistDirectory(parameter.DirectoryPath))
                {
                    return false;
                }
                if (!CheckImportFile(deviceNx, parameter))
                {
                    return false;
                }
            }
            else
            {
                if (!TargetShellLibrary.IsExistFile(parameter.FilePath))
                {
                    return false;
                }
            }
            return true;
        }

        /// <summary>
        /// 全デバイスのインポートファイルがあるかチェック
        /// </summary>
        /// <param name="deviceNx">デバイスリストのパラメータ</param>
        /// <param name="parameter">コマンドパラメータ</param>
        /// <returns>True：チェックOK  False：チェックNG</returns>
        private bool CheckImportFile(List<TargetEntry> deviceNx, CommandParameter parameter)
        {
            for (var i = 0; i < deviceNx.Count; i++)
            {
                var importFilePath = Path.Combine(parameter.DirectoryPath,
                        deviceNx[i].SerialNumber + ".json");
                if (!TargetShellLibrary.IsExistFile(importFilePath))
                {
                    // ファイルがすべてそろっていないときはログを出力して処理を行わないようにする
                    Console.WriteLine("No setting file of all targets.");
                    return false;
                }
            }
            return true;
        }
    }
}
