﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.Linq;
    using TargetShell.Library;
    using TargetShell.PluginInterface;

    /// <summary>
    /// TargetShellプログラムクラス
    /// </summary>
    public class Program
    {
        /// <summary>
        /// コマンドのリスト
        /// </summary>
        private static List<string[]> commandList;
        /// <summary>
        /// プラグインインスタンス
        /// </summary>
        private static InvokePlugin invokePlugin;
        /// <summary>
        /// デバイスプラグイン情報
        /// </summary>
        private static List<DevicePluginInfo> devicePluginInfo;
        /// <summary>
        /// サブコマンド引数のリスト
        /// </summary>
        private static List<ISubCommandArgument> subCommandArgumentList;
        /// <summary>
        /// TargetManagerの起動状態
        /// </summary>
        private static bool targetManagerLaunchedState = false;
        /// <summary>
        /// 実行されたサブコマンド
        /// </summary>
        private static List<ISubCommandArgument> executedSubCommandList
                = new List<ISubCommandArgument>();
        /// <summary>
        /// コマンドパーサー
        /// </summary>
        private static CommandLineParser CommandLineParser { get; set; }

        /// <summary>
        /// TargetShellメイン関数
        /// <param name="args">コンソール実行の引数</param>
        /// </summary>
        public static void Main(string[] args)
        {
            var pulginResult = false;
            // debug-log を検索し Utility にフラグ設定する
            Utility.DebugLogFlag = false;
            List<string> stringList = new List<string>();
            stringList.AddRange(args);
            foreach (string element in stringList.ToArray())
            {
                if ("debug-log" == element)
                {
                    stringList.Remove(element);
                    Utility.DebugLogFlag = true;
                    break;
                }
            }
            string[] localArgs = stringList.ToArray();
            Initializer(localArgs);
            try
            {
                pulginResult = ExecuteTargetShell(args);
            }
            catch (Utility.UtilityException utilityException)
            {
                Console.WriteLine(utilityException);
                pulginResult = false;
            }
            finally
            {
                Finalizer();
            }
            Environment.Exit(pulginResult ? 0 : 1);
        }

        /// <summary>
        /// TargetShell実行処理
        /// <param name="args">コンソール実行の引数</param>
        /// </summary>
        private static bool ExecuteTargetShell(string[] args)
        {
            var pulginResult = false;
            foreach (var subCommandArgument in subCommandArgumentList)
            {
                subCommandArgument.RegisterWith(CommandLineParser);
            }
            foreach (var command in commandList)
            {
                switch (CommandLineParser.ParseArgs(command))
                {
                    case CommandLineParserResult.Success:
                        // 引数先頭のサブコマンドから該当の解析後のサブコマンド引数を取得する
                        var parseCommand = subCommandArgumentList.Find(
                                x => x.SubCommandName == command[0]);
                        // サブコマンドプラグインを実行する
                        pulginResult = RunSubCommandPlugin(parseCommand);
                        executedSubCommandList.Add(parseCommand);
                        break;
                    case CommandLineParserResult.Failure:
                    case CommandLineParserResult.Help:
                    case CommandLineParserResult.Version:
                    default:
                        pulginResult = false;
                        break;
                }
            }
            return pulginResult;
        }

        /// <summary>
        /// TargetShellの開始処理
        /// <param name="args">引数</param>
        /// </summary>
        private static void Initializer(string[] args)
        {
            // Nintendo.ToolFoundation.CommandLine.dll の読み込みと準備
            CommandLineParser = new CommandLineParser();
            // TargetManagerの起動状態を取得
            var targetManagerProcess = Process.GetProcessesByName("NintendoTargetManager");
            if (targetManagerProcess.Length != 0)
            {
                targetManagerLaunchedState = true;
            }
            // UtilityにSDKルートを設定
            Utility.NintendoSdkRootPath = Utility.GetNintendoSdkRootPath();
            // サブコマンドを分割
            commandList = new List<string[]>();
            commandList = Utility.DivideMultiCommand(args);
            // プラグインの初期化
            invokePlugin = new InvokePlugin();
            // デバイスのプラグインをロードする
            invokePlugin.LoadDevicePlugin();
            // デバイスプラグインの情報取得
            devicePluginInfo = invokePlugin.GetDevicePluginInfo();
            subCommandArgumentList = invokePlugin.GetSubCommandArgument();
            // 重複した情報は削除する
            devicePluginInfo = devicePluginInfo.Distinct().ToList();
        }

        /// <summary>
        /// TargetShellの終了処理
        /// </summary>
        private static void Finalizer()
        {
            // サブコマンドがTargetManagerの終了を許可しているか確認する
            var keepTargetManager =
                    executedSubCommandList.Exists(x => x.KeepTargetManagerFlag == true);
            // TargetManagerを起動前の状況に戻す
            if (!targetManagerLaunchedState && !keepTargetManager)
            {
                TargetShellLibrary.StopTargetManager();
            }
        }

        /// <summary>
        /// サブコマンドプラグインを実行する
        /// <param name="subCommandArgument">サブコマンド引数</param>
        /// <returns>成功:true 失敗:false</returns>
        /// </summary>
        private static bool RunSubCommandPlugin(
                ISubCommandArgument subCommandArgument)
        {
            var pulginResult = false;
            var defaultTarget = string.Empty;
            // ターゲットの準備
            SetupTarget(out List<TargetEntry> registereDeviceList, out defaultTarget);
            // ターゲットの検索
            var deviceList = Utility.SearchDevice(subCommandArgument, devicePluginInfo);
            // デバイスが見つかった時のみプラグインを実行する
            if (deviceList.Count > 0)
            {
                pulginResult = invokePlugin.Run(subCommandArgument, deviceList);
                // プラグイン実行終了でターゲットの登録状態を元に戻す
                TeardownTarget(deviceList, registereDeviceList, defaultTarget);
                if (!pulginResult)
                {
                    // サブコマンドプラグインが失敗したらTargetShell自体を停止する
                    throw new Utility.UtilityException(
                        $"Error InvokePlugin. subcommand = {subCommandArgument.SubCommandName}");
                }
            }
            else
            {
                pulginResult = true;
            }
            return pulginResult;
        }

        /// <summary>
        /// ターゲットの準備を行う
        /// <param name="registereDeviceList">TargetManagerに登録されているターゲットリスト</param>
        /// <param name="defaultTarget">既定のターゲット</param>
        /// </summary>
        private static void SetupTarget(out List<TargetEntry> registereDeviceList,
                out string defaultTarget)
        {
            if (targetManagerLaunchedState)
            {
                // TargetShell起動時にTargetManagerが起動していたら終了する
                // ヘルプの実行の場合はTargetManagerの制御は不要なので、サーチ前に処理する
                TargetShellLibrary.StopTargetManager();
            }
            // TargetManagerに登録されているデバイスリストを取得しておく
            registereDeviceList = Utility.GetRegisteredTarget(devicePluginInfo);
            // TargetManagerに登録されている既定の開発機を取得しておく
            if (!Utility.GetDefaultTarget(out defaultTarget))
            {
                // 既定の開発機の取得に失敗しても処理は継続させるためログを出力する
                Console.WriteLine("GetDefaultTarget error");
            }
            return;
        }

        /// <summary>
        /// ターゲットの片付けを行う
        /// <param name="deviceList">プラグイン処理を要求したデバイスリスト</param>
        /// <param name="registereDeviceList">TargetManagerに登録されていたデバイスリスト</param>
        /// <param name="defaultTarget">既定のターゲット</param>
        /// </summary>
        private static void TeardownTarget(List<TargetEntry> deviceList,
                List<TargetEntry> registereDeviceList,
                string defaultTarget)
        {
            // デバイスリストチェック
            if (deviceList == null || registereDeviceList == null)
            {
                // 不正なデバイスリストの場合はログだけ残して処理は行わない
                Console.WriteLine("TeardownTarget DeviceList is null");
                return;
            }
            // プラグイン実行終了でTargetManagerから消去されているデバイスを登録しなおす
            if (!Utility.ReregisterTarget(registereDeviceList, deviceList, devicePluginInfo))
            {
                // TargetManagerへ登録失敗の場合はログだけ出力する
                Console.WriteLine("ReregisterTarget error");
            }
            // プラグイン実行終了で既定の開発機を元に戻す
            if (!(string.IsNullOrEmpty(defaultTarget) ||
                defaultTarget.Contains(CommandLineToolConstants.NoDefaultTarget)))
            {
                if (!Utility.SetDefaultTarget(defaultTarget))
                {
                    // TargetManagerへ登録失敗の場合はログだけ出力する
                    Console.WriteLine("SetDefaultTarget error");
                }
            }
        }
    }
}
