﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TargetShell
{
    using System;
    using System.Collections.Generic;
    using System.ComponentModel.Composition;
    using System.ComponentModel.Composition.Hosting;
    using System.IO;
    using System.Reflection;
    using TargetShell.PluginInterface;

    /// <summary>
    /// プラグイン実行クラス
    /// </summary>
    public class InvokePlugin
    {
        /// <summary>
        /// プラグイン呼び出しモジュール
        /// </summary>
        private PluginCaller pluginCaller;
        /// <summary>
        /// コンストラクタ
        /// </summary>
        public InvokePlugin()
        {
            this.pluginCaller = new PluginCaller();
            this.pluginCaller.Load();
        }
        /// <summary>
        /// プラグインを実行する
        /// </summary>
        /// <param name="subCommandArgument">サブコマンド引数</param>
        /// <param name="deviceList">サブコマンドを実行するデバイスのリスト</param>
        public bool Run(ISubCommandArgument subCommandArgument, List<TargetEntry> deviceList)
        {
            var result = this.pluginCaller.InvokePlugin(subCommandArgument, deviceList);
            return result;
        }
        /// <summary>
        /// デバイスプラグインを読み込む
        /// </summary>
        public void LoadDevicePlugin()
        {
            this.pluginCaller.LoadDevicePlugin();
        }
        /// <summary>
        /// デバイスプラグインの情報を取得する
        /// </summary>
        public List<DevicePluginInfo> GetDevicePluginInfo()
        {
            var devicePluginInfo = this.pluginCaller.GetDevicePluginInfo();
            return devicePluginInfo;
        }
        /// <summary>
        /// サブコマンド引数を取得する
        /// <returns>サブコマンド引数のリスト</returns>
        /// </summary>
        public List<ISubCommandArgument> GetSubCommandArgument()
        {
            var subCommandArgumentList = this.pluginCaller.GetSubCommandArgument();
            return subCommandArgumentList;
        }
    }

    /// <summary>
    /// プラグイン呼び出しの実装クラス
    /// </summary>
    public class PluginCaller
    {
        /// <summary>
        /// コンポジションコンテナ
        /// </summary>
        private CompositionContainer container;
        /// <summary>
        /// プラグインのインスタンスを受け取るためのフィールド
        /// </summary>
        [ImportMany]
        private IEnumerable<Lazy<ITargetShellPlugin>> operations = null;
        /// <summary>
        /// プラグインのロード
        /// </summary>
        public void Load()
        {
            // TergetShellの情報からパスとディレクトリ取得
            Assembly myAssembly = Assembly.GetEntryAssembly();
            string path = myAssembly.Location;
            string directory = Path.GetDirectoryName(path);
            // カタログにプラグインの情報を追加する
            using (var catalog = new AggregateCatalog())
            {
                catalog.Catalogs.Add(new DirectoryCatalog(
                        Path.Combine(directory, TargetShellPluginConstants.PluginDirectory)));
                // コンテナに集約プラグインで実行できるようにする(第2引数はスレッドセーフ有効)
                this.container = new CompositionContainer(catalog, true);
                try
                {
                    this.container.ComposeParts(this);
                }
                catch (CompositionException compositionException)
                {
                    var error = compositionException.Errors;
                    Console.WriteLine(compositionException.ToString());
                }
            }
        }
        /// <summary>
        /// プラグインの実行
        /// <param name="subCommandArgument">サブコマンド引数</param>
        /// <param name="deviceList">サブコマンドを実行するデバイスのリスト</param>
        /// </summary>
        public bool InvokePlugin(ISubCommandArgument subCommandArgument, List<TargetEntry> deviceList)
        {
            var result = false;
            // プラグイン実行処理
            foreach (Lazy<ITargetShellPlugin> plugin in operations)
            {
                // 引数の先頭のサブコマンドを判定してプラグインを実行する
                if (plugin.Value.GetSubCommandArgument().SubCommandName ==
                        subCommandArgument.SubCommandName)
                {
                    result = plugin.Value.InvokePlugin(subCommandArgument, deviceList);
                }
            }
            return result;
        }

        /// <summary>
        /// デバイスプラグインを読み込む
        /// </summary>
        public void LoadDevicePlugin()
        {
            // デバイスプラグイン読み込む
            foreach (Lazy<ITargetShellPlugin> plugin in operations)
            {
                plugin.Value.LoadDevicePlugin();
            }
            return;
        }

        /// <summary>
        /// デバイスプラグインの情報を取得する
        /// <returns>デバイスプラグイン情報</returns>
        /// </summary>
        public List<DevicePluginInfo> GetDevicePluginInfo()
        {
            var devicePluginInfo = new List<DevicePluginInfo>();
            // デバイスプラグインの情報取得
            foreach (Lazy<ITargetShellPlugin> plugin in operations)
            {
                devicePluginInfo.AddRange(plugin.Value.GetDevicePluginInfo());
            }
            return devicePluginInfo;
        }

        /// <summary>
        /// サブコマンド引数取得
        /// <returns>サブコマンド引数のリスト</returns>
        /// </summary>
        public List<ISubCommandArgument> GetSubCommandArgument()
        {
            var objectList = new List<ISubCommandArgument>();
            // プラグインからサブコマンド引数を取得する
            foreach (Lazy<ITargetShellPlugin> plugin in operations)
            {
                objectList.Add(plugin.Value.GetSubCommandArgument());
            }
            return objectList;
        }
    }
}
