﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace SampleLogFileCompare
{
    using System;
    using System.Collections.Generic;
    using System.IO;
    using System.Linq;
    using System.Text;
    using System.Text.RegularExpressions;

    /// <summary>
    /// ２つのログファイルを比較するクラス
    ///     以下３つのログ順番での比較が可能。
    ///     ・完全一致
    ///     ・正解パターンの行間にあるログは無視する
    ///     ・ログ順番不定
    /// </summary>
    internal class SampleLogFileComparator
    {
        /// <summary>
        /// 実機で動作するサンプルが出力したことを表すヘッダ文字列
        /// </summary>
        private const string TargetHeader = "[target] ";
        /// <summary>
        /// RunOnTarget が出力したことを表すヘッダ文字列
        /// </summary>
        private const string TraceHeader = "[trace] ";

        /// <summary>
        /// 正解パターンファイル名
        /// </summary>
        private string patternFile;
        /// <summary>
        /// TestRunnerが出力したログファイル名
        /// </summary>
        private string logFile;
        /// <summary>
        /// ログ比較順番
        /// </summary>
        private LogSequence logSequenceType;
        /// <summary>
        /// 文字列比較タイプ
        /// </summary>
        private MatchPattern matchPatternType;
        /// <summary>
        /// ヘッダ文字列を無視するかを指定
        /// </summary>
        private bool ignoresRunOnTargetLog;
        /// <summary>
        /// 正解パターンファイルのリスト
        /// </summary>
        private List<RegexComparableString> patternFileList;
        /// <summary>
        /// TestRunnerが出力したログファイルのリスト
        /// </summary>
        private List<RegexComparableString> logFileList;
        /// <summary>
        /// 正解パターンファイルでエラーを起こした行の内容
        /// </summary>
        private string patternErrorLine;
        /// <summary>
        /// TestRunnerが出力したログファイルでエラーを起こした行の内容
        /// </summary>
        private string logErrorLine;
        /// <summary>
        /// 正解パターンファイルでエラーを起こした行番号
        /// </summary>
        private int patternErrorLineNum;
        /// <summary>
        /// TestRunnerが出力したログファイルでエラーを起こした行番号
        /// </summary>
        private int logErrorLineNum;

        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="patternFileArg">正解パターンファイル</param>
        /// <param name="logFileArg">TestRunnerが出力したログファイル</param>
        /// <param name="logSequenceTypeArg">ログ順番比較タイプ</param>
        /// <param name="matchPatternArg">文字列比較タイプ</param>
        internal SampleLogFileComparator(string patternFileArg,
                                         string logFileArg,
                                         LogSequence logSequenceTypeArg,
                                         MatchPattern matchPatternArg,
                                         bool ignoresRunOnTargetLogArg)
        {
            this.patternFile = patternFileArg;
            this.logFile = logFileArg;
            this.logSequenceType = logSequenceTypeArg;
            this.matchPatternType = matchPatternArg;
            this.ignoresRunOnTargetLog = ignoresRunOnTargetLogArg;
            this.patternErrorLine = string.Empty;
            this.logErrorLine = string.Empty;
            this.patternErrorLineNum = 0;
            this.logErrorLineNum = 0;
            this.patternFileList = null;
            this.logFileList = null;
        }

        /// <summary>
        /// ログ比較順番の定義
        /// ・完全一致
        /// ・正解パターンの行間にあるログは無視する
        /// ・ログ順番不定
        /// </summary>
        internal enum LogSequence
        {
            // 完全一致
            SequenceMatch,
            // 正解パターンの行間にあるログは無視する
            SequenceBetweenLinesDontCare,
            // ログ順番不定
            SequenceRandom,
        }

        /// <summary>
        /// 文字列比較タイプ
        /// ・完全一致
        /// ・正規表現一致
        /// </summary>
        internal enum MatchPattern
        {
            // 完全一致
            PerfectMatch,
            // 正規表現一致
            RegexMatch,
        }

        /// <summary>
        /// ログファイルを比較する関数
        /// </summary>
        /// <returns>一致／不一致</returns>
        internal bool LogfileCompare()
        {
            try
            {
                this.patternErrorLine = string.Empty;
                this.logErrorLine = string.Empty;
                this.patternErrorLineNum = 0;
                this.logErrorLineNum = 0;

                if (string.IsNullOrEmpty(this.patternFile))
                {
                    Console.WriteLine("Argument is 'null' or 'empty' : Pattern Logfile");
                    return false;
                }
                if (string.IsNullOrEmpty(this.logFile))
                {
                    Console.WriteLine("Argument is 'null' or 'empty' : Actual Logfile");
                    return false;
                }

                this.patternFileList = LoadLogfile(this.patternFile);
                if (this.patternFileList == null)
                {
                    Console.WriteLine($"Pattern Logfile : {this.patternFile} does not exist.");
                    return false;
                }
                if (this.ignoresRunOnTargetLog)
                {
                    this.logFileList = LoadLogfileIgnoreHeader(this.logFile);
                }
                else
                {
                    this.logFileList = LoadLogfile(this.logFile);
                }
                if (this.logFileList == null)
                {
                    Console.WriteLine($"Actual Logfile : {this.logFile} does not exist.");
                    return false;
                }

                var isEqual = true;
                switch (this.logSequenceType)
                {
                    case LogSequence.SequenceMatch:
                        isEqual = SequenceMatch();
                        break;
                    case LogSequence.SequenceBetweenLinesDontCare:
                        isEqual = SequenceBetweenLinesDontCare();
                        break;
                    case LogSequence.SequenceRandom:
                        isEqual = SequenceRandom();
                        break;
                    default:
                        throw new ArgumentException("invalid LogSequenceType.");
                }

                //結果を表示する
                if (!isEqual)
                {
                    Console.WriteLine("Logfile is not matching.");
                    Console.WriteLine($"Pattern Logfile : {this.patternFile}");
                    Console.WriteLine(
                        $"# {this.patternErrorLineNum.ToString()} : {this.patternErrorLine}");
                    Console.WriteLine($"Actual Logfile : {this.logFile}");
                    Console.WriteLine($"# {this.logErrorLineNum.ToString()} : {this.logErrorLine}");
                }
                return isEqual;
            }
            catch (Exception exception)
            {
                Console.WriteLine($"Unexpected exception occurred. {exception.ToString()}");
                return false;
            }
        }

        /// <summary>
        /// ログファイルをロードしList化する
        /// </summary>
        /// <param name="logFileArg">ロードするログファイル名</param>
        /// <returns>List化したログファイル</returns>
        private List<RegexComparableString> LoadLogfile(string logFileArg)
        {
            if (!File.Exists(logFileArg))
            {
                return null;
            }

            var logfileList = new List<RegexComparableString>();

            using (var streamReader = new StreamReader(logFileArg, Encoding.UTF8))
            {
                var logLine = string.Empty;
                while ((logLine = streamReader.ReadLine()) != null)
                {
                    var listItem = new RegexComparableString(logLine, this.matchPatternType);
                    logfileList.Add(listItem);
                }
            }
            return logfileList;
        }

        /// <summary>
        /// ログファイルをロードしヘッダ文字列を無視したうえでList化する
        /// </summary>
        /// <param name="logFileArg">ロードするログファイル名</param>
        /// <returns>List化したログファイル</returns>
        private List<RegexComparableString> LoadLogfileIgnoreHeader(string logFileArg)
        {
            if (!File.Exists(logFileArg))
            {
                return null;
            }

            var logfileList = new List<RegexComparableString>();

            using (var streamReader = new StreamReader(logFileArg, Encoding.UTF8))
            {
                var logLine = string.Empty;
                while ((logLine = streamReader.ReadLine()) != null)
                {
                    // [trace]、[target] のヘッダ文字列を無視する
                    if (logLine.StartsWith(TraceHeader))
                    {
                        continue;
                    }
                    if (logLine.StartsWith(TargetHeader))
                    {
                        logLine = logLine.Remove(0, TargetHeader.Count());
                    }
                    var listItem = new RegexComparableString(logLine, this.matchPatternType);
                    logfileList.Add(listItem);
                }
            }
            if (logfileList.Count > 0 && logfileList.Last().TextValue == string.Empty)
            {
                logfileList.RemoveAt(logfileList.Count - 1);
            }
            return logfileList;
        }

        /// <summary>
        /// ログ順番を完全一致でログファイルを比較する関数
        /// </summary>
        /// <returns>一致／不一致</returns>
        private bool SequenceMatch()
        {
            var isEqual = true;
            var patternErrorLineTemp = string.Empty;
            var patternErrorLineNumTemp = 0;
            var logErrorLineTemp = string.Empty;
            var logErrorLineNumTemp = 0;

            foreach (var patternLine in this.patternFileList)
            {
                patternErrorLineNumTemp++;
                logErrorLineNumTemp++;

                if (this.logFileList.Count == 0)
                {
                    this.patternErrorLine = patternLine.TextValue;
                    this.logErrorLine = "<Empty Line>";
                    this.patternErrorLineNum = patternErrorLineNumTemp;
                    this.logErrorLineNum = logErrorLineNumTemp;
                    isEqual = false;
                    break;
                }

                // 1行目を削除する
                var logLine = this.logFileList.First();
                this.logFileList.RemoveAt(0);

                if (!patternLine.Compare(logLine.TextValue))
                {
                    this.patternErrorLine = patternLine.TextValue;
                    this.logErrorLine = logLine.TextValue;
                    this.patternErrorLineNum = patternErrorLineNumTemp;
                    this.logErrorLineNum = logErrorLineNumTemp;
                    isEqual = false;
                    break;
                }
            }
            if ((isEqual) && (this.logFileList.Count != 0))
            {
                patternErrorLineNumTemp++;
                logErrorLineNumTemp++;
                this.patternErrorLine = "<Empty Line>";
                this.logErrorLine = this.logFileList.First().TextValue;
                this.patternErrorLineNum = patternErrorLineNumTemp;
                this.logErrorLineNum = logErrorLineNumTemp;
                isEqual = false;
            }

            return isEqual;
        }
        /// <summary>
        /// ログ順番を正解パターンの行間にあるログは無視する比較方法でログファイルを比較する関数
        /// </summary>
        /// <returns>一致／不一致</returns>
        private bool SequenceBetweenLinesDontCare()
        {
            var isEqual = true;
            var patternErrorLineNumTemp = 0;
            var logFileListCount = 0;

            foreach (var patternLine in this.patternFileList)
            {
                isEqual = false;
                patternErrorLineNumTemp++;

                for (var i = logFileListCount; i < this.logFileList.Count; i++)
                {
                    if (patternLine.Compare(this.logFileList[i].TextValue))
                    {
                        logFileListCount = (i + 1);
                        isEqual = true;
                        break;
                    }
                }
                if (!isEqual)
                {
                    this.patternErrorLine = patternLine.TextValue;
                    this.logErrorLine = "<Not Exist>";
                    this.patternErrorLineNum = patternErrorLineNumTemp;
                    this.logErrorLineNum = 0;
                    break;
                }
            }
            return isEqual;
        }
        /// <summary>
        /// ログ順番を順番不定でログファイルを比較する関数
        /// </summary>
        /// <returns>一致／不一致</returns>
        private bool SequenceRandom()
        {
            var index = -1;
            var isEqual = true;
            var errorLineNum1Temp = 0;

            foreach (var patternLine in this.patternFileList)
            {
                isEqual = false;
                errorLineNum1Temp++;

                // 正解パターンファイルと一致する行を検索
                index = this.logFileList.FindIndex(logLine => patternLine.Compare(logLine.TextValue));
                // 0以上の場合は一致する行が存在する
                isEqual = (0 <= index);

                if (!isEqual)
                {
                    this.patternErrorLine = patternLine.TextValue;
                    this.logErrorLine = "<Not Exist>";
                    this.patternErrorLineNum = errorLineNum1Temp;
                    this.logErrorLineNum = 0;
                    break;
                }
                // 一致した行を削除する
                this.logFileList.RemoveAt(index);
            }
            return isEqual;
        }
    }
    /// <summary>
    /// 正規表現で比較可能な文字列クラス
    /// </summary>
    internal class RegexComparableString
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="textValue">文字列本体</param>
        /// <param name="matchPattern">文字列比較タイプ</param>
        internal RegexComparableString(string textValue,
                SampleLogFileComparator.MatchPattern matchPatternType)
        {
            this.TextValue = textValue;
            this.MatchPatternType = matchPatternType;
        }

        /// <summary>
        /// 文字列本体
        /// </summary>
        internal string TextValue { get; set; }
        /// <summary>
        /// 文字列比較タイプ
        /// </summary>
        internal SampleLogFileComparator.MatchPattern MatchPatternType { get; set; }

        /// <summary>
        /// 文字列を比較する関数
        /// </summary>
        /// <param name="comparedText">
        /// 比較対象の文字列（この文字列は正規表現として解釈しない）
        /// </param>
        /// <returns>一致／不一致</returns>
        public bool Compare(string comparedText)
        {
            if (this.MatchPatternType == SampleLogFileComparator.MatchPattern.PerfectMatch)
            {
                // 完全一致が指定されている場合
                return (this.TextValue == comparedText);
            }
            else
            {
                // 正規表現一致が指定されている場合
                if (this.TextValue == string.Empty)
                {
                    return (comparedText == string.Empty);
                }
                if (comparedText == string.Empty)
                {
                    return (this.TextValue == string.Empty);
                }

                return Regex.IsMatch(comparedText, this.TextValue, RegexOptions.None);
            }
        }
    }
}
