﻿#
#    COMMAND [-ConfigFile <files>] [-OutputDir <dirname>] [-PreApiUrl <url>]
#            [-Overwrite] [-English] [-Detail|-Quiet]
#
#       -ConfigFile <name>            Config ファイル。指定がない場合、言語によって
#                                     ".\config\ja\ChangelogConfig.xml" か ".\config\en\ChangelogConfig.xml" か
#       -OutputDir <dirname>          出力フォルダ。
#                                     指定がなければ、カレントを基準にし、指定があれば、そのフォルダを基準にする。
#                                     例えば html という指定なら html/ChangeLogXXX.html や
#                                     html/ReleaseNotes/ChangeLogXXX_XXX.html が生成される。
#       -TagInfoDir                   TagInfo.xml を参照するためのフォルダ指定。
#                                     ここを起点に、コンフィグに書かれたファイルを参照するようになる。
#                                     (コンフィグには Api/Build/TagInfo.xml のような指定が書かれているのでそれを差し引いた指定にする)
#                                     ファイル名を連結するので、終端に / や \ が必要。
#                                     指定しなければ ../../../../Documents/Outputs/ となる。
#       -Overwrite                    結果を上書きするか
#       -English                      指定すると、表示文字列などに英語版を用いるよう、用語ファイルの読み込みを切りかえる
#       -Detail                       動作の情報を詳細に表示する。省略化で、省略時は詳細を表示しない。
#                                     Detail と Quiet は片方しか指定できません。
#       -Quiet                        動作の情報の表示を出来るだけ抑制する。エラーや警告だけ表示する。
#                                     Detail と Quiet は片方しか指定できません。
#
#    (例) COMMAND -OutputDir html -Detail
#
#    .\config\ChangelogConfig.xml にコンフィグデータを書き込んでおく必要がある。
#      MainId:      ログのID割り振りに使われる。通常連番だが文字列でも問題ない。
#      PackageName: パッケージ名
#      ReleaseDate: リリース日
#      TopInformation: 先頭に置く重要なお知らせ。改行は "&lt;be>" とする。
#      PackageSupplementation : 追加情報
#
param
(
    [string]$AuthenticationFile = "",
    [String]$ConfigFile = "",
    [String]$OutputDir = ".",
    [String]$TagInfoDir = "../../../../Documents/Outputs/",
    [switch]$Overwrite = $false,
    [switch]$English = $false,
    [switch]$Detail = $false,
    [switch]$Quiet = $false
)

$AllCsv = @()   # 複数ファイルから CSVデータをすべてまとめて保持
$ResourceDir = Split-Path (Split-Path (Split-Path (Split-Path $MyInvocation.MyCommand.Path -Parent) -Parent) -Parent) -Parent

#---- トップHTMLのファイル名
$OutputName_TopHtml  = "ChangeLog.html"

#----------------------------------------------------------------
#---------------- 表示レベル
$LogLevel = 1
if ( $Detail -eq $true -and $Quiet -eq $true )
{
    Write-Host "*** Error: Cannot -Detail and -Quiet options are specified at a time."
}
if ( $Detail -eq $true ){ $LogLevel = 2 }
if ( $Quiet  -eq $true ){ $LogLevel = 0 }

function Write-HostLog( $level, $str )
{
    if ( $level -eq "Detail" )
    {
        if ( $LogLevel -ge 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "Log" )
    {
        if ( $LogLevel -ge 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "Error" )
    {
        Write-Host $str
    }
    elseif ( $level -eq "DetailOnly" )
    {
        if ( $LogLevel -eq 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "LogOnly" )
    {
        if ( $LogLevel -eq 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "QuietOnly" )
    {
        if ( $LogLevel -eq 0 ){ Write-Host $str }
    }
}
function Write-Error( $str )
{
    Write-HostLog "Error" ("*** Error: "+$str)
}
function Write-Warning( $str )
{
    Write-HostLog "Error" ("*** Warning: "+$str)
}
function Write-Detail( $str )
{
    Write-HostLog "Detail" $str
}
function Write-Log( $str )
{
    Write-HostLog "Log" $str
}

#----------------------------------------------------------------
# コンフィグファイルの読み込み
if ( "" -eq $ConfigFile )
{
    if ( $English -eq $true )
    {
        $ConfigFile = ".\config\en\ChangelogConfig.xml"
    }
    else
    {
        $ConfigFile = ".\config\ja\ChangelogConfig.xml"
    }
}

if ( -not ( Test-Path -path $ConfigFile ) )
{
    Write-Error( "Config file {0} not exists." -f $ConfigFile )
    exit 1
}
[Xml]$Config = Get-Content -Encoding UTF8 $ConfigFile

. ".\loadConfigRules.ps1"

# Packages に、履歴リスト
$Packages = [array](Make-PackageList( $Config ))

#----------------------------------------------------------------
# パッケージ名に対してhtml名を取得
function Get-MainHtmlName( $pk )
{
    return "ChangeLog_" + ( $pk -creplace "\.", "_" -creplace " ","_" ) + ".html"
}
function Get-MainComponentHtmlName( $pk )
{
    return "ChangeLog_" + ( $pk -creplace "\.", "_" -creplace " ","_" ) + "_component.html"
}
function Get-OldHtmlName( $pk )
{
    return "ChangeLog_" + ( $pk  -creplace "\.", "_" -creplace " ","_" ) + "_old.html"
}


#================================================================================
#----------------
# 対象 html を config から調べる


$items = @()

. .\loadConfigInConfluence.ps1 -AuthenticationFile $AuthenticationFile -AliasFile $AliasFile -ConfigFile $ConfigFile


# 現行バージョンの処理

$majorVersionList = @();
$majorVersionList += Get-MajorVersionList

# メジャーバージョンごとにループ
for( $i=0; $i -lt $majorVersionList.Length; $i++)
{
	$majorVersionStr = $majorVersionList[$i]  -replace "\.", "_"
	$OutputVersionDirName = "ReleaseNotes\Changelog-" + $majorVersionStr
	
	$versionListNum = Get-EachMajorVersionsListNum($majorVersionList[$i])

	# メジャーバージョンの中の各バージョンでループしてそれぞれの HTML を出力する
	for( $j=0; $j -lt $versionListNum; $j++)
	{
		$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-EachMajorVersionsList $majorVersionList[$i] $j

		$eachVersionStr = $PackageVersionList[0]  -replace "\.", "_"
		$eachVersionHtmlName = "Changelog-{0}.html" -f $eachVersionStr

		# 絶対パス
		$OutFile = $OutputDir + "\" + $OutputVersionDirName + "\" + $eachVersionHtmlName
		$items += $OutFile
	}
}

# 過去バージョンの処理

$OutputDirName = "ReleaseNotes\Changelog-Old"
$versionListNum = Get-OldChangelogListNum

for( $i=0; $i -lt $versionListNum; $i++)
{
	$PackageVersionList, $PackageVersionString, $PackageSupplementation = Get-OldChangelogList $i
	
	$eachVersionStr = $PackageVersionList[0]  -replace "\.", "_"
	$eachVersionHtmlName = "Changelog-{0}.html" -f $eachVersionStr

	# 絶対パス
	$OutFile = $OutputDir + "\" + $OutputDirName + "\" + $eachVersionHtmlName
	$items += $OutFile
}

$TargetFiles = ($items -join ",")

#---- Top ファイル
$OutFile = $OutputName_TopHtml
if ( "" -ne $OutputDir )
{
    $OutFile = $OutputDir + "/" + $OutFile
}
$TargetTopFile = $OutFile

#----------------
# 対象タグファイルを config から読む
$items = @()
$TagInfoNum = Get-NumberOfTagInfo
for( $IndexPList=0; $IndexPList -lt $TagInfoNum; $IndexPList++ )
{
    $TagInfoName = Get-TagInfoName( $IndexPList )
    if ( ! ( Test-Path ($TagInfoDir + $TagInfoName) ) )
    {
        continue
    }
    $TagInfoName = $(Get-ChildItem ($TagInfoDir + $TagInfoName)).FullName
    $items += $TagInfoName
}

#---- 基本の引数
$args0 = @()
if ( $items.Length -gt 0 )
{
    $args0 += "--tag-info-xml"
    $args0 += [string]($items -join ",")
    Write-Detail ( "TagInfoXml file : " + [string]($items -join ",") )
}
if ( $Overwrite -eq $false )
{
    $args0 += "--save-as"
}

#---- 基本の引数に必要な情報を足してリンク解決のスクリプト呼び出し(mainファイル)
$args = @()
$args += $args0
$args += @( "--input-file", $TargetFiles, "--pre-api-url", "../../../../" )
if ( $Detail -eq $true )
{
    $args += "--detail"
}

$exePath = "../../../../Integrate/Outputs/AnyCPU/Tools/ApiLinkRewriter/ApiLinkRewriter/Release/ApiLinkRewriter.exe"
& $exePath $args

#---- 基本の引数に必要な情報を足してリンク解決のスクリプト呼び出し(topファイル)
$args = $args0
$args += @( "--input-file", $TargetTopFile, "--pre-api-url", "../../../../" )
if ( $Detail -eq $true )
{
    $args += "--detail"
}
Write-Detail ("Convert : " + $TargetTopFile)

$exePath = "../../../../Integrate/Outputs/AnyCPU/Tools/ApiLinkRewriter/ApiLinkRewriter/Release/ApiLinkRewriter.exe"
& $exePath $args

#---- 終了表示
Write-Log "Done."
