﻿#
#    (ライブラリ用変更履歴)
#    COMMAND [-AuthenticationFile <file>] [-Project <project>] [-User <user>] [-Password <password>] -Version <version> -Package <package> [-OutputName <name>]
#            [-ConfigFile <configfile>] [-CheckHtml] [-CheckHtmlName <htmlname>]
#            [-MaxArticle <maxarticle>] [-English|-Japanese] [-Detail|-Quiet]
#
#    (ドキュメント用変更履歴)
#    COMMAND [-AuthenticationFile <file>] [-Project <project>] [-User <user>] [-Password <password>] -Version <version> -Package <package> [-OutputName <name>]
#            [-ConfigFile <configfile>] [-CheckHtml] [-CheckHtmlName <htmlname>]
#            [-MaxArticle <maxarticle>] [-English|-Japanese] [-Detail] -Document
#
#       -AuthenticationFile <file>    認証情報が書かれている xml ファイル指定。
#                                     User, Password, Project 全てが指定されるのなら指定不要
#                                     <file> に DEFAULT が指定されると .\config\Authentication.xml と見なす。
#       -AliasFile <file>             エイリアス情報が書かれている xml ファイル指定。
#                                     指定がなければ .\config\Aliases.xml を参照する。
#                                     指定したくなければ -AliasFile "" を指定のこと。
#       -ComponentRuleFile <file>     コンポーネント情報変換ファイルの指定。
#                                     指定がなければ .\cofig\componentRule.json となる。
#       -Project <project>            対象プロジェクト。省略時は "SIGLO" となる。
#                                     -ConfigFile によってファイルから設定を読むときは、
#                                     <Project>タグで指定することも可能であり、そちらが優先される。
#       -User <user>                  JIRA アクセスの認証のためのユーザ名。  認証情報 xml ファイルより優先する。
#       -Password <password>          JIRA アクセスの認証のためのパスワード。認証情報 xml ファイルより優先する。
#       -Version <version>            (ライブラリ向け) 検索で指定するバージョン。
#                                     JIRAの「修正バージョン」との比較。
#       -Package <package>            (ライブラリ向け) 検索で指定するパッケージ。
#                                     JIRAの「変更履歴の対象パッケージ」との比較。
#       -OutputName <name>            出力する中間出力ファイル名。
#                                     <name> は .\cvsdata\datFile.dat のようにフォルダを含めることもできる。
#                                     省略時はバージョンやパッケージ名からシステムが生成する。
#                                     通常は省略する。
#       -ConfigFile <configfile>      <configfile> を読み、何の履歴データを収集するかをそこから決める。
#                                     <configfile> は outputChangeLog.ps1 で使うものと同じなので、
#                                     それと違う収集をするならダミーで収集用の config ファイルを作成すること。
#                                     <configfile> に DEFAULT を指定するとデフォルトのものとなる。
#                                     なおこの指定があると -Version や -Package は不要であり指定されていても無視される。
#       -CheckHtml                    フィールドチェックを行うか。結果を出力するhtmlファイル名はバージョン、パッケージから決定される。
#       -CheckHtmlName <htmlname>     -CheckHtml 指定で結果を出力するファイル名を指定する。
#                                     複数パッケージが対象になっていると同じファイルに書いていくので実質単体調査用です。
#       -MaxArticle <maxarticle>      調査する JIRA 課題の最大数。省略時(あるいは0を指定時) はすべて。
#                                     実験的に一部を取得する用途で用意されていますので、普段は指定なしで結構です。
#       -English                      指定があると英語用データと判断
#       -Japanese                     指定があると日本語用データと判断(デフォルト)
#       -Document                     ドキュメント向け変更履歴の収集(まだ収集していないもの)
#       -Old                          ドキュメント向け変更履歴の収集(収集済みなもの)
#       -Detail                       動作の情報を詳細に表示する。省略可で、省略時は詳細を表示しない。
#                                     Detail と Quiet は片方しか指定できません。
#       -Quiet                        動作の情報の表示を出来るだけ抑制する。エラーや警告だけ表示する。
#                                     Detail と Quiet は片方しか指定できません。
#
#   (例) COMMAND -AuthenticationFile auth.xml -Version 0.4.0 -Package NintendoSDK -MaxArticle 500
#   (例) COMMAND -Project SIGLO -Version "0.5.0 (20150227)" -Package NintendoSDK -User ninten -Password "123456AAA"
#   (例) COMMAND -Project SIGLO -ConfigFile ".\config\ja\ChangelogConfig.xml" -User ninten -Password "123456AAA"
#   (例) COMMAND -Project SIGLO -Document                   -Package NintendoSDK -User ninten -Password "123456AAA"
#
#   -Document 指定がない場合、ライブラリ等の履歴情報を収集する。
#   その場合 -Version で指定したバージョンのものを探す。
#
#   -Document 指定があると、ドキュメント向けの履歴情報を収集する。-Version は無視する。
#
#   ドキュメント用もライブラリ用も、収集する対象は、JIRA のラベルフィールドに "変更履歴収集済" ラベルがないもの。
#
#
param
(
    [string]$AuthenticationFile = "",
    [string]$AliasFile = ".\config\Aliases.xml",
    [string]$ComponentRuleFile = ".\config\componentRule.json",
    [string]$Project = "SIGLO",
    [string]$User = "",
    [string]$Password = "",
    [string]$Version = "",
    [string]$Package = "",
    [string]$OutputName = "",
    [string]$ConfigFile = "",
    [string]$CheckHtmlName = "",
    [string]$SdkVersionFile = "..\..\..\..\Common\Versions\SdkVersion.xml",
    [string]$NXAddonVersionFile = "..\..\..\..\Common\Versions\NXAddonVersion.xml",

    [Int]$MaxArticle = 0,
    [switch]$CheckHtml = $false,
    [switch]$Document = $false,
    [switch]$Old = $false,
    [switch]$English = $false,
    [switch]$Japanese = $false,
    [switch]$Detail = $false,
    [switch]$Quiet = $false
)

#----------------------------------------------------------------
# チェック用
$script:CheckHtmlList = @()
$script:IsCheckHtmlOut = $false

#----------------------------------------------------------------
# 必要パラメータが揃っているかのチェック

#---------------- 表示レベル
$LogLevel = 1
if ( $Detail -eq $true -and $Quiet -eq $true )
{
    Write-Host "*** Error: Cannot -Detail and -Quiet options are specified at a time."
}
if ( $Detail -eq $true ){ $LogLevel = 2 }
if ( $Quiet  -eq $true ){ $LogLevel = 0 }

function Write-HostLog( $level, $str )
{
    if ( $level -eq "Detail" )
    {
        if ( $LogLevel -ge 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "Log" )
    {
        if ( $LogLevel -ge 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "Error" )
    {
        Write-Host $str
    }
    elseif ( $level -eq "DetailOnly" )
    {
        if ( $LogLevel -eq 2 ){ Write-Host $str }
    }
    elseif ( $level -eq "LogOnly" )
    {
        if ( $LogLevel -eq 1 ){ Write-Host $str }
    }
    elseif ( $level -eq "QuietOnly" )
    {
        if ( $LogLevel -eq 0 ){ Write-Host $str }
    }
}
function Write-Error( $str )
{
    Write-HostLog "Error" ("*** Error: "+$str)
}
function Write-Warning( $str )
{
    Write-HostLog "Error" ("*** Warning: "+$str)
}
function Write-Detail( $str )
{
    Write-HostLog "Detail" $str
}
function Write-Log( $str )
{
    Write-HostLog "Log" $str
}


#----------------------------------------------------------------
# 必要パラメータが揃っているかのチェック
#

#---------------- 英語／日本語の決定
$Lang = "Ja"
if ( $English -eq $true -and $Japanese -eq $true )
{
    Write-Error "Illegal language."
    exit 1
}
if ( $English -eq $true )
{
    $Lang = "En"
}

#---------------- ConfigFile 指定や、Package / Version
if ( $ConfigFile -eq "" )
{
    #---- ConfigFile 指定がない場合に Package と Version チェックを行う
    if ( $Package -eq "" )
    {
        Write-Error "Not specified package name."
        exit 1
    }

    if ( $Document -eq $false -and $Version -eq "" )
    {
        # (ドキュメントパッケージの場合バージョンは必須ではない)
        Write-Error "Not specified version."
        exit 1
    }
}
else
{
    if ( $ConfigFile -eq "DEFAULT" )
    {
        if ( $Document -eq $false )
        {
            if ( $Lang -eq "En" )
            {
                $ConfigFile = ".\config\en\ChangelogConfig.xml"
            }
            else
            {
                $ConfigFile = ".\config\ja\ChangelogConfig.xml"
            }
        }
        else
        {
            if ( $Lang -eq "En" )
            {
                $ConfigFile = ".\config\en\ChangelogDocumentConfig.xml"
            }
            else
            {
                $ConfigFile = ".\config\ja\ChangelogDocumentConfig.xml"
            }
        }
    }

    #---- 指定の Config ファイルが存在しているかをチェック
    if ( -not ( Test-Path -path $ConfigFile ) )
    {
        Write-Error ("Config file {0} not exists." -f $ConfigFile)
        exit 1
    }
}

#---------------- プロジェクトの決定
if ( "" -eq $Project )
{
    $Project = "SIGLO"
}

#---------------- プロジェクト名変換関数
. ".\convertProjectName.ps1"

#----------------  User,Password の決定
if ( "" -eq $User -or "" -eq $Password )
{
    #--- Authentication File が指定されていない場合
    if ( "" -eq $AuthenticationFile )
    {
        Write-Error "Authentication file is not specified."
        exit 1
    }

    if ( $AuthenticationFile -eq "DEFAULT" )
    {
        $AuthenticationFile = ".\config\Authentication.xml"
    }

    [Xml]$Config = Get-Content $AuthenticationFile

    #---- ユーザ
    if ( "" -eq $User )
    {
        $User = $Config.Configuration.User
    }
    #---- パスワード
    if ( "" -eq $Password )
    {
        $Password = $Config.Configuration.Password
    }
    #---- プロジェクト
    if ( 0 -lt ([string]$Config.Configuration.Project).Length )
    {
        $Project = $Config.Configuration.Project
    }
}
if ( "" -eq $User )
{
    Write-Error "User name is not specified."
    exit 1
}
if ( "" -eq $Password )
{
    Write-Error "Password is not specified."
    exit 1
}

#---------------- エイリアス情報の読み込み
$projectFnameAliasRule = @{}
$packageAliasRule = @{}
$versionAliasRule = @{}
$packageFnameAliasRule = @{}
$versionFnameAliasRule = @{}
$IsLoadAliasList = $false
. .\loadAliasList.ps1
if ( $script:IsLoadAliasList -eq $true )
{
    $projectFnameAliasRule, $projectAliasArray                                             = Create-ProjectAliases $Tags
    $packageAliasRule, $packageFnameAliasRule, $packageAliasArray, $packageFnameAliasArray = Create-PackageAliases $Tags
    $versionAliasRule, $versionFnameAliasRule, $versionAliasArray, $versionFnameAliasArray = Create-VersionAliases $Tags
}

# Confluence ページからバージョンリストを読む
. .\loadConfigInConfluence.ps1 -AuthenticationFile $AuthenticationFile -AliasFile $AliasFile -ConfigFile $ConfigFile
# バージョンのエイリアス情報を Confluence のもので上書き
$versionAliasRule, $versionFnameAliasRule, $versionAliasArray, $versionFnameAliasArray = Create-VersionAliases_Confluence

#----------------------------------------------------------------
#---- 日付文字列を取得
$dateStr = Get-Date -Format "yyyyMMdd"

#----------------------------------------------------------------
#---- コンポーネント英語化ルール
$global:componentJsonFile = $ComponentRuleFile

. ".\makeComponentRules.ps1"
$componentTranslationRule = Make-ComponentTranslationRules
if ( $null -eq $componentTranslationRule )
{
    Write-Error "Failed create component translation rules."
    exit 1
}
#---- コンポーネントカテゴリルール
$componentCategoryRule = Make-ComponentCategoryRules
if ( $null -eq $componentCategoryRule )
{
    Write-Error "Failed create component category rules."
    exit 1
}
#---- コンポーネント変換ルール
$componentConvertRule = Make-ComponentRules
if ( $null -eq $componentConvertRule )
{
    Write-Error "Failed create component conversion rules."
    exit 1
}

#================================================================================
#  関数群
#================================================================================
#----------------------------------------------------------------
function Invoke-RestApi( $Url, $User, $Password )
{
    $Auth = '{0}:{1}' -f $User, $Password
    $AuthBytes = [System.Text.Encoding]::ASCII.GetBytes($Auth)
    $Base64 = [System.Convert]::ToBase64String($AuthBytes)
    $Headers = @{"Authorization" = "Basic $Base64"}

    $request = [System.Net.WebRequest]::Create($url)
    $request.Method = "GET"
    $request.Headers["Authorization"] = "Basic $Base64"

    try
    {
        $response = $request.GetResponse()
        $stream = $response.GetResponseStream()
        $reader   = New-Object "System.IO.StreamReader" $stream
        return $reader.ReadToEnd()
    }
    catch [System.Net.WebException] # サーバアクセス中に例外発生
    {
        Write-Error "Error occurred in accessing JIRA server."
        return $null
    }
    finally
    {
        if ( $reader ){ $reader.Dispose() }
        if ( $strem  ){ $reader.Dispose() }
        if ( $reader ){ $reader.Dispose() }
    }
}

#----------------------------------------------------------------
# カテゴリ取得 (customfield_12205)
#
function Get-Category( $c )
{
    if ( $null -eq $c -or "" -eq $c )
    {
        return "None"
    }

    switch( $c["id"] )
    {
        11707 { return "1" } # 新機能
        11708 { return "2" } # 改善
        11709 { return "3" } # 不具合修正
        11713 { return "0" } # 未入力
        default { return "0" }
    }
}

#----------------------------------------------------------------
# 重要度取得 (customfield_12206)
function Get-Priority( $c )
{
    if ( $null -eq $c -or "" -eq $c )
    {
        return "None"
    }

    switch( $c["id"] )
    {
        11710 { return "1" } # 普通
        11711 { return "2" } # 重要(SDK使用者が特に注意すべき変更)
        11712 { return "3" } # 重要(互換性のない変更)
        default { return "0" }
    }
}

#----------------------------------------------------------------
# コンポーネント (customfield_12207)
function Get-Component( $c )
{
    if ( $null -eq $c -or "" -eq $c )
    {
        return "None"
    }

    if ( $c.Length -ne 1 )
    {
        return "None"
    }
    return $c[0]

    # コンポーネント複数は禁止。もし将来許可するなら以下を返す
    # return $c[ 0 .. ($c.Length - 1) ] -join ","
}

#----------------------------------------------------------------
# 対象パッケージ (customfield_12202)
function Get-Package( $c )
{
    if ( $null -eq $c -or "" -eq $c )
    {
        return $null
    }
    #return $c[ 0 .. ($c.Length - 1) ] -join ","
    return [array]$c[ 0 .. ($c.Length - 1) ]
}

#----------------------------------------------------------------
# タイトル (customfield_12201)
function Get-Title( $c )
{
    if ( $null -eq $c -or "" -eq $c )
    {
        return "None"
    }

    return ( $c.Trim() -replace '"','""' -replace '\\','\\' -replace "`r`n",'\r\n' )
}

#----------------------------------------------------------------
# コンテンツ (customfield_12400)
function Get-Contents( $c )
{
    if ( $null -eq $c -or "" -eq $c )
    {
        return "None"
    }

    $c_replaced = $c.Trim() -replace '"','""' -replace '\\','\\' -replace "`r`n",'\r\n'
    return $c_replaced
}

#リストの考慮(未完)
function Get-Contents2( $c )
{
    if ( $null -eq $c -or "" -eq $c )
    {
        return "None"
    }

    $c_replaced = $c.Trim() -replace '"','""' -replace '\\','\\' -replace "&","&amp;" -replace "<","&lt;" -replace "\r\n\r\n","`r`n`_&&DUMMY_LINE&&_`r`n"
    $isInList = $false
    [string[]]$ret = @()
    $ret += '<p class=""release_note"">'
    foreach( $s in ($c_replaced -split "`r`n") )
    {
        if ( $s -match "^・.*$" )
        {
            if ( $false -eq $isInList )
            {
                $ret += "</p><ul>"
                $isInList = $true
            }
            $ret += "<li>" + ($s -replace "^・","" ) + "</li>"
        }
        else
        {
            if ( $true -eq $isInList )
            {
                $ret += '</ul><p class=""release_note"">'
                $isInList = $false
            }
            $ret += $s + "<br>"
        }
    }
    $ret += "</p>"
    return ( ([string]::join("",$ret)) -replace "`<p`>`<br`>`<br`>","<p>" -replace "_&&DUMMY_LINE&&_","" )
}
#----------------------------------------------------------------
function Get-FixVersions( $f )
{
    # Write-Host $issue["key"] $fixVersions[0]["name"]
    $vers = @()
    for ($i=0; $i -lt $f.Length; $i++)
    {
        $vers += $f[$i]["name"]
    }
    return $vers[ 0 .. ($vers.Length - 1) ] -join ","
}

#----------------------------------------------------------------
#    文章途中での改行がないかをチェック
function Check-SplitLine( $key, $line )
{
    $tmpc = "\r\n" + $line + "\r\n"

    while($true)
    {
         # 箇条書きは除外
         if ( $tmpc -match "(.*\\r\\n)(-.*?\\r\\n)(.*)" )
         {
             $tmpc = $matches[1] + $matches[3]
             continue
         }
         #文末に 。？?！!）)｝}：:→← が来ている行は除外
         if ( $tmpc -match "(.*\\r\\n)(.*?[。？?！!）\)｝}：:→←][ 　]*\\r\\n)(.*)" )
         {
             $tmpc = $matches[1] + $matches[3]
             continue
         }
         break
    }
    $tmpc = $tmpc -replace "\\r\\n",""

    if ( $tmpc -ne "" )
    {
        $script:IsCheckHtmlOut = $true
        $script:CheckHtmlList += ("<b><font color=`"red`">Error:</font> <a href=`"http://spdlybra.nintendo.co.jp/jira/browse/{0}`">{0}</a></b><br>" -f $issue["key"])
        $script:CheckHtmlList += "May has split line.<br>"
        $script:CheckHtmlList += ("<PRE>{0}</PRE><br>" -f $tmpc)

        Write-Warning( "{0} may has split line. [{1}]" -f $key, $tmpc )
    }
}

#----------------------------------------------------------------
function Add-ChangeLogs( $issue )
{
    if ( $issue["fields"] )
    {
        $category0    = Get-Category  $issue["fields"]["customfield_12205"] # カテゴリ (未選択だと対象外)
        $priority0    = Get-Priority  $issue["fields"]["customfield_12206"] # 重要度
        [array]$package0     = Get-Package   $issue["fields"]["customfield_12202"] # パッケージ (未選択だと対象外)
        $component0   = Get-Component $issue["fields"]["customfield_12207"] # コンポーネント(複数選択禁止)
        $title0       = Get-Title     $issue["fields"]["customfield_12201"] # タイトル  (未入力だと対象外)
        $content0     = Get-Contents  $issue["fields"]["customfield_10400"] # 内容   (未入力だと対象外)
        if ( $Documents -eq $false )
        {
            $fixVersions0 = Get-FixVersions $issue["fields"]["fixVersions"] # 修正バージョン
        }
        else
        {
            $fixVersions0 = "-"
        }
    }
    else
    {
        Write-Warning( "{0} has no field." -f $issue["key"] )
        return
    }

    #---- 除外判定
    if ( $Document -eq $false )
    {
        #---- ライブラリの変更履歴向け
        # パッケージ
        if ( $package0 -eq $null )
        {
            #return
            #次の判定でチェックするのでここは敢えてスキップ
        }
        if ( -not ( $package0 -contains $Package ) )
        {
            return
        }

        # チェックをスキップする指示の確認
        $skipCheck = $false
        if ( $content0.Contains("{{SkipCheck}}") )
        {
            $content0 -cmatch "(.*){{SkipCheck}}(\\r\\n|)(.*)" | Out-Null
            $content0 = $matches[1] + $matches[3]
            $skipCheck = $true
        }

        # フィールドが不完全かのチェック
        if ( $category0 -eq "None" -or $priority0 -eq "None" -or $package0 -eq $null -or $component0 -eq "None" -or $title0 -eq "None" -or $content0 -eq "None" )
        {
            #---- フィールドチェック
            if ( $CheckHtml -eq $true -and $skipCheck -eq $false )
            {
                Write-Warning( "{0}" -f $issue["key"] )
                $script:IsCheckHtmlOut = $true
                $script:CheckHtmlList += ("<b><font color=`"red`">Error:</font> <a href=`"http://spdlybra.nintendo.co.jp/jira/browse/{0}`">{0}</a></b><br>" -f $issue["key"])
                if ( $category0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Category"
                    $script:CheckHtmlList += "Incomplete field: Category<br>"
                }
                if ( $priority0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Priority"
                    $script:CheckHtmlList += "Incomplete field: Priority<br>"
                }
                if ( $package0 -eq $null )
                {
                    Write-Warning "Incomplete field: Package"
                    $script:CheckHtmlList += "Incomplete field: Package<br>"
                }
                if ( $component0 -eq "None" )
                {
                    if ( $issue["fields"]["customfield_12207"].Length -ge 2 )
                    {
                        Write-Warning "Incomplete field: Component (specified more than one component)"
                        $script:CheckHtmlList += "Incomplete field: Component (specified more than one component)<br>"
                    }
                    else
                    {
                        Write-Warning "Incomplete field: Component"
                        $script:CheckHtmlList += "Incomplete field: Component<br>"
                    }
                }
                if ( $title0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Title"
                    $script:CheckHtmlList += "Incomplete field: Title<br>"
                }
                if ( $content0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Content"
                    $script:CheckHtmlList += "Incomplete field: Content<br>"
                }
                $script:CheckHtmlList += "<br>"
            }

            Write-Detail( "** {0} has imcomplete field." -f $issue["key"] )
            return
        }

        $skipSplitCheck = $false
        if ( $content0.Contains("{{SkipSplitCheck}}") )
        {
            $content0 -cmatch "(.*){{SkipSplitCheck}}(\\r\\n|)(.*)" | Out-Null
            $content0 = $matches[1] + $matches[3]
            $skipSplitCheck = $true
        }

        if ( $CheckHtml -eq $true -and $skipSplitCheck -eq $false -and $skipCheck -eq $false )
        {
            #---- 文章途中での改行がないかをチェック
            Check-SplitLine $issue["key"] $content0
        }
    }
    else
    {
        #---- ドキュメントの変更履歴向け
        # パッケージ
        if ( $package0 -eq $null )
        {
            #return
            #次の判定でチェックするのでここは敢えてスキップ
        }
        if ( -not ( $package0 -contains $Package ) )
        {
            return
        }

        #---- ドキュメントの場合、日付情報は issue の updated から取得する。
        #     ただし内容に {{Modified=yyyy/mm/dd}} がある場合、そちらの情報を優先する
        if ( $content0.Contains("{{Modified=") )
        {
            $content0 -cmatch "(.*)({{Modified=.*?}})(\\r\\n|)(.*)" | Out-Null
            $content0 = $matches[1] + $matches[4]
            $modStr = $matches[2]

            $modStr -cmatch "{{Modified=(....)/(..)/(..)}}" | Out-Null
            $dateStr0 = $matches[1] + $matches[2] + $matches[3]
        }
        else
        {
            $issue["fields"]["updated"] -match "(....)-(..)-(..)" | Out-Null
            $dateStr0 = $matches[1] + $matches[2] + $matches[3]
        }
        $dateStr = $dateStr0

        # フィールドが不完全かのチェック
        if ( $category0 -eq "None" -or $priority0 -eq "None" -or $package0 -eq $null -or $component0 -eq "None" -or $title0 -eq "None" -or $content0 -eq "None" -or $dateStr0.Length -ne 8 )
        {
            if ( $CheckHtml -eq $true )
            {
                Write-Warning( "{0}" -f $issue["key"] )
                "<b><font color=`"red`">Error:</font> <a href=`"http://spdlybra.nintendo.co.jp/jira/browse/{0}`">{0}</a></b><br>" -f $issue["key"] | Out-File -Encoding UTF8 $CheckHtmlName -Append
                if ( $category0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Category"
                    "Incomplete field: Category<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                }
                if ( $priority0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Priority"
                    "Incomplete field: Priority<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                }
                if ( $package0 -eq $null )
                {
                    Write-Warning "Incomplete field: Package"
                    "Incomplete field: Package<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                }
                if ( $component0 -eq "None" )
                {
                    if ( $issue["fields"]["customfield_12207"].Length -ge 2 )
                    {
                        Write-Warning "Incomplete field: Component (specified more than one component)"
                        "Incomplete field: Component (specified more than one component)<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                    }
                    else
                    {
                        Write-Warning "Incomplete field: Component"
                        "Incomplete field: Component<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                    }
                }
                if ( $title0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Title"
                    "Incomplete field: Title<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                }
                if ( $content0 -eq "None" )
                {
                    Write-Warning "Incomplete field: Content"
                    "Incomplete field: Content<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                }
                if ( $dateStr0.Length -ne 8 )
                {
                    Write-Warning "Incomplete field: Modified date"
                    "Incomplete field: Modified date<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
                }
                "<br>" | Out-File -Encoding UTF8 $CheckHtmlName -Append
            }

            Write-Detail( "** {0} has imcomplete field." -f $issue["key"] )
            return
        }

        $skipSplitCheck = $false
        if ( $content0.Contains("{{SkipSplitCheck}}") )
        {
            $content0 -cmatch "(.*){{SkipSplitCheck}}(\\r\\n|)(.*)" | Out-Null
            $content0 = $matches[1] + $matches[3]
            $skipSplitCheck = $true
        }

        if ( $CheckHtml -eq $true -and $skipSplitCheck -eq $false )
        {
            #---- 文章途中での改行がないかをチェック
            Check-SplitLine $issue["key"] $content0
        }
    }

    #---- コンポーネント
    # 日本語
    if ( $null -ne $componentConvertRule[ $component0 ] )
    {
        $component_ja = $componentConvertRule[ $component0 ]
    }
    else
    {
        # コンポーネント表にない場合
        $component_ja = $component0 + "(not firm)"
    }

    # 英語
    if ( $null -ne $componentTranslationRule[ $component_ja ] )
    {
        $component_en = $componentTranslationRule[ $component_ja ]
    }
    else
    {
        # コンポーネント表にない場合
        $component_en = $component_ja
    }

    # コンポーネントカテゴリ
    if ( $null -ne $componentCategoryRule[ $component0 ] )
    {
        $componentCategory = $componentCategoryRule[ $component0 ]
    }
    else
    {
        $componentCategory = "-"
    }

    # 内容フィールドのチェック
    if ( $CheckHtml -eq $true )
    {
        $warnArray = @()
        # ()関数 という表現を探す
        if ( $content0 -match "\(\)[ ]*関数" )
        {
            $warnArray += "May use XXX()関数 "
        }
        if ( $content0 -match "[a-zA-Z]+[ ]*関数" )
        {
            $warnArray += "May use XXX関数 "
        }
        if ( $warnArray -ne @() )
        {
            Write-Warning( "{0}" -f $issue["key"] )
            Write-Warning( $warnArray[ 0 .. ($warnArray.Length - 1) ] -join "`r`n" )
            $script:CheckHtmlList += ("<b><font color=`"orange`">Warning:</font> <a href=`"http://spdlybra.nintendo.co.jp/jira/browse/{0}`">{0}</a></b><br>" -f $issue["key"])
            $script:CheckHtmlList += ( ( $warnArray[ 0 .. ($warnArray.Length - 1) ] -join "<br>" ) + "<br>" )
            $script:CheckHtmlList += "<br>"
            $script:IsCheckHtmlOut = $true
        }
    }

    Write-Detail("** Create {0} log data." -f $issue["key"] )   # 課題番号 (SIGLO-1234など)
    Write-Detail("category     : {0}" -f $category0 )
    Write-Detail("priority     : {0}" -f $priority0 )
    Write-Detail("package      : {0}" -f $Package )
    Write-Detail("component(ja): {0}" -f $component_ja )
    Write-Detail("component(en): {0}" -f $component_en )
    Write-Detail("title        : {0}" -f $title0 )
    Write-Detail("content      : {0}" -f $content0 )
    Write-Detail("fixVersions  : {0}" -f $fixVersions0 )

    #---- キー変換
    $CvsKey = Convert-ProjectName $issue["key"]

    #---- ハッシュキー生成
    $md5 = [System.Security.Cryptography.MD5]::Create()
    $hash = $md5.ComputeHash( [System.Text.Encoding]::UTF8.GetBytes( $title0 ) )
    $result = [System.BitConverter]::ToString($hash).ToLower().Replace("-","")

    if ( $CvsKey -cmatch "([A-Z]+)-([0-9]+)" )
    {
        $h = "{0}{1}x{2}" -f $matches[1].ToLower(),$matches[2],$result
        $hashValue = $h[0..15] -join ""
    }
    else
    {
        $hashValue = $result[0..15] -join ""
    }


    '"{0}","{1}","{2}","{3}","{4}","{5}","{6}","{7}","{8}","{9}","{10}","{11}","{12}"' -f
        $hashValue,$Lang,$CvsKey,$category0,$priority0,$Package,$component_ja,$component_en,$componentCategory,$title0,$content0,$fixVersions0,$dateStr | Out-File -Encoding UTF8 $OutputFile -Append
}

#================================================================================
#  各バージョンやパッケージでのメイン処理
#================================================================================
function Process-EachPattern()
{
#---- バージョンとパッケージはエイリアス考慮
$Package = Filter-AliasStr $packageAliasRule $Package
if ( $Document -eq $false )
{
    $Version = Filter-AliasStr $versionAliasRule $Version
}

#---------------- 出力ファイル名
#  (例) Resources/ReleaseNotes/Ja/NintendoSDK_0_6_0_Ja.dat  (日本語の変更履歴を作成するための日本語の履歴データ)
#  (例) Resources/ReleaseNotes/Ja/NintendoSDK_0_6_0_En.dat  (日本語の変更履歴を作成するための英語の履歴データ)(後で翻訳される)

if ( $Document -eq $false )
{
    # (ライブラリパッケージ向け)
    $ProjectStr = Filter-AliasStr $projectFnameAliasRule $Project
    $PackageStr = Filter-AliasStr $packageFnameAliasRule $Package
    $VersionStr = Filter-AliasStr $versionFnameAliasRule $Version

    if ( $OutputName -eq "" )
    {
        $ResourceDir = Split-Path (Split-Path (Split-Path (Split-Path $script:MyInvocation.MyCommand.Path -Parent) -Parent) -Parent) -Parent
        $OutputName    = "{0}_{1}_{2}_{3}.dat"    -f $ProjectStr, $PackageStr, $VersionStr, $Lang

        # 作業フォルダ以下にファイルを作成
        New-Item ( "{0}\Outputs\Resources\ReleaseNotes\{1}" -f $ResourceDir, $Lang ) -itemType Directory -Force | Out-NUll
        $OutputFile = "{0}\Outputs\Resources\ReleaseNotes\{1}\{2}" -f $ResourceDir, $Lang, $OutputName
    }
    else
    {
        $OutputFile    = $OutputName
    }
}
else
{
    # (ドキュメントパッケージ向け)
    $PackageStr = Filter-AliasStr $packageFnameAliasRule $Package
    $VersionStr = "NoVersion"

    if ( $OutputName -eq "" )
    {
        $ResourceDir = Split-Path (Split-Path (Split-Path (Split-Path $script:MyInvocation.MyCommand.Path -Parent) -Parent) -Parent) -Parent
        if ( $Old -eq $false )
        {
            $OutputName = "{0}_{1}_{2}.dat"     -f $ProjectStr, $PackageStr, $Lang
        }
        else
        {
            $OutputName = "{0}_{1}_old_{2}.dat" -f $ProjectStr, $PackageStr, $Lang
        }
        $OutputFile = "{0}\Resources\ReleaseNotes\{1}\{2}" -f $ResourceDir, $Lang, $OutputName
    }
    else
    {
        $OutputFile    = $OutputName
    }
}

Write-Log( "** Gathering log for {0} ..." -f $OutputFile )

# HTML ファイル名
$script:CheckHtmlList = @()
$script:IsCheckHtmlOut = $false
if ( $CheckHtml -eq $true )
{
    if ( $CheckHtmlName -eq "" )
    {
        $CheckHtmlName = "Check\CheckField_{0}_{1}_{2}_{3}.html" -f $ProjectStr, $PackageStr, $VersionStr, $Lang
    }
}

#---- 種別によって処理を分ける
if ( $Document -eq $false )
{
    #---- (ライブラリの変更履歴向け)

    # プロジェクトが等しく、バージョンが等しくて、変更履歴のカテゴリに"変更履歴不要"以外の設定があるものを対象
    $Jql  = "project={0} AND fixVersion=`"$Version`" AND 変更履歴のカテゴリ!=null AND 変更履歴のカテゴリ!=`"変更履歴不要`"" -f $Project, $Version
    #$Jql  = "project={0} AND fixVersion=`"$Version`" AND key=SIGLO-3894" -f $Project, $Version

    Write-Detail( "** Project is {0}" -f $Project )
    Write-Detail( "** Version is {0}" -f $Version )
}
else
{
    #---- (ドキュメントの変更履歴向け)
    if ( $Old -eq $false )
    {
        #  変更履歴のタイトルがEMPTYでなく、履歴収集済ラベルがついていないものを集める
        #$Jql  = "project={0} AND 変更履歴のタイトル ~ `"NOT EMPTY`" AND ( labels is empty OR labels not in ( `"変更履歴収集済`" ) ) AND assignee in (yada)" -f $Project
        $Jql  = "project={0} AND 変更履歴のタイトル ~ `"NOT EMPTY`" AND ( labels is empty OR labels not in ( `"変更履歴収集済`" ) )" -f $Project
        $Version = "Not released"
    }
    else
    {
        #  変更履歴のタイトルがEMPTYでなく、履歴収集済ラベルがついているものを集める
        $Jql  = "project={0} AND 変更履歴のタイトル ~ `"NOT EMPTY`" AND ( labels in ( `"変更履歴収集済`" ) )" -f $Project
        $Version = "Released"
    }

    Write-Detail( "** Project is {0}" -f $Project )
    Write-Detail( "** Query is {0}" -f $Jql )
    Write-Detail( "** For documents" )
}

$StartAt = 0       # REST API で指定する読み込み開始位置
$MaxResults = 20   # REST API で指定する最大読み込み数
if ( $MaxArticle -ne 0 -and $MaxResults -gt $MaxArticle )
{
    $MaxResults = $MaxArticle
}
$ReadCount = 0

#---- 取得するフィールドの指定
if ( $Document -eq $false )
{
    $Fields = "customfield_12205,customfield_12202,customfield_12206,customfield_12207,customfield_12201,customfield_10400,fixVersions"
}
else
{
    $Fields = "customfield_12205,customfield_12202,customfield_12206,customfield_12207,customfield_12201,customfield_10400,fixVersions,updated"	
}
#
#  (カスタムフィールドのID)
#  customfield_12205 変更履歴のカテゴリ
#  customfield_12202 変更履歴の対象パッケージ(SIGLO)
#  customfield_12206 変更履歴の重要度
#  customfield_12207 変更履歴のコンポーネント(SIGLO)
#  customfield_12201 変更履歴のタイトル
#  customfield_10400 変更履歴の内容
#  (標準フィールドのID)
#  fixVersions       修正バージョン

#---- 中間ファイル名
Write-Detail( "** Output file name : {0}" -f $OutputFile )
Write-Detail( "** Language         : {0}" -f $Lang )

#---- フィールドチェック結果html
if ( $CheckHtml -eq $true )
{
    $script:CheckHtmlList += "<html>"
    $script:CheckHtmlList += "<title>Result of check JIRA</title>"
    $script:CheckHtmlList += "<body>"
    $script:CheckHtmlList += "<b>*** Result of check JIRA field related to the release note ***</b><br>"
    $script:CheckHtmlList += ("<b>Project={0}</b><br>" -f $Project)
    $script:CheckHtmlList += ("<b>Version={0}</b><br>" -f $Version)
    $script:CheckHtmlList += "<br>"
}

#---- 日付によるバージョン番号
$VersionNum = (Get-Date).ToString("yyyyMMddHHmmss")

#---- 設定情報をコメント行でファイルに出力
$ProjectConverted = Convert-ProjectName $Project
New-Item -force -ItemType file $OutputFile | Out-Null
@"
# DataVersion = $VersionNum
# Project = $ProjectConverted
# Version = $Version
# Lang    = $Lang
"@ | Out-File -Encoding UTF8 $OutputFile -Append

#---- 取得ループ
while(1)
{
    #---- Rest URL の作成
    $RestUrl = "http://spdlybra.nintendo.co.jp/jira/rest/api/2/search/?jql={0}&fields={1}&startAt={2}&maxResults={3}" -f $Jql, $Fields, $StartAt, $MaxResults
    # (例) このようなURLになる
    #      http://spdlybra.nintendo.co.jp/jira/rest/api/2/search/?jql=project=DBGTEST&fields=customfield_12205,customfield_12202,customfield_12206,customfield_12207,customfield_12201,customfield_10400

    #---- JIRA からデータを取得
    $RestResult = Invoke-RestApi $RestUrl $User $Password
    if ( $null -eq $RestResult )
    {
        Write-Detail "** No article matched."
        return
    }

    #---- JSON パース
$code=@"
static function parseJSON(json)
{
    return eval('(' +json + ')');
}
"@
    $JSONUtil = (Add-Type -Language JScript -MemberDefinition $code -Name "JSONUtil" -PassThru)[1]
    $jsobj = $JSONUtil::parseJSON($RestResult)

    #---- 取得状況の表示 (Detail 指定時)
    Write-Detail( "** Total articles  : {0}" -f $jsobj["total"] )
    Write-Detail( "** Length obtained : {0}" -f $jsobj["issues"].Length )
    Write-Detail( "** Start index     : {0}" -f $ReadCount )
    Write-Detail( "** Max index       : {0}" -f $MaxArticle )

    #---- なければ終了
    if ( $jsobj["issues"].Length -eq 0 )
    {
        break
    }

    #---- 各課題について調べる
    foreach ($num in $jsobj["issues"])
    {
        Add-ChangeLogs $jsobj["issues"][$num]
    }

    #---- 次の取得インデックス
    $StartAt += $MaxResults

    #---- 最大読み込みチェック
    $ReadCount += $jsobj["issues"].Length
    if ( $MaxArticle -gt 0 -and $ReadCount -ge $MaxArticle )
    {
        break
    }
}

#---- フィールドチェック結果htmlを閉じる
if ( $CheckHtml -eq $true )
{
    $script:CheckHtmlList += "<hr>`r`n</body>`r`n</html>"

    if ( $script:IsCheckHtmlOut )
    {
        New-Item -force -ItemType file $CheckHtmlName | Out-Null
        ( $script:CheckHtmlList[ 0 .. ($script:CheckHtmlList.Length - 1) ] -join "`r`n" ) | Out-File -Encoding UTF8 $CheckHtmlName -Append
    }
}

Write-Log( "** Done. Output file: {0}" -f $OutputFile )
}

#================================================================================
#  メイン
#================================================================================
if ( $ConfigFile -eq "" )
{
    Process-EachPattern
}
else
{
    $DefaultProject = $Project

    [Xml]$Config = Get-Content -Encoding UTF8 $ConfigFile
    $Packages = @([array]$Config.Configuration.ContentsList.Package)
    
	[array]$ConfVersionList = Get-GatherVersions

    #---- 各 Package タグ内を走査
    for( $i=0; $i -lt $Packages.Length; $i++ )
    {
        #---- Project タグがなければ引数指定のものにする
        if ( $Packages[$i].Project -eq $null )
        {
            $Projects = @( $DefaultProject )
        }
        else
        {
            $Projects = @([array]($Packages[$i]).Project)
        }

        #---- 各 Project に対して
        for ( $j=0; $j -lt $Projects.Length; $j++ )
        {
            $Project = $Projects[$j]
            $CList = @(([array]$Packages)[$i].PName)
            $VList = @(([array]$Packages)[$i].Version)

            # GatherConfig にバージョンが書かれていなければ Confluence から取得したバージョンを使う
            if(!$VList)
            {
                $VList = Get-GatherVersions
            }

            foreach( $cname in $CList )
            {
                foreach( $vname in $VList )
                {
                    if ( $vname -ne $null )
                    {
                        $Package = $cname
                        $Version = $vname
                        Process-EachPattern
                    }
                }
            }
        }
    }
}
exit 0
