﻿#
# 展開したフォルダとバージョンを指定し、
# 変更履歴中間ファイル dat から翻訳提出のための xml ファイルを作成する。
# 併せてコマンド動作の記録のために Command_Report.txt を出力する。
# この Command_Report.txt は英語翻訳されて戻ってきた xml と日本語 dat をマージ
# するときにも使用される。
#
# COMMAND -DatDirectory ./dat -Version 0.10.1
# COMMAND -DatDirectory ./dat -Versions "0.10.1,0.11.0"
# ( 複数バージョンは上のように""で囲うか、0.10.1`,0.11.0 とカンマをエスケープ )
#
param
(
    [string]$DatDirectory="./dat",
    [string][array]$Version="",
    [string]$AliasFile = "",
    [switch]$Help = $false
)

#---- ヘルプ
if ( $Help -eq $True )
{
Write-Host (`
@"
*** Usage of {0}
    Convert changelog dat file to xml file for sending translation.

Command
    {0} [options]

Options
    -DatDirectory <directory>   Specify data directory. (Defailt: ./dat)
    -Version <version(s)>       Specify version(s).
                                ex. 0.7.0 or "0.7.0,0.8.0" or All
    -AliasFile <aliasFile>      Specify alias file. (Defailt: config/AliasFile.xml)
    -Help                       Show help (this).
"@ `
-f $MyInvocation.MyCommand.Name )

    exit 0
}

#---- データファイルのあるフォルダチェック
# 引数指定のチェック
if ($DatDirectory -eq "")
{
    Write-Host "*** Error: DatDirectory is not specified."
    exit 1
}

# 存在チェック
if ( !(Test-Path $DatDirectory) )
{
    Write-Host "*** Error: Not found DatDirectory."
    exit 1
}

# ディレクトリであるかのチェック
if ( Test-Path $DatDirectory -PathType Leaf )
{
    Write-Host "*** Error: Specified file to DatDirectory option."
    exit 1
}

#---- 指定バージョンの取得(カンマで複数指定可能)
$Versions = $Version.Split(",")

# バージョン指定がない場合
if ( $Version -eq "" -or $Versions.Length -eq 0 )
{
    Write-Host "*** Error: Not specified version."
    exit 1
}

#---- エイリアス情報の読み込み
#---------------- エイリアス情報の読み込み
$ScriptPath = (Split-Path $MyInvocation.MyCommand.Path -Parent)
if ( $AliasFile -eq "" )
{
    $AliasFile = "$ScriptPath\config\Aliases.xml"
}

$packageAliasRule = @{}
$versionAliasRule = @{}
$packageFnameAliasRule = @{}
$versionFnameAliasRule = @{}
$idAliasRule = @{}
$IsLoadAliasList = $false

. "$ScriptPath\loadAliasList.ps1"
if ( $IsLoadAliasList -eq $true )
{
    $packageAliasRule, $packageFnameAliasRule, $packageAliasArray, $packageFnameAliasArray = Create-PackageAliases $Tags
    $versionAliasRule, $versionFnameAliasRule, $versionAliasArray, $versionFnameAliasArray = Create-VersionAliases $Tags
    $idAliasRule                              = Create-IdAliases      $Tags
}

#----------------------------------------------------------------
#
function Check-Convert( $file )
{
   $IsValid = $false

   #---- 有効なファイルかを各行を見てチェック
   Get-Content $file | Foreach-Object `
   {
       #---- コメント行はスキップ
       if ( $_.StartsWith( "#" ) )
       {
           # ただし DataVersion は覚えて置く
           if ( $_.StartsWith( "# DataVersion" ) )
           {
               $DataVersion = $_
           }

           return
       }

       #---- 有効な行か
       if ( $_ -match "SGL" -or $_ -match "IAA" -or $_ -match "BEZEL" )
       {
           $IsValid = $true
       }
   }

   #---- 中身がない場合はスキップ
   if ( $IsValid -eq $false )
   {
       return
   }

   #---- 変換スクリプト呼び出し
   PowerShell $ScriptPath\convertDatToXml.ps1 -InFile $file -Detail

   #---- 動作記録
   "Convert $file" | Out-File -Encoding UTF8 $ReportFile -Append
   $script:DataInfo += "# File = " + $file
   $file -match ".*?/(.*)" | Out-Null
   $script:DataInfo += "# Name = " + $matches[1]
   $script:DataInfo += $DataVersion
}

#----------------------------------------------------------------
# Main
#
#---- 現在の時刻
$CurTime = Get-Date -Format "yyyy-MM-dd HH:mm::ss"

#---- レポート出力のためのファイル
$ReportFile = "$DatDirectory/Command_Report.txt"
New-Item -force -ItemType file $ReportFile | Out-Null
"*** Convert changelog dat to xml" | Out-File -Encoding UTF8 $ReportFile -Append
"*** Date: $CurTime" | Out-File -Encoding UTF8 $ReportFile -Append
"*** Version: $Version" | Out-File -Encoding UTF8 $ReportFile -Append
"*** Current Folder: {0}" -f (Convert-Path .) | Out-File -Encoding UTF8 $ReportFile -Append
"*** Data Folder $DatDirectory" | Out-File -Encoding UTF8 $ReportFile -Append

$DataInfo = @()

#---- 各バージョンを調べる
foreach ( $Ver in $Versions )
{
    #---- ファイル名バージョンに変換
    $Ver = Filter-AliasStr $versionAliasRule $Ver
    $vname = Filter-AliasStr $versionFnameAliasRule $Ver

    #---- フォルダ以下を走査
    Get-ChildItem $DatDirectory | Foreach-Object `
    {
        #---- ディレクトリは除外
        if ( Test-Path "$DatDirectory/$_" -PathType Container )
        {
            return
        }

        #---- "_Ja.dat" で終わらないものは除外
        if ( ([string]$_).EndsWith( "_Ja.dat" ) -eq $False )
        {
            return
        }

        #---- ファイル名に指定バージョンが含まれるか
        if ( ( ([string]$_).Contains($vname + "_Ja.dat" ) -eq $True ) -or ($Version -eq "All" ) )
        {
            #---- 変換しようとしてみる
            Check-Convert $DatDirectory/$_
        }
    }
}

#---- 各対象ファイルのデータバージョンの情報を出力
"----------------" | Out-File -Encoding UTF8 $ReportFile -Append
($DataInfo -join "`r`n") | Out-File -Encoding UTF8 $ReportFile -Append
