﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Management.Automation;

namespace Nintendo.PlatformCodeFilter
{
    using FPath = System.IO.Path;

    /// <summary>
    /// 指定したプラットフォームを有効なプラットーフォームとして、
    /// 指定されなかったプラットーフォームを削除対象として、
    /// ソースファイル中のプラットフォーム固有コードをソースコードから削除します。
    /// 同時に、削除対象となるプラットフォームの名前のディレクトリも削除します。
    ///
    /// 開発メモ: Windows 7 上の PowerShell のインストールバージョン(2.0)で動作するように、
    ///           csproj で参照する System.Management.Automation.dll は 2.0 で動作するものを
    ///           参照するように設定しています。
    /// </summary>
    [Cmdlet(VerbsData.Convert, "PlatformCode")]
    public class ConvertPlatformCode : PSCmdlet
    {
        private delegate bool FilterFunc(string outFile, string inFile, uint includeAttr, IDictionary<string, uint> macroMap, out string errorMessage);

        private bool _enableBackup;
        private Dictionary<string, uint> _macroMap;
        private uint _includeAttr;
        private string[] _cExtention = new[] { "c", "cpp", "cxx", "cc", "h", "hpp", "hxx", "hh" };
        private HashSet<string> _cExtSet;

        /// <summary>
        /// 有効にするプラットフォーム関連のマクロを指定します。
        /// </summary>
        [Parameter(
            ParameterSetName = "IncludeSet",
            Mandatory = true,
            Position = 0)]
        public object[] Include { get; set; }

        /// <summary>
        /// 無効にするプラットフォーム関連のマクロを指定します。
        /// ここで指定したマクロが定義されたときに有効になるコードブロックを削除します。
        /// </summary>
        [Parameter(
            ParameterSetName = "IncludeSet",
            Mandatory = true,
            Position = 1)]
        public object[] Exclude { get; set; }

        /// <summary>
        /// プラットフォーム関連のマクロを「マクロ名 = ビット値」の組み合わせで指定します。
        /// </summary>
        [Parameter(
            ParameterSetName = "EnableBits",
            Mandatory = true,
            Position = 0)]
        public System.Collections.Hashtable Macro { get; set; }

        /// <summary>
        /// 有効にするマクロのビット値を指定します。
        /// Macroで指定されたマクロが定義されたときに有効になるコードブロックのうち、
        /// ここで指定したビット値との論理積で0となるコードブロックを削除します。
        /// </summary>
        [Parameter(
            ParameterSetName = "EnableBits",
            Mandatory = true,
            Position = 1)]
        public int Enable { get; set; }

        /// <summary>
        /// 処理対象のファイルのパスを指定します。
        /// </summary>
        [Parameter(
            Mandatory = true,
            Position = 2,
            ValueFromPipeline = true,
            ValueFromPipelineByPropertyName = true)]
        [Alias("PSPath")]
        public string[] Path { get; set; }

        /// <summary>
        /// バックアップを作成します。
        ///
        /// ファイルの場合は".cfbak"、ディレクトリの場合は"_cfbak"を
        /// 末尾につけた名前でバックアップを作成します。
        /// </summary>
        [Parameter()]
        public SwitchParameter Backup
        {
            get { return _enableBackup; }
            set { _enableBackup = value; }
        }

        /// <summary>
        /// C/C++言語ファイルの拡張子を指定します。
        /// </summary>
        [Parameter()]
        public string[] CExtention
        {
            get { return _cExtention; }
            set { _cExtention = value; }
        }

        protected override void BeginProcessing()
        {
            switch (ParameterSetName)
            {
            case "IncludeSet":
                CheckIncludeSetParam();
                break;
            case "EnableBits":
                CheckEnableBitsParam();
                break;
            default:
                ThrowTerminatingError(
                    new ErrorRecord(
                        new ArgumentException("Bad ParameterSetName"),
                        "UnableToAccessMacroList",
                        ErrorCategory.InvalidOperation,
                        null));
                return;
            }

            // C/C++言語の有効な拡張子のセット
            _cExtSet = new HashSet<string>(
                from ext in _cExtention
                select ext.ToUpperInvariant());
        }

        protected override void ProcessRecord()
        {
            // 指定されたファイルについて、filterCode 関数を呼び出します。
            //
            // TODO:    ここで、makefile, *.mkも対象に含め、
            //          それらはそれで、異なるフィルタ関数を実行する。

            // ファイル毎の処理
            foreach (var psPath in this.Path)
            {
                foreach (var pathInfo in SessionState.Path.GetResolvedPSPathFromPSPath(psPath))
                {
                    var filePath = pathInfo.ProviderPath;
                    var fileName = FPath.GetFileNameWithoutExtension(filePath).ToUpperInvariant();
                    var ext = FPath.GetExtension(filePath).ToUpperInvariant();
                    FilterFunc filter = null;
                    if (ext.Length >= 1 && _cExtSet.Contains(ext.Substring(1)))
                    {
                        filter = FilterCode;
                    }
                    /* 実装中 */
#if false
                    else if (fileName == "MAKEFILE" || ext == ".MK")
                    {
                        filter = MakefileFilter.filter;
                    }
#endif
                    if (filter != null)
                    {
                        var tempFile = FPath.GetTempFileName(); // 一時ファイルを作成

                        // 一時ファイルでフィルタ
                        string errorMessage;
                        var bSuccess = filter(tempFile, filePath, _includeAttr, _macroMap, out errorMessage);
                        if (bSuccess)
                        {
                            if (_enableBackup)
                            {
                                var renFileName = GenBackupFileName(FPath.GetDirectoryName(filePath), FPath.GetFileName(filePath));
                                if (null != renFileName)
                                {
                                    File.Move(filePath, renFileName);       // 現状のソースをバックアップファイルに退避
                                }
                                else
                                {
                                    WriteError(
                                        new ErrorRecord(
                                            new InvalidOperationException(
                                                string.Format(
                                                    "バックアップファイルがこれ以上作成できません。\".cfbak\"の名前のファイルを削除してください。 - [{0}]",
                                                    filePath)),
                                            "CannotCreateBackupFile",
                                            ErrorCategory.ResourceExists,
                                            filePath));
                                }
                            }
                            else
                            {
                                File.Delete(filePath);
                            }
                            File.Move(tempFile, filePath);   // フィルタしたファイルを元のファイル名に変更
                        }
                        else
                        {
                            File.Delete(tempFile);
                            if (null != errorMessage)
                            {
                                WriteError(
                                    new ErrorRecord(
                                        new InvalidOperationException(errorMessage),
                                        "ParserError",
                                        ErrorCategory.ParserError,
                                        filePath));
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// ファイル中のプラットフォーム固有コードのうち、対象外となっているコードを
        /// 取り除くようにします。
        /// </summary>
        /// <param name="inFile">対象のファイルのパス。</param>
        private static bool FilterCode(string outFile, string inFile, uint includeAttr, IDictionary<string, uint> macroMap, out string errorMessage)
        {
            var inStr = File.ReadAllText(inFile, Encoding.UTF8);
            List<StringIndexInfo> strIndexList;
            var esInStr = RemoveEscapedNewline(inStr, out strIndexList);
            var finder = new StringIndexInfoFinder(strIndexList);

            using (var sw = new StreamWriter(outFile, false, Encoding.UTF8))
            {
                // 特定のプラットフォームマクロのプリプロセッサディレクティブを解析します。
                var resultTp = CFilter.ParseDirective(esInStr, macroMap.ContainsKey);
                if (resultTp.Success)
                {
                    // 解析が成功したら、情報を元に公開対象のプラットフォーム固有のコードのみ出力します。
                    var cLineFilter = new CLineFilter(sw, inStr, finder, esInStr, macroMap, includeAttr);
                    errorMessage = null;
                    return cLineFilter.Exec(resultTp.CCodeBlocks, new IfExpr());
                }
                else
                {
                    // 解析に失敗したらエラーを出力します。
                    var index = finder.FindIndex(resultTp.Index);
                    var addMsg = string.Format("解析に失敗しました。 - {0}({1},{2}):", inFile, index + 1, resultTp.Index - strIndexList[index].Index + 1);
                    errorMessage = addMsg + Environment.NewLine + resultTp.ErrorMessage;
                    return false;
                }
            }
        }

        /// <summary>
        /// 文字列からエスケープされた改行を取り除きます。
        /// 同時に、元の文字列に対応する位置への差分情報のリストを取得します。
        /// </summary>
        /// <param name="str"></param>
        /// <param name="offsetList"></param>
        /// <returns></returns>
        private static string RemoveEscapedNewline(string str, out List<StringIndexInfo> offsetList)
        {
            offsetList = new List<StringIndexInfo>();

            var add = 0;
            var sb = new StringBuilder();
            foreach (var tp in EscapedNewlineParser.Parse(str))
            {
                sb.Append(str, (int)tp.Item1, (int)tp.Item2);
                offsetList.Add(new StringIndexInfo((int)tp.Item1 - add, (int)tp.Item2, add));
                add += (int)tp.Item3;
            }

            return sb.ToString();
        }

        /// <summary>
        /// バックアップファイル名を作成します。
        /// </summary>
        /// <param name="dir">バックアップファイル名を作成するファイルが存在するディレクトリ。</param>
        /// <param name="passagePlatforms">バックアップファイル名を作成するファイルの名前。</param>
        /// <returns>バックアップファイルのパス名を返します。</returns>
        private static string GenBackupFileName(string dir, string fileName)
        {
            for (var num = 0; num < 100; ++num)
            {
                var backSuffix =
                    num == 0 ?
                        ".cfbak" :
                        string.Format(".({0}).cfbak", num);
                var renFileName = FPath.Combine(dir, fileName + backSuffix);
                if (!File.Exists(renFileName))
                {
                    return renFileName;
                }
            }

            // バックアップ用ファイル名が作成できない場合は null を返します。
            return null;
        }

        private void CheckIncludeSetParam()
        {
            // Include, Exclude とも0個は許容しないがこれは、引数のバインド時にエラーとなるので
            // ここではチェックしない。

            if (Include.Length + Exclude.Length > 32)
            {
                ThrowTerminatingError(
                    new ErrorRecord(
                        new ArgumentException(string.Format("プラットフォームの数が多すぎます。Include, Exclude 合わせて 32個までです。")),
                        "TooMuchPlatforms",
                        ErrorCategory.InvalidArgument,
                        null));
            }

            try
            {
                _macroMap =
                    Include.Concat(Exclude)
                    .SelectMany(
                        (macro, i) =>
                            macro.GetType().IsArray ?
                                (from sub in (object[])macro
                                 select Tuple.Create(sub.ToString(), 0x1U << i)) :
                                new Tuple<string, uint>[] { Tuple.Create(macro.ToString(), 0x1U << i) })
                    .ToDictionary(tp => tp.Item1, tp => tp.Item2);
            }
            catch (ArgumentException)   // ToDictionary() で例外発生を想定
            {
                ThrowTerminatingError(
                    new ErrorRecord(
                        new ArgumentException(string.Format("プラットフォームのマクロ名が重複しています。")),
                        "MultipleDefinePlatforms",
                        ErrorCategory.InvalidArgument,
                        null));
            }

            // 有効にするプラットフォームのビット値の論理和。
            // 上記の制約により、Include.Length は 1 以上、32未満であるはず。
            _includeAttr = (0x1U << Include.Length) - 1U;
        }

        private void CheckEnableBitsParam()
        {
            // Macroは Hashtable であるため、指定されたマクロが重複している場合は、
            // Hashtable 作成時にエラーになる。
            // そのため、ここではマクロ名の重複はチェックしない。

            _macroMap =
                this.Macro
                .OfType<System.Collections.DictionaryEntry>()
                .Select(
                    entry =>
                    {
                        uint bits;
                        if (entry.Value is int)
                        {
                            bits = (uint)(int)entry.Value;
                        }
                        else
                        {
                            // int でなかった場合は文字列とみなしてパースしてみる。
                            var numStr = entry.Value.ToString().Trim().ToUpperInvariant();
                            var numStyles = NumberStyles.None;
                            if (numStr.StartsWith("0X"))
                            {
                                numStr = numStr.Substring(2);
                                numStyles = NumberStyles.AllowHexSpecifier;
                            }
                            int intBits;
                            if (!int.TryParse(numStr, numStyles, null, out intBits))
                            {
                                ThrowTerminatingError(
                                    new ErrorRecord(
                                        new ArgumentException(string.Format("無効なビット値です。 - [{0}]", entry.Value)),
                                        "UnableParseBit",
                                        ErrorCategory.InvalidArgument,
                                        Macro));
                            }
                            bits = (uint)intBits;
                        }

                        if (bits == 0U || bits == uint.MaxValue)
                        {
                            ThrowTerminatingError(
                                new ErrorRecord(
                                    new ArgumentException(string.Format("全ビットが0や1のものは指定できません。 - [{0}]", entry.Value)),
                                    "InvalidBits",
                                    ErrorCategory.InvalidArgument,
                                    Macro));
                        }

                        return Tuple.Create(entry.Key.ToString(), bits);
                    })
                .ToDictionary(tuple => tuple.Item1, tuple => tuple.Item2);

            // 有効にするプラットフォームのビット値の論理和
            _includeAttr = (uint)this.Enable;
        }
    }

    /// <summary>
    /// 文字列の位置・長さと元の文字列の位置への加算値を持つクラスです。
    /// </summary>
    internal class StringIndexInfo
    {
        /// <summary>
        /// コンストラクタです。
        /// </summary>
        /// <param name="offset">インデックス。</param>
        /// <param name="length">長さ。</param>
        /// <param name="add">加算値。</param>
        public StringIndexInfo(int offset, int length, int add)
        {
            Index = offset;
            Length = length;
            Add = add;
        }

        /// <summary>
        /// インデックスを取得します。
        /// </summary>
        public int Index { get; private set; }

        /// <summary>
        /// 長さを取得します。
        /// </summary>
        public int Length { get; private set; }

        /// <summary>
        /// 加算値を取得します。
        /// </summary>
        public int Add { get; private set; }
    }

    /// <summary>
    /// Cのコードツリーに対して、プラットフォームマクロ定義を判断して、
    /// 有効なプラットフォームのコードのみを出力するクラスです。
    /// </summary>
    internal class CLineFilter
    {
        private readonly TextWriter _tw;
        private readonly string _inStr;
        private readonly StringIndexInfoFinder _finder;
        private readonly string _enInStr;
        private readonly IDictionary<string, uint> _macroMap;
        private readonly uint _includeAttr;

        /// <summary>
        /// 最初の書き込み時に false に設定します。
        /// 改行文字の制御のために使用します。
        /// </summary>
        private bool _isFirstLine = true;

        public CLineFilter(TextWriter tw, string inStr, StringIndexInfoFinder finder, string enInStr, IDictionary<string, uint> macroMap, uint includeAttr)
        {
            _tw = tw;
            _inStr = inStr;
            _finder = finder;
            _enInStr = enInStr;
            _macroMap = macroMap;
            _includeAttr = includeAttr;
        }

        public bool Exec(IEnumerable<CCodeBlock> cLineBases, IfExpr expr)
        {
            var isFileMod = false;
            foreach (var token in cLineBases)
            {
                if (token.IsIfBlock)
                {
                    if (IsValidCode(expr))
                    {
                        var cLineIfEndif = (IfEndifBlock)token;
                        var preIfExpr = expr;

                        foreach (var ifInfo in cLineIfEndif.IfInfos)
                        {
                            // 除外するプラットフォームマクロが存在する場合はフィルタする。
                            if (ContainsExcludeMacro(ifInfo.Expr))
                            {
                                // #if, #ifdef, #ifndef, #elif のプラットフォーム定義マクロについて、
                                // 有効なものだけにする。
                                var ifLine = GetRangeOrignalString(ifInfo.IfLine);
                                var parsedDirLine = CFilter.ParseDirectiveLine(_enInStr, ifInfo.IfLine);
                                var fitStr = FilterDirectiveLine(parsedDirLine, ifInfo.IfLine);
                                if (fitStr != ifLine)
                                {
                                    isFileMod = true;
                                }
                                WriteLine(fitStr);
                            }
                            else
                            {
                                WriteLine(ifInfo.IfLine);
                            }

                            // 現在の#ifの式にANDして子供を再帰的に処理
                            var nestIsFileMod =
                                Exec(
                                    ifInfo.CCodeBlocks,
                                    new IfExprOp2(IfExprType.And, preIfExpr, ifInfo.Expr));

                            isFileMod = isFileMod || nestIsFileMod;

                            // 今回の式の否定を繋げて、#elif,#elseに回す。
                            preIfExpr =
                                new IfExprOp2(
                                    IfExprType.And, preIfExpr, new IfExprOp1(IfExprType.Not, ifInfo.Expr));
                        }

                        WriteLine(cLineIfEndif.EndifLine);
                    }
                    else
                    {
                        isFileMod = true;
                    }
                }
                else
                {
                    if (IsValidCode(expr))
                    {
                        WriteLine(((CLineBlock)token).Line);
                    }
                    else
                    {
                        isFileMod = true;
                    }
                }
            }

            return isFileMod;
        }

        private static uint? EvaluateAnd(uint? leftMaskOpt, uint? rightMaskOpt)
        {
            return
                (leftMaskOpt.HasValue && rightMaskOpt.HasValue) ?
                    (uint?)(leftMaskOpt.Value & rightMaskOpt.Value) :
                    (leftMaskOpt.HasValue ?
                        leftMaskOpt :
                        rightMaskOpt);
        }

        private static uint? EvaluateOr(uint? leftMaskOpt, uint? rightMaskOpt)
        {
            return
                (leftMaskOpt.HasValue && rightMaskOpt.HasValue) ?
                    (uint?)(leftMaskOpt.Value | rightMaskOpt.Value) :
                    null;
        }

        /// <summary>
        /// IfExpr の末尾要素を列挙します。
        /// </summary>
        /// <param name="expr">IfExpr</param>
        /// <returns>IfExpr の IEnumerable。</returns>
        private static IEnumerable<IfExpr> EnumIfExpr(IfExpr expr)
        {
            switch (expr.Type)
            {
            case IfExprType.Not:
                foreach (var subExpr in EnumIfExpr(((IfExprOp1)expr).Expr))
                {
                    yield return subExpr;
                }
                break;
            case IfExprType.And:
            case IfExprType.Or:
            case IfExprType.Usop2:
                {
                    var crExpr = (IfExprOp2)expr;
                    foreach (var subExpr in EnumIfExpr(crExpr.Left))
                    {
                        yield return subExpr;
                    }
                    foreach (var subExpr in EnumIfExpr(crExpr.Right))
                    {
                        yield return subExpr;
                    }
                }
                break;
            default:
                yield return expr;
                break;
            }
        }

        /// <summary>
        /// #if 式のツリーをもとに、コードブロックに関係するプラットフォームマクロのビット値を
        /// 算出します。
        ///
        /// #if の式で対応しているものは、次のものです。
        /// ・defined(マクロ)
        /// ・数値
        ///
        /// 「defined(マクロ)」は、マクロがプラットフォームのマクロとして指定されたものであれば、
        /// そのビット値にします。プラットフォームのマクロとして指定されていなければ、
        /// null にします。
        ///
        /// 数値の場合は 0 でないときは、全ビットが1、0のときは 0 とします。
        ///
        /// 演算子は、&&, ||, ! のみ対応します。
        /// && は左右のビット値のビットANDを行います。
        /// 一方が null の場合は null でない方のビット値を採用します。
        /// 両方が null の場合は null にします。
        ///
        /// || は左右のビット値のビットORを行います。
        /// いずれか一方に null がある場合は null にします。
        ///
        /// ! 場合、ビット値の反転ですが、null の場合は変化しないものとします。
        /// そのため、否定は末端で解決させるようにし、途中の &&, || では状態を反転して伝搬させます。
        ///
        /// 最終的に null になった場合は、プラットフォームマクロの判定以外の未確定の状態を
        /// 含むことを表します。そのため、コードブロックの除外対象にはなりません。
        /// </summary>
        /// <param name="expr">式。</param>
        /// <param name="bReverse">式の内容を否定する場合は true。</param>
        /// <returns></returns>
        private uint? ResolveExpr(IfExpr expr, bool bReverse)
        {
            switch (expr.Type)
            {
            case IfExprType.Not:
                // bReverseを反転。
                return ResolveExpr(((IfExprOp1)expr).Expr, !bReverse);
            case IfExprType.And:
                // bReserseがtrueのときは、両方の式が反転されるため Or にします。
                // そうでないときは、And の処理を行います。
                {
                    var crExpr = (IfExprOp2)expr;
                    return bReverse ?
                        EvaluateOr(ResolveExpr(crExpr.Left, bReverse), ResolveExpr(crExpr.Right, bReverse)) :
                        EvaluateAnd(ResolveExpr(crExpr.Left, bReverse), ResolveExpr(crExpr.Right, bReverse));
                }
            case IfExprType.Or:
                // bReserseがtrueのときは、両方の式が反転されるため And にします。
                // そうでないときは、Or の処理を行います。
                {
                    var crExpr = (IfExprOp2)expr;
                    return bReverse ?
                        EvaluateAnd(ResolveExpr(crExpr.Left, bReverse), ResolveExpr(crExpr.Right, bReverse)) :
                        EvaluateOr(ResolveExpr(crExpr.Left, bReverse), ResolveExpr(crExpr.Right, bReverse));
                }
            case IfExprType.Usop2:
                {
                    // サポートしない演算子なので、左右の式両方ともプラットフォームマクロを
                    // 含まないときに nullを返します。
                    // そうでないときは例外を発生します。
                    var crExpr = (IfExprOp2)expr;
                    if (!ContainsMacro(crExpr.Left) && !ContainsMacro(crExpr.Right))
                    {
                        return null;
                    }

                    throw new NotImplementedException();
                }
            case IfExprType.DefinedMacro:
                // bReserseがtrueのときは、値があるときのみその値を反転します。
                {
                    var crExpr = (IfExprDefinedMacro)expr;
                    uint bits;
                    if (_macroMap.TryGetValue(crExpr.Macro, out bits))
                    {
                        bits = bReverse ? ~bits : bits;
                        return bits;
                    }
                    else
                    {
                        return null;
                    }
                }
            case IfExprType.Num:
                {
                    // bReserse のときは、0 のときは 1そうでないときは 0にします。
                    var num = ((IfExprNum)expr).Num;
                    return
                        bReverse ?
                            (num == 0 ? uint.MaxValue : 0U) :
                            (num != 0 ? uint.MaxValue : 0U);
                }
            case IfExprType.Null:
            default:
                return null;
            }
        }

        /// <summary>
        /// 除外対象のマクロが存在するかどうかを判定します。
        /// </summary>
        /// <param name="expr">#if の式</param>
        /// <param name="macros">判別するプラットフォームマクロ</param>
        /// <returns>存在する場合はtrueを返します。</returns>
        private bool ContainsExcludeMacro(IfExpr expr)
        {
            return
                EnumIfExpr(expr)
                .Any(
                    subExpr =>
                    {
                        switch (subExpr.Type)
                        {
                        case IfExprType.DefinedMacro:
                            {
                                var crExpr = (IfExprDefinedMacro)subExpr;
                                return !IsIncludeMacro(crExpr.Macro);
                            }
                        default:
                            return false;
                        }
                    });
        }

        /// <summary>
        /// プラットフォームマクロが存在するかどうかを判定します。
        /// </summary>
        /// <param name="expr">#if の式</param>
        /// <returns>存在する場合はtrueを返します。</returns>
        private bool ContainsMacro(IfExpr expr)
        {
            return
                EnumIfExpr(expr)
                .Any(
                    subExpr =>
                    {
                        switch (subExpr.Type)
                        {
                        case IfExprType.DefinedMacro:
                            {
                                var crExpr = (IfExprDefinedMacro)expr;
                                return _macroMap.ContainsKey(crExpr.Macro);
                            }
                        default:
                            return false;
                        }
                    });
        }

        private string GetSnippetString(StringSnippet snippet, int baseOffset)
        {
            DefinedSnippet defSni;
            StringRangeSnippet rngSni;

            if (null != (defSni = snippet as DefinedSnippet))
            {
                var openStr = GetRangeOrignalString(defSni.Open, baseOffset);
                var closeStr = GetRangeOrignalString(defSni.Close, baseOffset);
                if (IsIncludeMacro(defSni.MacroString))
                {
                    var definedStr = GetRangeOrignalString(defSni.Defined, baseOffset);
                    // #ifdef, #ifndef の場合は definedStr と openStr は空になっている。
                    if (definedStr == string.Empty)
                    {
                        definedStr = "defined";
                    }
                    if (openStr == string.Empty)
                    {
                        openStr = " ";
                    }
                    return definedStr + openStr + GetRangeOrignalString(defSni.Macro, baseOffset) + closeStr;
                }
                else
                {
                    return openStr + "0" + closeStr;
                }
            }
            else if (null != (rngSni = snippet as StringRangeSnippet))
            {
                return GetRangeOrignalString(rngSni.Snippet, baseOffset);
            }
            else
            {
                return ((SimpleStringSnippet)snippet).Snippet;
            }
        }

        private string FilterDirectiveLine(StringSnippet[] stringSnippet, StringRange baseRange)
        {
            var sniEnum =
                from str in stringSnippet
                select GetSnippetString(str, baseRange.Offset);

            return string.Concat(sniEnum);
        }

        /// <summary>
        /// 指定したマクロが未定義か有効にするマクロかを判定します。
        /// </summary>
        private bool IsIncludeMacro(string id)
        {
            uint bits;
            return
                _macroMap.TryGetValue(id, out bits) ?
                    0U != (_includeAttr & bits) :
                    true;
        }

        private bool IsValidCode(IfExpr expr)
        {
            var result = ResolveExpr(expr, false);
            if (!result.HasValue)
            {
                return true;
            }

            return 0U != (result.Value & _includeAttr);
        }

        private string GetRangeOrignalString(StringRange range, int baseOffset = 0)
        {
            if (range.Length == 0)
            {
                return string.Empty;
            }

            var startInfo = _finder.Find(baseOffset + range.Offset);
            var endInfo = _finder.Find(baseOffset + range.Offset + range.Length);
            var stIndex = baseOffset + range.Offset + startInfo.Add;
            var etIndex = baseOffset + range.Offset + range.Length + endInfo.Add;
            return _inStr.Substring(stIndex, etIndex - stIndex);
        }

        private void WriteLine(StringRange range)
        {
            WriteLine(GetRangeOrignalString(range));
        }

        private void WriteLine(string str)
        {
            if (_isFirstLine)
            {
                _isFirstLine = false;
            }
            else
            {
                _tw.WriteLine();
            }

            _tw.Write(str);
        }
    }
}
