﻿namespace Nintendo.PlatformCodeFilter

open System
open System.Collections.Generic
open System.IO
open System.Linq
open System.Text
open FParsec

module EscapedNewlineParser =

    let getIndex = getPosition |>> fun pos -> int pos.Index

    let internal createTp (startIdx, endIdx) = (startIdx, endIdx - startIdx, 0)

    let internal parseLines =
        Inline.SepBy (
            (fun tp -> [ createTp tp ]),
            (fun preRange escapedNewline (startIdx, endIdx) ->
                // 改行がエスケープされていたかどうかで、直前の行の長さを調整します。
                match preRange with
                | [] -> failwith "list empty" // ここには来ないはず。
                | (preStartIdx, preLen, _) :: tail ->
                    let lineLen = startIdx - preStartIdx    // 行頭から次の行頭までの文字数
                    let preRange =
                        if escapedNewline then  // 改行の前にバックスラッシュがある?
                            let newPreLen = preLen - 1  // 行の長さをバックスラッシュまでとする
                            (preStartIdx, newPreLen, lineLen - newPreLen)
                        else
                            (preStartIdx, lineLen, 0)   // 行の長さに改行文字も含める
                    createTp (startIdx, endIdx) :: preRange :: tail
            ),
            List.rev,
            getIndex .>> skipRestOfLine false .>>. getIndex,
            // 行のセパレータである改行をスキップしますが、その前に直前の文字が
            // バックスラッシュかどうかの真偽値を返すようにします。
            (previousCharSatisfies ((=) '\\') >>. preturn true) <|>% false .>> skipNewline,
            resultForEmptySequence = fun () -> [])

    /// 改行を解析して、行ごとの (行頭の位置, 行の長さ、次の行の位置までの差分) を計算します。
    /// バックスラッシュが先行する改行の場合は行の長さに含めません。
    /// そうでない改行は行の長さに含めます。
    let Parse (lines: string) : (int * int * int) seq =
        match runParserOnString parseLines () "" lines with
        | Success (result, _, _) -> upcast result
        | Failure (errorMsg, _, _) ->
            // 失敗の可能性があるパースはしていないので、ここには来ないはず。
            raise <| InvalidOperationException errorMsg
