﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

using System.Text.RegularExpressions;

namespace NxAgingHelper
{
    public class RegExWatcher
    {
        private string m_watchString;
        private int[] m_regExId;                    // 登録番号（＝GUI でのボタン番号）
        private string[] m_regExString;             // 正規表現を示した文字列
        public delegate void MatchFunction(int id);
        private MatchFunction[] m_regExFunctions;   // マッチしたときに実行する関数

        private bool m_IsMatched { get; set; } = false;
        public bool IsMatched { get { return m_IsMatched; } }

        private const int RegExItemNum = 3;         // 「GUI 版の入力欄の数」 or 「CUI で扱える正規表現の数」 の大きい方
        public const int InvalidId = -1;
        public const int InvalidIndex = -2;
        private const string LineFeedStr = "\r\n";

        public RegExWatcher()
        {
            m_watchString = "";
            m_regExId = new int[RegExItemNum];
            m_regExString = new string[RegExItemNum];
            m_regExFunctions = new MatchFunction[RegExItemNum];
            for(int i=0; i<RegExItemNum; i++)
            {
                SetRegExData(i, InvalidId, "", null);
            }
        }

        // 処理開始時に変数をリセットする
        public void Refresh()
        {
            m_watchString = "";
            m_IsMatched = false;
        }

        // 正規表現と処理関数の登録
        private void SetRegExData(int index, int id, string regExString, MatchFunction matchFunc)
        {
            m_regExId[index] = id;
            m_regExString[index] = regExString;
            m_regExFunctions[index] = matchFunc;
        }

        // 正規表現と処理関数を登録する
        public int RegisterRegExFunc(int id, string regExString, MatchFunction matchFunc)
        {
            int emptyIndex = InvalidIndex;
            for(int i=0; i<RegExItemNum; i++)
            {
                if (m_regExId[i] == id)
                {
                    // 重複する ID がすでに登録されている場合は登録失敗
                    return InvalidId;
                }
                if (m_regExId[i] == InvalidId && emptyIndex == InvalidIndex)
                {
                    // 書き込み候補の Index を探す
                    emptyIndex = i;
                }
            }
            if (emptyIndex == InvalidIndex)
            {
                // 配列が埋まっている場合は登録失敗
                return InvalidIndex;
            }
            SetRegExData(emptyIndex, id, regExString, matchFunc);
            return emptyIndex;
        }

        // 正規表現と処理関数の登録を解除する
        public void ReleaseRegExFunc(int id)
        {
            for(int i=0; i<RegExItemNum; i++)
            {
                if (m_regExId[i] == id)
                {
                    SetRegExData(i, InvalidId, "", null);
                }
            }
            // 解除しようとしたものが見つからなくてもエラーではない
        }

        // 正規表現によってログを監視し、マッチしたら特定のアクションを実行する
        // …のために改行が来るまで文字列をプールする処理が必要。（結構こまぎれの文字列が来る）
        public void WatchLogWithRegEx(string text)
        {
            // 既存の文字列に今回の文字列を連結する
            m_watchString = m_watchString + text;

            // 処理対象の文字列が改行文字を含まないなら処理を抜ける
            if (!m_watchString.Contains(LineFeedStr)) { return; }

            // 処理対象の文字列を Split で分割する
            string[] separator = { "\r", "\n" };
            string[] splitStrings = m_watchString.Split(separator, StringSplitOptions.RemoveEmptyEntries);

            // 処理対象の文字列の終端が改行文字かどうかを調べる
            int targetNum = 0;
            if (m_watchString.EndsWith(LineFeedStr))
            {
                // 終端が改行なら分割した全てを処理対象とする、m_watchString は空にする
                targetNum = splitStrings.Length;
                m_watchString = "";
            }
            else
            {
                // 終端が改行でない場合、最後の要素は m_watchString に残す
                targetNum = splitStrings.Length - 1;
                m_watchString = splitStrings[splitStrings.Length - 1];
            }

            // 分割した要素について正規表現との一致を確認する（処理に渡す文字列には改行文字は付随しない）
            for (int i = 0; i < targetNum; i++)
            {
                MatchingAndAction(splitStrings[i]);
            }
        }
        // 正規表現とのマッチングと、登録関数の実行
        private void MatchingAndAction(string text)
        {
            for (int i = 0; i < RegExItemNum; i++)
            {
                // 適切に登録されていない項目は処理対象外
                if (m_regExId[i] == InvalidId) { continue; }
                if (m_regExString[i] == "") { continue; }
                if (m_regExFunctions[i] == null) { continue; }

                // 正規表現マッチング
                if (!Regex.IsMatch(text, m_regExString[i])) { continue; }

                // マッチしたので与えられたアクションを実行する
                m_IsMatched = true;
                m_regExFunctions[i](m_regExId[i]);
            }
        }

        // コマンドライン版で、正規表現にマッチしたときに呼び出す関数
        // （重要な役割はなく、関数が存在すればとりあえずOK）
        public void MatchFuncForCommand(int id)
        {
            for (int i=0; i < RegExItemNum; i++)
            {
                if (id == m_regExId[i])
                {
                    Console.WriteLine("Regular expression has matched with UART log. : " + m_regExString[i]);
                }
            }
        }
    }
}
