﻿using System;
using System.Threading;

namespace NxAgingHelper
{
    public class Program
    {
        public static void Main(string[] args)
        {
            // コマンドライン引数のパース
            CommandLineArgs opt = new CommandLineArgs();
            try
            {
                if (!opt.ParseArgs(args))
                {
                    return;
                }
            }
            catch
            {
                Environment.ExitCode = 1;
                return;
            }

            var useSdev = opt.Params.Sdev != null;
            var useEdev = opt.Params.Edev != null;

            var eventLogPath = opt.Params.EventLogPath ?? string.Format("{0}_Event.log", DateTime.Now.ToString("yyyyMMdd"));
            var uartLogPath = opt.Params.UartLogPath ?? string.Format("{0}_Uart.log", DateTime.Now.ToString("yyyyMMdd"));

            var sdevIpAddr = opt.Params.Sdev;
            var mcciPort = opt.Params.McciPort;
            var hdmiSwitcherIpAddr = opt.Params.HdmiSwitcherIpAddr;
            var hdmiSwitcherComPortName = opt.Params.HdmiSwitcherPortName;
            var useHdmiSwitcherComPort = hdmiSwitcherIpAddr == null;

            // コンフィグファイル読み込み
            var parser = new CommandYamlParser();
            parser.EnableHdmi = opt.Params.EnableHdmiEvent;
            parser.EnableMcci = opt.Params.EnableMcciEvent;
            parser.EnablePowerUsb = opt.Params.EnablePowerUsbEvent;
            parser.EnableSdevUsb = opt.Params.EnableSdevUsbEvent;
            parser.EnablePowerButton = opt.Params.EnablePowerButtonEvent;
            parser.EnableBatteryLevelEmulation = opt.Params.EnableBatteryEmulationEvent;

            ExecuteCommand command;
            try
            {
                command = parser.Parse(opt.Params.Config);
            }
            catch (Exception e)
            {
                Console.WriteLine(e.Message);
                Environment.ExitCode = 1;
                return;
            }

            // UART ロギング開始
            var uartLogger = new UartLogger();
            if (opt.Params.EnableUartLog)
            {
                if (useSdev)
                {
                    uartLogger.StartForSdev(opt.Params.Sdev, opt.Params.UartLogPath);
                }
                else
                {
                    uartLogger.StartForEdev(opt.Params.Edev, opt.Params.UartLogPath);
                }

                Console.WriteLine("Start uart logging.");

                // エージング開始まで少し待つ
                Thread.Sleep(3000);
            }

            // エージング開始
            AgingManagerPrivate agingManager = new AgingManagerPrivate()
            {
                UseHdmi = opt.Params.EnableHdmiEvent,
                UseMcci = opt.Params.EnableMcciEvent,
                UsePowerUsb = opt.Params.EnablePowerUsbEvent,
                UseSdev = opt.Params.EnableSdevUsbEvent || opt.Params.EnablePowerButtonEvent
            };

            agingManager.ExceptionHandleEvent += e =>
            {
                Console.WriteLine(e.Message);
                Environment.ExitCode = 1;
                return;
            };

            try
            {
                agingManager.Start(command, eventLogPath, sdevIpAddr, mcciPort, useHdmiSwitcherComPort ? hdmiSwitcherComPortName : hdmiSwitcherIpAddr, true);
            }
            catch (Exception e)
            {
                Console.WriteLine(e.Message);
                Environment.ExitCode = 1;
                return;
            }

            Console.WriteLine("Aging started.");

            // 無限ループ or 一定時間待つ
            bool endless = opt.Params.Time == 0;
            int timeLeft = opt.Params.Time;
            while (endless || timeLeft > 0)
            {
                Thread.Sleep(1000);
                timeLeft--;
            }

            // エージング終了
            agingManager.Stop();

            Console.WriteLine("Stop event.");

            // UART ロギング終了
            if (opt.Params.EnableUartLog)
            {
                // コマンドラインオプションの指定によってはここで無限ループ
                while (opt.Params.ContinueUartLogAfterAging)
                {
                    Thread.Sleep(50000);
                }

                uartLogger.Stop();

                Console.WriteLine("Stop uart logging.");
            }
        }
    }
}
