﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using Microsoft.VisualStudio.Shell.Interop;
using Nintendo.NintendoSdkVsExtension.Base;
using Nintendo.NintendoSdkVsExtension.VcAccessors;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace Nintendo.NintendoSdkVsExtension.Logic
{
    public static class VcProjectConfigurationUtil
    {
        /// <summary>
        /// V1仕様のプロパティシートを使用している場合、Configuration ラベルのプロパティグループから NintendoSDK の設定値を読み取る。
        /// V0仕様のプロパティシートを使用しているかいずれも使用していない場合、評価済みプロパティから NintendoSDK の設定値を読み取る。
        /// </summary>
        public static SdkProjectSettingsStore ReadSdkProjectSettingsStoreFromVcxproj(IVsHierarchy h)
        {
            var vcProject = new VcProjectAccessor(h);
            var projectCollection = new Microsoft.Build.Evaluation.ProjectCollection();
            var constructionProject = new Lazy<MsBuildConstructionProjectAccessor>(
                () => new MsBuildConstructionProjectAccessor(vcProject.ProjectFilePath, projectCollection));

            try
            {
                var configurations = vcProject.Configurations.Select(
                    x => ReadSdkConfigurationSettings(x, constructionProject)).ToArray();
                return new SdkProjectSettingsStore(configurations);
            }
            finally
            {
                if (constructionProject.IsValueCreated)
                {
                    constructionProject.Value.Dispose();
                }
            }
        }

        private static SdkConfigurationSettingsStore ReadSdkConfigurationSettings(
            VcProjectConfigurationAccessor configuration,
            Lazy<MsBuildConstructionProjectAccessor> constructionProject)
        {
            if (IsSdkEnabledInV1Manner(configuration))
            {
                // Configuration ラベルのプロパティグループから評価前プロパティを採用する
                var conditionEvaluator = new MsBuildConditionEvaluator(configuration.ConfigurationName, configuration.PlatformName);
                var propertyGroup = constructionProject.Value.GetPropertyGroups(conditionEvaluator, BaseConstants.ProjectGroupConfigurationLabel).FirstOrDefault();

                // プロパティグループが存在しない場合（前回は存在していたのに削除された！）、デフォルト値を返しておく。
                // 後段でプロパティグループが存在しないため反映できず警告が出力される。
                if (propertyGroup == null)
                {
                    return new SdkConfigurationSettingsStore(configuration.ConfigurationPair)
                    {
                        SdkRoot = SdkRoot.Empty,
                        IsSdkEnabled = false,
                        Spec = BaseConstants.DefaultSpec,
                        BuildType = BaseConstants.DefaultBuildType,
                    };
                }

                var sdkRoot = new SdkRoot(
                    SdkRootKind.DirectProperty,
                    propertyGroup.GetPropertyOrDefault("NintendoSdkRoot") ?? string.Empty);
                var specString = propertyGroup.GetPropertyOrDefault("NintendoSdkSpec");
                var buildTypeString = propertyGroup.GetPropertyOrDefault("NintendoSdkBuildType");

                return new SdkConfigurationSettingsStore(configuration.ConfigurationPair)
                {
                    SdkRoot = sdkRoot,
                    IsSdkEnabled = true,
                    Spec = ToDefaultedSdkSpecString(specString),
                    BuildType = ToSdkBuildTypeOrDefault(buildTypeString),
                };
            }
            else
            {
                // NintendoSDK のプロパティシートが見つからないか V0 仕様の場合、
                // 評価された最終のプロパティ値から値を取得する
                var sdkRoot = new SdkRoot(
                    SdkRootKind.DirectProperty,
                    configuration.GetEvaluatedPropertyValue("NintendoSdkRoot") ?? string.Empty);
                var specString = configuration.GetEvaluatedPropertyValue("NintendoSdkSpec");
                var buildTypeString = configuration.GetEvaluatedPropertyValue("NintendoSdkBuildType");

                return new SdkConfigurationSettingsStore(configuration.ConfigurationPair)
                {
                    SdkRoot = sdkRoot,
                    IsSdkEnabled =
                        IsSdkEnabledInV0Manner(configuration) ||
                        (!sdkRoot.IsEmpty && !string.IsNullOrEmpty(specString) && !string.IsNullOrEmpty(buildTypeString)),
                    Spec = ToDefaultedSdkSpecString(specString),
                    BuildType = ToSdkBuildTypeOrDefault(buildTypeString),
                };
            }
        }

        private static string ToDefaultedSdkSpecString(string specString)
        {
            if (string.IsNullOrEmpty(specString))
            {
                return BaseConstants.DefaultSpec;
            }
            else
            {
                return specString;
            }
        }

        private static SdkBuildType ToSdkBuildTypeOrDefault(string buildTypeString)
        {
            if (string.IsNullOrEmpty(buildTypeString))
            {
                return BaseConstants.DefaultBuildType;
            }
            else
            {
                return Util.ParseEnumOrDefault(buildTypeString, SdkBuildType.Develop);
            }
        }

        private static bool IsSdkEnabledInV0Manner(VcProjectConfigurationAccessor configuration)
        {
            return configuration.PropertySheets.Any(x => x.Name.Contains("NintendoSdkVcProjectSettings"));
        }

        private static bool IsSdkEnabledInV1Manner(VcProjectConfigurationAccessor configuration)
        {
            return configuration.PropertySheets.Any(x => x.Name.Contains("ImportNintendoSdk"));
        }
    }
}
